package corpusapi.tei.readers;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import org.codehaus.stax2.XMLStreamReader2;

import corpusapi.tei.TEICorpusText;
import corpusapi.tei.TEISegmentGroup;
import corpusapi.tei.TEISenseSegmentGroup;

public class TEISenseReader extends TEIReader {

	protected TEISenseReader(String fileName, TEICorpusText corpusText) {
		super(fileName, corpusText);
	}
	
	/**
	 * Probuje zwolnic wykorzystywane zasoby poprzez zwolnienie pamieci
	 * zarezerwowanej na wczytany plik.
	 * 
	 * @throws XMLStreamException
	 * @throws IOException 
	 * 
	 */
	public void close() throws XMLStreamException, IOException {		
		lastSegmentGroup = null;
		if (streamReader != null) {
			streamReader.close();
			streamReader = null;
		}
		super.close();
	}
	
	/** Pobiera pierwsze SenseSegmentGroup z pliku xml. Strumień jest tworzony zawsze na nowo.
	 * @return grupa lub null, gdy nie ma
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEISegmentGroup getFirstSenseSegmentGroup() throws XMLStreamException, IOException {
		XMLStreamReader2 sr = createStreamReader();
		
		String id = null;
		String fVal = null;
		List<String> segmentIdList = new ArrayList<String>();
		
		while (sr.hasNext()) {
			int event = sr.next();
			if (event == XMLStreamReader2.START_ELEMENT) {
				
				QName name = sr.getName();
				String local = name.getLocalPart();
				
				if (id != null) {
					if ("f".equals(local)) {
						fVal = sr.getAttributeValue(null, "fVal");
					}
				}
				if ("seg".equals(local)) {
					String segmentId = sr.getAttributeValue(null, "corresp").split("#")[1];
					
					if (id != null) {											
						String nextId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						sr.close();
						return new TEISenseSegmentGroup(id, corpusText, segmentIdList, nextId, null, fVal);  
					} else {
						id = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						segmentIdList.add(segmentId);
					}
				}
			}
		}
		
		sr.close();
		
		if (fVal != null) {
			return new TEISenseSegmentGroup(id, corpusText, segmentIdList, null, null, fVal);		
		} else {
			return null;
		}
	}
	
	/** Pobiera SenseSegmentGroup dla segmentu o podanym id lub null, gdy nie ma.
	 * @param segmentId
	 * @return SenseSegmentGroup dla segmentu o podanym id lub null, gdy nie ma
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEISenseSegmentGroup getSenseSegmentGroupBySegmentId(String segmentId) throws XMLStreamException, IOException {
		List<String> segmentIdList = new ArrayList<String>();
		segmentIdList.add(segmentId);
		
		XMLStreamReader2 sr = createStreamReader();
		
		String id = null;
		String lastId = null;
		String fVal = null;
		
		while (sr.hasNext()) {
			int event = sr.next();
			if (event == XMLStreamReader2.START_ELEMENT) {
				QName name = sr.getName();
				String local = name.getLocalPart();
				if (id != null) {
					if ("f".equals(local)) {
						fVal = sr.getAttributeValue(null, "fVal");
					}
				}
				if ("seg".equals(local)) {
					String corresp = sr.getAttributeValue(null, "corresp");
					if (id != null) {
						String nextId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						sr.close();
						return new TEISenseSegmentGroup(id, corpusText, segmentIdList, nextId, lastId, fVal);  
					} else if (corresp.contains(segmentId)) {
						id = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
					} else {
						lastId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						sr.skipElement();
					}
				}
			}
		}
		sr.close();
		if (fVal != null) {
			return new TEISenseSegmentGroup(id, corpusText, segmentIdList, null, lastId, fVal);		
		} else {
			return null;
		}
	}

	private TEISegmentGroup lastSegmentGroup = null;
	private XMLStreamReader2 streamReader;
	private boolean setStreamReaderForGroupId(String groupId) throws XMLStreamException, IOException {
		if (lastSegmentGroup == null || !groupId.equals(lastSegmentGroup.nextGroupId)) {
			streamReader = createStreamReader();
			return false;
		}
		return true;
	}
	/** Pobiera SenseSegmentGroup o podanym id lub null, gdy nie ma. Jeśli pytamy o kolejne grupy w pliku,
	 * korzysta z jednego strumienia.
	 * 
	 * @param groupId
	 * @return SenseSegmentGroup o podanym id lub null, gdy nie ma
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEISenseSegmentGroup getSenseSegmentGroupById(String groupId) throws XMLStreamException, IOException {
		boolean oldStream = setStreamReaderForGroupId(groupId);
		
		List<String> segmentIdList = new ArrayList<String>();
		String segmentId = null;
		String lastId = null;
		String nextId = null;
		String fVal = null;
		
		while (streamReader.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = streamReader.next();
			}
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
					lastId = lastSegmentGroup.getId();
				}
				QName name = streamReader.getName();
				String local = name.getLocalPart();
				if (segmentId != null) {
					if ("f".equals(local)) {
						fVal = streamReader.getAttributeValue(null, "fVal");
					}
				}
				if ("seg".equals(local)) {
					if (fVal != null) {
						nextId = streamReader.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						break;  
					} else {
						String id = streamReader.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						if (id.equals(groupId)) {
							segmentId = streamReader.getAttributeValue(null, "corresp").split("#")[1];
							segmentIdList.add(segmentId);
						} else {
							lastId = id;
							streamReader.skipElement();
						}
					}
				}
			}
		}
		if (fVal != null) {
			lastSegmentGroup = new TEISenseSegmentGroup(groupId, corpusText, segmentIdList, nextId, lastId, fVal);	
			return (TEISenseSegmentGroup) lastSegmentGroup;
		} else {
			lastSegmentGroup = null;
			return null;
		}
	}
}