package corpusapi;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import corpusapi.tei.TEICorpus;
import corpusapi.tei.TEISenseInventory;

public class CorpusFactory {
	private static CorpusFactory INSTANCE;
	
	private CorpusFactory() {}
	
	public static CorpusFactory getInstance() {
		if (INSTANCE == null) {
			INSTANCE = new CorpusFactory();
		}
		
		return INSTANCE;
	}
	
	/**
	 * Tworzy obiekt klasy implementującej interfejs SenseInventory wskazanego typu.
	 * @param type typ (np. TEI), v. getCorporaTypes()
	 * 
	 * @return implementacja SenseInventory
	 */
	public SenseInventory getSenseInventory(String configFile) {
		String senseInventoryFile = null;
		String senseInventoryType = null;
		
		String pathToConfig = getPathToConfig(configFile);
		
		/* Odczytaj plik konfiguracyjny */
		XMLInputFactory factory = XMLInputFactory.newInstance();
		FileReader reader;
		try {
			reader = new FileReader(configFile);
			XMLEventReader eventReader = factory.createXMLEventReader(reader);
			while (eventReader.hasNext()) {
				XMLEvent event = eventReader.nextEvent();
				if (event.isStartElement()) {
					StartElement element = (StartElement) event;					
					
					if ("senseInventory".equalsIgnoreCase(element.getName().toString())) {
						Attribute attr = element.getAttributeByName(new QName("path"));
						senseInventoryFile = attr.getValue();
						
						attr = element.getAttributeByName(new QName("type"));
						senseInventoryType = attr.getValue();
					}
				}
			}
			reader.close();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (XMLStreamException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		if (senseInventoryType == null) {
			return null;
		}
		
		if ("TEI".equals(senseInventoryType.toUpperCase())) {
			return TEISenseInventory.createFromXML(pathToConfig+senseInventoryFile);
		}

		return null;
		
	}
	
	private String getPathToConfig(String configFile) {
		String result = "";
		int slash = configFile.lastIndexOf(File.separator);
		if (slash > -1) {
			result = configFile.substring(0, slash);
			result = result.concat(File.separator);
		}
		return result;		
	}
	
	/**
	 * Tworzy obiekt klasy implementującej interfejs Corpus dla wskazanego typu korpusu.
	 * @param type typ korpusu (np. TEI, XCES), v. getCorporaTypes()
	 * @param cached czy tworzyć reprezentację segmentów i ich anotacji w pamięci
	 * @return implementacja api korpusu
	 */
	public Corpus getCorpus(String configFile, boolean cached) {
		String corpusType = null;
		String corpusId = null;
		List<String> corpusTexts = null;
		
		String pathToConfig = getPathToConfig(configFile);
		
		/* Odczytaj plik konfiguracyjny */
		XMLInputFactory factory = XMLInputFactory.newInstance();
		FileReader reader;
		try {
			reader = new FileReader(configFile);
			XMLEventReader eventReader = factory.createXMLEventReader(reader);
			while (eventReader.hasNext()) {
				XMLEvent event = eventReader.nextEvent();
				if (event.isStartElement()) {
					StartElement element = (StartElement) event;
					
					if ("corpus".equalsIgnoreCase(element.getName().toString())) {
						Attribute attr = element.getAttributeByName(new QName("type"));
						corpusType = attr.getValue();
						Attribute attr2 = element.getAttributeByName(new QName("id"));
						corpusId = attr2.getValue();
					}
					
					if ("text".equalsIgnoreCase(element.getName().toString())) {						
						Attribute attr = element.getAttributeByName(new QName("path"));

						corpusTexts = getCorpusTextsPaths(pathToConfig+attr.getValue());
					}
				}
			}
			reader.close();
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (XMLStreamException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		
		if (corpusType == null) {
			return null;
		}
		
		if ("TEI".equals(corpusType.toUpperCase())) {
			return new TEICorpus(corpusTexts, cached, corpusId);
		}

		return null;
	}

	private List<String> getCorpusTextsPaths(String path) {
		List<String> result = new ArrayList<String>();
		walkIn(new File(path), result);
		return result;
    }
 
    /**
     * Recursive function to descend into the directory tree and find all the directories 
     * that contain file "header.xml"
     * @param dir A file object defining the top directory
     **/
    public static void walkIn(File dir, List<String> result) {
        String pattern = "header.xml";
        
        File listFile[] = dir.listFiles();
        if (listFile != null) {
            for (File file : listFile) {
                if (file.isDirectory()) {
                	walkIn(file, result);              	
                }
                else {
                    if(file.getName().equals(pattern)) {
                    	result.add(file.getParent());
                    }
                }
            }
        }      
    }

	/**
	 * Zwraca dostępne typy korpusów.
	 * @return dostępne typy korpusów.
	 */
	public String[] getCorporaTypes() {
		return new String []{"TEI"}; 
	}

	/**
	 * Zwraca dostępne typy słowników sensów (SenseInventory).
	 * @return dostępne typy słowników sensów (SenseInventory)
	 */	
	public String[] getSenseInventoryTypes() {
		return new String []{"TEI"};
	}
}
