package corpusapi.tei;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Characters;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import corpusapi.Corpus;
import corpusapi.CorpusText;
import corpusapi.MorphoSegmentGroup;
import corpusapi.SegmentGroup;
import corpusapi.SenseSegmentGroup;
import corpusapi.SentenceSegmentGroup;
import corpusapi.tei.readers.TEIMorphoReader;
import corpusapi.tei.readers.TEIReader;
import corpusapi.tei.readers.TEISegmentationReader;
import corpusapi.tei.readers.TEISenseReader;

public class TEICorpusText implements CorpusText {
	private TEICorpus corpus;
	private String textId;
	private String path;
	
	private int blockIterator = 0;
	private int blockCount = Integer.MAX_VALUE;
	protected List<TEITextBlock> blockList = new ArrayList<TEITextBlock>();

	XMLEventReader eventReader = null;
	XMLEvent eventBuffer = null;

	public TEICorpusText(TEICorpus corpus, String textId, String path) {
		this.corpus = corpus;
		this.textId = textId;
		this.path = path;
	}
	
	@Override
	public TEITextBlock getFirstTextBlock() {
		eventReader = null;
		blockIterator = 0;
		return getNextTextBlock();
	}

	@Override
	public TEITextBlock getNextTextBlock() {
		if (blockIterator >= blockCount) {
			return null;
		}
		
		if (blockIterator < blockList.size()) {
			TEITextBlock textBlock =  blockList.get(blockIterator);
			blockIterator++;
			return textBlock;
		}
		
		if (eventReader == null) {
			try {
				TEIReader teiReader = TEIReader.getTEITextReader(this);
				eventReader = teiReader.createEventReader();
			} catch (FileNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (XMLStreamException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		
		TEITextBlock textBlock = null;
		boolean prevRelated = true;

		try {
			boolean blockOpened = false;
			while ((eventBuffer != null) || (eventReader.hasNext())) {
				XMLEvent event;
				if (eventBuffer != null) {
					event = eventBuffer;
					eventBuffer = null;
				} else {
					event = eventReader.nextEvent();
				}

				if (event.isEndElement()) {
					EndElement element = (EndElement) event;
					QName qName = element.getName();
					
					if ((qName.getLocalPart().equalsIgnoreCase("ab"))
							|| (qName.getLocalPart().equalsIgnoreCase("p"))
							|| (qName.getLocalPart().equalsIgnoreCase("u"))) {
						if (textBlock != null) {
							textBlock.length = element.getLocation().getCharacterOffset() - textBlock.filePosition;
						}
						blockOpened = false;
					}
				}
				
				if (event.isStartElement()) {
					StartElement element = (StartElement) event;
					QName qName = element.getName();
					
					// Nastepny tag okazal sie byc nowym znacznikiem div
					if (qName.getLocalPart().equalsIgnoreCase("div")) {
						if (textBlock != null) {
							eventBuffer = element;
							textBlock.nextRelated = false;
							return textBlock;
						} else {
							prevRelated = false;
						}
					}
					
					if ((qName.getLocalPart().equalsIgnoreCase("ab"))
							|| (qName.getLocalPart().equalsIgnoreCase("p"))
							|| (qName.getLocalPart().equalsIgnoreCase("u"))) {
						
						blockOpened = true;
						
						// Nastepny tag okazal sie byc nastepnym blokiem
						if (textBlock != null) {
							eventBuffer = element;
							textBlock.nextRelated = true;
							return textBlock;
						}
						
						QName attrName = new QName("http://www.w3.org/XML/1998/namespace", "id");
						
						textBlock = new TEITextBlock(this, element.getAttributeByName(attrName).getValue(), 
								qName.getLocalPart(), 0, 0, 
								prevRelated, false, blockIterator);
						
						blockIterator++;
						blockList.add(textBlock);
					}
				}
				
				if (event.isCharacters()) {
					if ((blockOpened) && (textBlock != null)) {
						Characters chars = (Characters)event;
						textBlock.filePosition = chars.getLocation().getCharacterOffset();
					}
				}
			}
			blockCount = blockList.size();
			eventReader.close();
			
		} catch (XMLStreamException e) {
			System.out.println(e);
		}
		
		return textBlock;
	}

	@Override
	public boolean hasNextTextBlock() {
		int oldIterator = blockIterator;
		TEITextBlock textBlock = getNextTextBlock();
		blockIterator = oldIterator;
		return textBlock != null;
	}

	@Override
	public Corpus getCorpus() {
		return corpus;
	}

	@Override
	public void closeCorpusText() {
		try {
			TEIReader.closeTEIReader(getTextFilePath());
			TEIReader.closeTEIReader(getMorphoFilePath());
			TEIReader.closeTEIReader(getSegmentationFilePath());
			TEIReader.closeTEIReader(getSensesFilePath());		
			if (eventReader != null) eventReader.close();
		} catch (XMLStreamException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		eventBuffer = null;
		if (blockList != null) {
			blockList.clear();
		}
		
		blockList = null;
		blockIterator = 0;
		blockCount = Integer.MAX_VALUE;
	}

	@Override
	public String getPath() {
		return path;
	}
	
	public String getTextFilePath() {
		return path + File.separator + "text.xml";
	}
	
	public String getSegmentationFilePath() {
		return path + File.separator + "ann_segmentation.xml";
	}
	
	public String getMorphoFilePath() {
		return path + File.separator + "ann_morphosyntax.xml";
	}
	
	public String getSensesFilePath() {
		return path + File.separator + "ann_senses.xml";
	}

	@Override
	public TEITextBlock getTextBlock(int position) {
		if (position < 0) {
			return null;
		}

		if (position < blockList.size()) {
			return blockList.get(position);
		}

		int oldIterator = blockIterator;

		TEITextBlock textBlock = null;
		while (blockIterator != position) {
			textBlock = getNextTextBlock();
		}

		textBlock = getNextTextBlock();

		blockIterator = oldIterator;
		return textBlock;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((textId == null) ? 0 : textId.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		TEICorpusText other = (TEICorpusText) obj;
		if (textId == null) {
			if (other.textId != null)
				return false;
		} else if (!textId.equals(other.textId))
			return false;
		return true;
	}
	
	/** przeniesc do interfejsu?
	 * Zwraca pierwsze segment group podanego typu w tekście, null jeśli nie ma
	 * @return
	 * @throws Exception 
	 */
	public TEISegmentGroup getFirstSegmentGroup(Class<? extends SegmentGroup> type) throws Exception {
		if (type.equals(SentenceSegmentGroup.class) || type.equals(TEISentenceSegmentGroup.class)) {			
			TEISegmentationReader reader = TEIReader.getTEISegmentationReader(this);			
			return reader.getFirstSentenceSegmentGroup();
		} else if (type.equals(SenseSegmentGroup.class) || type.equals(TEISenseSegmentGroup.class)) {
			TEISenseReader reader = TEIReader.getTEISenseReader(this);
			return reader.getFirstSenseSegmentGroup();
		} else if (type.equals(MorphoSegmentGroup.class) || type.equals(TEIMorphoSegmentGroup.class)) {
			TEIMorphoReader reader = TEIReader.getTEIMorphoReader(this);
			return reader.getFirstMorphoSegmentGroup();
		}
		return null;
	}
	
	/** przeniesc do interfejsu?
	 * Zwraca pierwszy segment w tekście, null jeśli nie ma
	 * @return
	 */
	public TEISegment getFirstSegment() throws Exception {
		TEISegmentationReader reader = TEIReader.getTEISegmentationReader(this);
		return reader.getFirstSegmentItText();
	}
}