package corpusapi.tei.readers;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringReader;
import java.util.HashMap;
import java.util.Map;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;

import org.codehaus.stax2.XMLStreamReader2;

import corpusapi.tei.TEICorpusText;

public class TEIReader {
	protected static Map<String, TEIReader> teiReaderMap = new HashMap<String, TEIReader>();
	protected static XMLInputFactory factory = XMLInputFactory.newInstance();

	protected TEICorpusText corpusText;
	protected File file;
	protected String fileContents;
	protected Reader reader;
	protected boolean cached;
	protected Integer cacheMaxItems = null;

	protected TEIReader(String fileName, TEICorpusText corpusText) {
		this.file = new File(fileName);
		this.corpusText = corpusText;
		this.cached = corpusText.getCorpus().isCached();
		if (this.cached) {
			cacheMaxItems = 100;
		}
	}

	public static void closeTEIReader(String filename) throws XMLStreamException, IOException {
		TEIReader t = teiReaderMap.remove(filename);
		if (t != null) {
			t.close();
		}
	}
	
	public static TEIReader getTEIReader(String fileName, TEICorpusText corpusText) {
		TEIReader teiReader = null;

		if ((teiReader = teiReaderMap.get(fileName)) == null) {
			teiReader = new TEIReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
		}

		return teiReader;
	}

	public static TEISegmentationReader getTEISegmentationReader(TEICorpusText corpusText) {
		String fileName = corpusText.getSegmentationFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEISegmentationReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEISegmentationReader) teiReader;
		} else if (teiReader instanceof TEISegmentationReader) {
			return (TEISegmentationReader) teiReader;
		} else {
			return null;
		}
	}

	public static TEISenseReader getTEISenseReader(TEICorpusText corpusText) {
		String fileName = corpusText.getSensesFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEISenseReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEISenseReader) teiReader;
		} else if (teiReader instanceof TEISenseReader) {
			return (TEISenseReader) teiReader;
		} else {
			return null;
		}
	}

	public static TEIMorphoReader getTEIMorphoReader(TEICorpusText corpusText) {
		String fileName = corpusText.getMorphoFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEIMorphoReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEIMorphoReader) teiReader;
		} else if (teiReader instanceof TEIMorphoReader) {
			return (TEIMorphoReader) teiReader;
		} else {
			return null;
		}
	}

	public static TEITextReader getTEITextReader(TEICorpusText corpusText) {
		String fileName = corpusText.getTextFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEITextReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEITextReader) teiReader;
		} else if (teiReader instanceof TEITextReader) {
			return (TEITextReader) teiReader;
		} else {
			return null;
		}
	}

	/**
	 * Zwraca domyslny strumien do odczytu z przypisanego pliku XML lub tworzy
	 * nowy strumien, jesli wczesniej nie istnial. Ta metoda zawsze tworzy
	 * maksymalnie jeden strumien.
	 * 
	 * @return strumien do odczytu z pliku.
	 * @throws IOException
	 */
	public Reader getTextReader() throws IOException {
		if (reader == null) {
			reader = createTextReader();
		}
		return reader;
	}

	/**
	 * Wczytuje plik XML do pamieci, jesli wczesniej nie zostal wczytany oraz
	 * tworzy nowy obiekt klasy StringReader do odczytu z tego pliku. Ta metoda
	 * odczytuje plik z dysku maksymalnie jeden raz.
	 * 
	 * @return obiekt klasy StringReader do czytania z pliku XML
	 * @throws IOException
	 */
	public Reader createTextReader() throws IOException {

//		return new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8"));
		
		if (fileContents == null) {
			BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF-8"));

			String line = null;
			StringBuilder buffer = new StringBuilder((int) file.length());

			while ((line = br.readLine()) != null) {
				buffer.append(line);
			}

			br.close();
			
			fileContents = buffer.toString();
			
			if (TEITextReader.class.isInstance(this)) {
				// zamiana zachowuje długość napisów, aby dobrze wyliczać startowe pozycje text blocków
				fileContents = fileContents.replace("&quot;", "\""); // zamiana dla text.xml
				fileContents = fileContents.replace("&amp;", "*"); // zamiana dla text.xml
				fileContents = fileContents.replace("&gt;", "}"); // zamiana dla text.xml
				fileContents = fileContents.replace("&lt;", "{"); // zamiana dla text.xml
				fileContents = fileContents.replace("&apos;", "'"); // zamiana dla text.xml
			}
		}
		StringReader sr = new StringReader(fileContents);
		sr.mark(0);

		return sr;
	}

	/**
	 * Tworzy nowy EventReader na nowym strumieniu StringReader.
	 * 
	 * @return obiekt XMLEventReader dla pliku
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public XMLEventReader createEventReader() throws XMLStreamException, IOException {
		return factory.createXMLEventReader(createTextReader());
	}

	/**
	 * Tworzy nowy StreamReader na nowym strumieniu StringReader.
	 * 
	 * @return obiekt XMLStreamReader dla pliku
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public XMLStreamReader2 createStreamReader() throws XMLStreamException, IOException {
		return (XMLStreamReader2)factory.createXMLStreamReader(createTextReader());
	}

	/**
	 * Probuje zwolnic wykorzystywane zasoby poprzez zwolnienie pamieci
	 * zarezerwowanej na wczytany plik, zamkniecie domyslnego parsera i
	 * strumienia znakowego.
	 * 
	 * @throws XMLStreamException
	 * @throws IOException 
	 * 
	 */
	public void close() throws XMLStreamException, IOException {
				
		if (reader != null) {
			reader.close();
			reader = null;
		}
		
		fileContents = null;
		file = null;
		corpusText = null;
	}

}
