package corpusapi.tei.readers;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import org.codehaus.stax2.XMLStreamReader2;

import corpusapi.tei.TEICorpusText;
import corpusapi.tei.TEISegment;
import corpusapi.tei.TEISegmentGroup;
import corpusapi.tei.TEISentenceSegmentGroup;

/**
 * @author Mateusz Kopeć
 *
 */
public class TEISegmentationReader extends TEIReader {

	/**
	 * Cache'owanie segmentów następuje względem id. Może być wyłączone lub
	 * ograniczone z góry przez zadaną liczbę segmentów.
	 */
	private Map<String, TEISegment> segmentCacheId;
	
	/*TODO: moznaby zrobic cache'owanie segmentow takze wzgledem innych metod niz pobieranie
	 * po id. */

	protected TEISegmentationReader(String fileName, TEICorpusText corpusText) {
		super(fileName, corpusText);
		if (cached) {
			if (cacheMaxItems == null) {
				segmentCacheId = new LinkedHashMap<String, TEISegment>();
			} else {
				segmentCacheId = new LinkedHashMap<String, TEISegment>() {
					private static final long serialVersionUID = -6839392540762832442L;

					protected boolean removeEldestEntry(Map.Entry<String, TEISegment> eldest) {
						return size() > cacheMaxItems;
					}
				};
			}
		}
	}

	/**
	 * Probuje zwolnic wykorzystywane zasoby poprzez zwolnienie pamieci
	 * zarezerwowanej na wczytany plik.
	 * 
	 * @throws XMLStreamException
	 * @throws IOException 
	 * 
	 */
	public void close() throws XMLStreamException, IOException {
		currentComment1 = null;
		currentComment4 = null;
		currentSegment1 = null;
		currentSegmentId4 = null;
		currentSentence2 = null;
		currentSentenceId1 = null;
		currentSentenceId4 = null;
		currentTextBlockId3 = null;
		currentTextBlockId4 = null;
		prevSegmentId4 = null;
		if (segmentCacheId != null) {
			segmentCacheId.clear();
			segmentCacheId = null;
		}
		if (sr1 != null) {
			sr1.close();
			sr1 = null;
		}
		if (sr2 != null) {
			sr2.close();
			sr2 = null;
		}
		if (sr3 != null) {
			sr3.close();
			sr3 = null;
		}
		if (sr4 != null) {
			sr4.close();
			sr4 = null;
		}
		super.close();
	}

	/**
	 * Metoda fabrykująca segment
	 * 
	 * @param segmentId
	 *            identyfikator segmentu
	 * @param nextSegmentId
	 * @param prevSegmentId
	 * @param startIdx
	 * @param endIdx
	 * @param segmentPosition
	 * @param sentenceId
	 * @param textBlockId
	 * @param corpusText
	 * @return utworzony segment
	 */
	public TEISegment createSegment(String segmentId, TEICorpusText corpusText, String nextSegmentId,
			String prevSegmentId, String textBlockId, String sentenceId, int startIdx, int endIdx, int segmentPosition,
			String pId, String resultOrth) {

		// w komentarzach są niepotrzebne spacje na początku i końcu
		resultOrth = resultOrth.trim();
		
		if (cached) {
			TEISegment segment = null;
			if ((segment = segmentCacheId.get(segmentId)) == null) {
				segment = new TEISegment(segmentId, corpusText, nextSegmentId, prevSegmentId, textBlockId, sentenceId,
						startIdx, endIdx, segmentPosition, pId, resultOrth);
				segmentCacheId.put(segmentId, segment);
			}
			return segment;
		} else {
			return new TEISegment(segmentId, corpusText, nextSegmentId, prevSegmentId, textBlockId, sentenceId,
					startIdx, endIdx, segmentPosition, pId, resultOrth);
		}
	}

	/* Metody dotyczące Segment */

	/**
	 * Zwraca pierwszy segment w tekście. Strumień jest zawsze tworzony od nowa.
	 * 
	 * @return TEISegment lub null, gdy nie ma
	 * @throws IOException
	 * @throws XMLStreamException
	 */
	public TEISegment getFirstSegmentItText() throws XMLStreamException, IOException {
		XMLStreamReader2 sr = createStreamReader();

		/* parametry wyniku */
		int resultStartIdx = 0;
		int resultEndIdx = 0;
		int resultSegmentPosition = 0;
		String resultSegmentId = null;
		String resultNextSegmentId = null;
		String resultSentenceId = null;
		String resultPrevSegmentId = null;
		String resultTextBlockId = null;
		String resultPId = null;
		String resultOrth = null;

		/* zmienne pomocnicze */
		boolean found = false;
		String currentSegmentId = null;
		String currentSentenceId = null;
		String currentTextBlockId = null;
		String currentPId = null;
		String prevSegmentId = null;
		String currentComment = null;

		while (sr.hasNext()) {
			int event = sr.next();
			if (event == XMLStreamReader2.COMMENT) {
				currentComment = sr.getText();
			} else if (event == XMLStreamReader2.START_ELEMENT) {
				QName name = sr.getName();
				String local = name.getLocalPart();

				if ("seg".equals(local)) {
					String rejected = sr.getAttributeValue("http://www.nkjp.pl/ns/1.0", "rejected");
					if (rejected != null && rejected.equals("true")) {
						continue;
					}
					currentSegmentId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");

					String corresp = sr.getAttributeValue(null, "corresp");
					String[] chopped = corresp.split("[(,)]");
					if (chopped.length > 3) {
						currentTextBlockId = chopped[1];
					} else {
						// System.out.println("WARNING! "+currentSegmentId+
						// " has wrong xpointer!");
						currentTextBlockId = "";
					}

					if (!found) { // jeszcze nie znaleziono segmentu
						found = true;

						resultOrth = currentComment;
						resultSegmentId = currentSegmentId;
						resultSentenceId = currentSentenceId;
						resultPrevSegmentId = prevSegmentId;
						resultTextBlockId = currentTextBlockId;
						resultPId = currentPId;
						resultSegmentPosition = 0;

						if (chopped.length > 3) {
							resultStartIdx = Integer.parseInt(chopped[2]);
							resultEndIdx = resultStartIdx + Integer.parseInt(chopped[3]);
						}
					} else { // segment juz znalezlismy wczesniej
						resultNextSegmentId = currentSegmentId;
						break; // wszystko juz wiemy
					}
					prevSegmentId = currentSegmentId;					
				} else if ("s".equals(local)) {
					currentSentenceId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
				} else if ("p".equals(local)) {
					currentPId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
				}
			}
		}

		TEISegment result = null;
		if (found) {
			result = createSegment(resultSegmentId, corpusText, resultNextSegmentId, resultPrevSegmentId,
					resultTextBlockId, resultSentenceId, resultStartIdx, resultEndIdx, resultSegmentPosition, resultPId, resultOrth);
		}

		return result;
	}

	/* zmienne zachowujące dane dotyczące aktualnej pozycji strumienia sr1 */
	private String currentComment1 = null;
	private String currentSentenceId1 = null;
	private String currentPId1 = null;
	private TEISegment currentSegment1 = null;
	private XMLStreamReader2 sr1;

	/**
	 * Sprawdza, czy strumień sr1 jest ustawiony na segmencie o danym id, jeżeli
	 * tak, to zwraca true, w przeciwnym przypadku tworzy nowy strumień
	 * ustawiony na początek pliku i zwraca false.
	 * 
	 * @param segmentId id szukanego segmentu
	 * @return czy strumień nie jest nowy
	 * 
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	private boolean setStreamReader1(String segmentId) throws XMLStreamException, IOException {
		if (currentSegment1 == null || !segmentId.equals(currentSegment1.getNextSegmentId())) {
			sr1 = createStreamReader();
			return false;
		}
		return true;
	}

	/**
	 * Zwraca TEISegment o zadanym id. Przy pytaniu o kolejne segmenty, korzystamy z jednego strumienia,
	 * więc działanie jest liniowe.
	 * 
	 * @param segmentId
	 * @return TEISegment lub null, gdy nie ma
	 * @throws IOException
	 * @throws XMLStreamException
	 */
	public TEISegment getSegmentById(String segmentId) throws XMLStreamException, IOException {

		/* korzystamy z cache'a, o ile to mozliwe */
		if (cached && segmentCacheId.containsKey(segmentId)) {
			return segmentCacheId.get(segmentId);
		}
		boolean oldStream = setStreamReader1(segmentId);

		/* parametry wyniku */
		int resultStartIdx = 0;
		int resultEndIdx = 0;
		int resultSegmentPosition = 0;
		String resultSegmentId = null;
		String resultNextSegmentId = null;
		String resultSentenceId = null;
		String resultPrevSegmentId = null;
		String resultTextBlockId = null;
		String resultPId = null;
		String resultOrth = null;

		/* zmienne pomocnicze */
		boolean found = false;
		int currentSegmentPosition = -1;
		String currentSegmentId = null;
		String currentSentenceId = null;
		String currentTextBlockId = null;
		String currentPId = null;
		String prevSegmentId = null;
		String currentComment = null;

		while (sr1.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = sr1.next();
			}
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
					currentSegmentPosition = currentSegment1.getSegmentPosition();
					currentSegmentId = currentSegment1.getNextSegmentId();
					currentSentenceId = currentSentenceId1;
					currentPId = currentPId1;
					currentComment = currentComment1;
					currentTextBlockId = currentSegment1.getTextBlockId();
					prevSegmentId = currentSegment1.getId();
				}

				QName name = sr1.getName();
				String local = name.getLocalPart();

				if ("seg".equals(local)) {
					String rejected = sr1.getAttributeValue("http://www.nkjp.pl/ns/1.0", "rejected");
					if (rejected != null && rejected.equals("true")) {
						continue;
					}
					currentSegmentId = sr1.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");

					if (!found) {
						String corresp = sr1.getAttributeValue(null, "corresp");
						String[] chopped = corresp.split("[(,)]");

						String textBlockId = "";
						if (chopped.length > 3) {
							textBlockId = chopped[1];
						} else {
							// System.out.println("WARNING! "+currentSegmentId+
							// " has wrong xpointer!");
						}

						if (textBlockId.equals(currentTextBlockId)) {
							currentSegmentPosition++;
						} else { // nowy blok
							currentTextBlockId = textBlockId;
							currentSegmentPosition = 0;
						}

						if (currentSegmentId.equals(segmentId)) {
							// znaleziono szukany segment
							found = true;

							resultOrth = currentComment;
							resultSegmentId = currentSegmentId;
							resultSentenceId = currentSentenceId;
							resultPrevSegmentId = prevSegmentId;
							resultTextBlockId = currentTextBlockId;
							resultPId = currentPId;
							resultSegmentPosition = currentSegmentPosition;

							if (chopped.length > 3) {
								resultStartIdx = Integer.parseInt(chopped[2]);
								resultEndIdx = resultStartIdx + Integer.parseInt(chopped[3]);
							}
						} else {
							// to jeszcze nie ten
							prevSegmentId = currentSegmentId;
						}
					} else { // juz mamy segment
						resultNextSegmentId = currentSegmentId;
						break;
					}
				} else if ("s".equals(local)) {
					currentSentenceId = sr1.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
				} else if ("p".equals(local)) {
					currentPId = sr1.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
				}
			} else if (event == XMLStreamReader2.COMMENT) {
				currentComment = sr1.getText();
			}
		}

		TEISegment result = null;
		if (found) {
			result = createSegment(resultSegmentId, corpusText, resultNextSegmentId, resultPrevSegmentId,
					resultTextBlockId, resultSentenceId, resultStartIdx, resultEndIdx, resultSegmentPosition, resultPId, resultOrth);

			currentComment1 = currentComment;
			currentSegment1 = result;
			currentSentenceId1 = currentSentenceId;
			currentPId1 = currentPId;
		} else {
			currentSegment1 = null;
		}

		return result;
	}

	/* Metody dotyczące SentenceSegmentGroup */

	/** Pobiera pierwszą SentenceSegmentGroup w tekście. Strumień jest tworzony za każdym razem.
	 * 
	 * @return SentenceSegmentGroup lub null, gdy nie ma. 
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEISegmentGroup getFirstSentenceSegmentGroup() throws XMLStreamException, IOException {
		XMLStreamReader2 sr = createStreamReader();

		boolean start = false;
		String currentSegmentId = null;
		List<String> segmentIdList = new ArrayList<String>();

		String resultId = null;
		String nextSentId = null;

		TEISentenceSegmentGroup result = null;

		while (sr.hasNext()) {
			int event = sr.next();
			if (event == XMLStreamReader2.START_ELEMENT) {

				QName name = sr.getName();
				String local = name.getLocalPart();

				if ("s".equals(local)) {
					String sentenceId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
					if (!start) {
						resultId = sentenceId;
						start = true;
					} else {
						nextSentId = sentenceId;
						break; // zaczyna sie nastepne zdanie
					}
				} else if (start && "seg".equals(local)) {
					String rejected = sr.getAttributeValue("http://www.nkjp.pl/ns/1.0", "rejected");
					if (rejected != null && rejected.equals("true")) {
						continue;
					}
					currentSegmentId = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
					segmentIdList.add(currentSegmentId);
				}
			}
		}

		if (resultId != null) {
			result = new TEISentenceSegmentGroup(resultId, segmentIdList, nextSentId, null, corpusText);
		}
		sr.close();
		return result;
	}

	/* zmienne zachowujące dane dotyczące aktualnej pozycji strumienia sr2 */
	private TEISentenceSegmentGroup currentSentence2 = null;
	private XMLStreamReader2 sr2;
	
	/**
	 * Sprawdza, czy strumień sr2 jest ustawiony na grupie o danym id, jeżeli
	 * tak, to zwraca true, w przeciwnym przypadku tworzy nowy strumień
	 * ustawiony na początek pliku i zwraca false.
	 * 
	 * @param groupId
	 * @return czy strumień nie jest nowy
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	private boolean setStreamReader2(String groupId) throws XMLStreamException, IOException {
		if (currentSentence2 == null || !groupId.equals(currentSentence2.nextGroupId)) {
			sr2 = createStreamReader();
			return false;
		}
		return true;
	}

	/**
	 * Zwraca SentenceSegmentGroup o zadanym id. Przy pytaniu o kolejne grupy, korzystamy z jednego
	 * strumienia, więc działanie jest liniowe.
	 * 
	 * @param sentenceGroupId
	 * @return SentenceSegmentGroup lub null, gdy nie ma
	 * @throws IOException
	 * @throws XMLStreamException
	 */
	public TEISentenceSegmentGroup getSentenceSegmentGroupById(String sentenceGroupId) throws XMLStreamException,
			IOException {
		boolean oldStream = setStreamReader2(sentenceGroupId);

		boolean start = false;
		String currentSegmentId = null;
		List<String> segmentIdList = new ArrayList<String>();

		String nextSentId = null;
		String prevSentId = null;

		TEISentenceSegmentGroup result = null;

		while (sr2.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = sr2.next();
			}
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
					prevSentId = currentSentence2.getId();
				}

				QName name = sr2.getName();
				String local = name.getLocalPart();

				if ("s".equals(local)) {
					if (!start) {
						String sentenceId = sr2.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");

						if (sentenceGroupId.equals(sentenceId)) {
							start = true;
						} else {
							prevSentId = sentenceId;
						}
					} else {
						nextSentId = sr2.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
						break; // zaczyna sie nastepne zdanie
					}
				} else if (start && "seg".equals(local)) {
					String rejected = sr2.getAttributeValue("http://www.nkjp.pl/ns/1.0", "rejected");
					if (rejected != null && rejected.equals("true")) {
						continue;
					}
					currentSegmentId = sr2.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
					segmentIdList.add(currentSegmentId);
				}
			}
		}

		if (sentenceGroupId != null) {
			result = new TEISentenceSegmentGroup(sentenceGroupId, segmentIdList, nextSentId, prevSentId, corpusText);
			currentSentence2 = result;
		} else {
			currentSentence2 = null;
		}
		return result;
	}

	/* Metody dotyczące TextBlockow */

	/* zmienne zachowujące dane dotyczące aktualnej pozycji strumienia sr3 */
	private String currentTextBlockId3 = null;
	private XMLStreamReader2 sr3;

	/**
	 * Sprawdza, czy strumień sr3 jest ustawiony na początku bloku o danym id, jeżeli
	 * tak, to zwraca true, w przeciwnym przypadku tworzy nowy strumień
	 * ustawiony na początek pliku i zwraca false.
	 * 
	 * @param blockId
	 * @return
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	private boolean setStreamReader3(String blockId) throws XMLStreamException, IOException {
		if (currentTextBlockId3 == null || !currentTextBlockId3.equals(blockId)) {
			sr3 = createStreamReader();
			return false;
		}
		return true;
	}

	/**
	 * Zwraca ilosc segmentow w podanym bloku. Jeżeli pobieramy długości kolejnych bloków,
	 * to korzystamy z jednego strumienia, więc działanie jest liniowe.
	 * 
	 * @param blockId
	 * @return ilosc segmentow bloku lub 0, gdy blok nie znaleziony
	 * @throws IOException
	 * @throws XMLStreamException
	 */
	public int getBlockSize(String blockId) throws XMLStreamException, IOException {
		boolean oldStream = setStreamReader3(blockId);

		currentTextBlockId3 = null;
		int foundSegments = 0;

		while (sr3.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = sr3.next();
			}
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
				}

				QName name = sr3.getName();
				String local = name.getLocalPart();

				if ("seg".equals(local)) {
					String rejected = sr3.getAttributeValue("http://www.nkjp.pl/ns/1.0", "rejected");
					if (rejected != null && rejected.equals("true")) {
						continue;
					}
					
					String corresp = sr3.getAttributeValue(null, "corresp");

					if (corresp.contains(blockId)) {
						foundSegments++;
					} else if (foundSegments > 0) { // kolejny blok
						String[] chopped = corresp.split("[(,)]");
						if (chopped.length > 3) {
							currentTextBlockId3 = chopped[1];
						} else {
							// System.out.println("WARNING! Segment no."+foundSegments+
							// " in block "+
							// blockId+" has wrong xpointer!");
						}
						break;
					}
				}
			}
		}	

		return foundSegments;
	}

	/* zmienne zachowujące informacje o aktualnej pozycji strumienia sr4 */
	private String currentTextBlockId4 = null;
	private String currentSentenceId4 = null;
	private String currentPId4 = null;
	private String currentSegmentId4 = null;
	private String prevSegmentId4 = null;
	private String currentComment4 = null;
	private XMLStreamReader2 sr4 = null;

	/**
	 * Sprawdza, czy strumień sr4 jest ustawiony na początku bloku o danym id, jeżeli
	 * tak, to zwraca true, w przeciwnym przypadku tworzy nowy strumień
	 * ustawiony na początek pliku i zwraca false.
	 *  
	 * @param blockId
	 * @return
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	private boolean setStreamReader4(String blockId) throws XMLStreamException, IOException {
		if (currentTextBlockId4 == null || !currentTextBlockId4.equals(blockId)) {
			sr4 = createStreamReader();
			return false;			
		}				
		return true;
	}

	/**
	 * Zwraca pierwszy segment w danym bloku. Jeśli blok jest pusty, zwraca null. 
	 * Cacheowanie strumienia działa, gdy prosimy o pierwsze segmenty kolejnych bloków tekstu. 
	 * 
	 * TODO: nie zwraca null, kiedy dojdzie do końca bloku! Zwraca pusty element
	 * MK: teraz powinno byc dobrze
	 * 
	 * @param blockId
	 * @return TEISegment lub null, gdy nie ma
	 * @throws IOException
	 * @throws XMLStreamException
	 */
	public TEISegment getFirstSegmentInBlock(String blockId) throws XMLStreamException, IOException {
		boolean oldStream = setStreamReader4(blockId);

		currentTextBlockId4 = null;
		
		/* parametry wyniku */
		int resultStartIdx = 0;
		int resultEndIdx = 0;
		int resultSegmentPosition = 0;
		String resultSegmentId = null;
		String resultNextSegmentId = null;
		String resultSentenceId = null;
		String resultPrevSegmentId = null;
		String resultTextBlockId = null;
		String resultPId = null;
		String resultOrth = null;
		
		/* zmienne pomocnicze */
		boolean found = false;
		String currentSegmentId = null;
		String currentSentenceId = null;
		String currentTextBlockId = null;
		String currentPId = null;
		String prevSegmentId = null;
		String currentComment = null;

		while (sr4.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = sr4.next();
			}
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
					currentComment = currentComment4;
					currentSegmentId = currentSegmentId4;
					currentSentenceId = currentSentenceId4;
					currentPId = currentPId4;
					currentTextBlockId = blockId;
					prevSegmentId = prevSegmentId4;
				}
				QName name = sr4.getName();
				String local = name.getLocalPart();

				if ("seg".equals(local)) {
					String rejected = sr4.getAttributeValue("http://www.nkjp.pl/ns/1.0", "rejected");
					if (rejected != null && rejected.equals("true")) {
						continue;
					}
					currentSegmentId = sr4.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
					
					String corresp = sr4.getAttributeValue(null, "corresp");
					String[] chopped = corresp.split("[(,)]");
					if (chopped.length > 3) {
						currentTextBlockId = chopped[1];
					} else {
						// System.out.println("WARNING! "+currentSegmentId+
						// " has wrong xpointer!");
						currentTextBlockId = "";
					}

					if (!found) { // jeszcze nie znaleziono segmentu
						if (blockId.equals(currentTextBlockId)) { 
							// znalezlismy segment
							found = true;

							resultOrth = currentComment;
							resultPId = currentPId;
							resultSegmentId = currentSegmentId;
							resultSentenceId = currentSentenceId;
							resultPrevSegmentId = prevSegmentId;
							resultTextBlockId = currentTextBlockId;
							resultSegmentPosition = 0;

							if (chopped.length > 3) {
								resultStartIdx = Integer.parseInt(chopped[2]);
								resultEndIdx = resultStartIdx + Integer.parseInt(chopped[3]);
							}
						}
					} else { // segment juz znalezlismy wczesniej
						if (resultNextSegmentId == null) {
							resultNextSegmentId = currentSegmentId;
						}
						if (!currentTextBlockId.equals(blockId)) {
							prevSegmentId4 = prevSegmentId;
							currentSegmentId4 = currentSegmentId;
							currentSentenceId4 = currentSentenceId;
							currentTextBlockId4 = currentTextBlockId;
							currentPId4 = currentPId;
							currentComment4 = currentComment;
							break; // wszystko juz wiemy
						}
					}
					prevSegmentId = currentSegmentId;
					
				} else if ("s".equals(local)) {
					currentSentenceId = sr4.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
				} else if ("p".equals(local)) {
					currentPId = sr4.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
				}
			} else if (event == XMLStreamReader2.COMMENT) {
				currentComment = sr4.getText();
			}
		}

		TEISegment result = null;

		if (found) {
			result = createSegment(resultSegmentId, corpusText, resultNextSegmentId, resultPrevSegmentId,
					resultTextBlockId, resultSentenceId, resultStartIdx, resultEndIdx, resultSegmentPosition, resultPId, resultOrth);
		}

		return result;
	}

}
