/*
 * This file is part of the Poliqarp suite.
 * 
 * Copyright (C) 2004-2009 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact
 * Michal.Ciesiolka@ipipan.waw.pl or ipi@ipipan.waw.pl for more
 * information).  Licensees holding a valid commercial license from IPI
 * PAN may use this file in accordance with that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

package ipipan.poliqarp.connection;

import java.util.EnumMap;
import java.util.Map;

/**
 * Manager of the options that can be set for a connection. These include
 * widths of context, which interpretations are to be transmitted, etc.
 */
public final class ConnectionOptions
{
   private PoliqarpConnection connection; 
   private boolean disamb;
   
   /**
    * Constructs a default set of options for the given connection.
    * 
    * @param connection connection associated with these options
    */
   public ConnectionOptions(PoliqarpConnection connection)
   {
      this.connection = connection;
      disamb = true;
   }

   private static void appendDigit(StringBuilder b, boolean digit)
   {
      b.append(digit ? '1' : '0');
   }

   private static void appendMatchPartsDescription(StringBuilder b, 
      boolean lc, boolean lm, boolean rm, boolean rc)
   {
      appendDigit(b, lc);
      appendDigit(b, lm);
      appendDigit(b, rm);
      appendDigit(b, rc);
   }

   /**
    * Set default flags for the queries.
    * 
    * @param qi whether main part of query should be case-insensitive
    * @param qx if true, then main part of query should not be constrained to 
    *           whole words
    * @param mi whether metadata constraint should be case-insensitive
    * @param mx if true, then metadata constraint should not be constrained to 
    *           whole words
    */
   public void setQueryFlags(boolean qi, boolean qx, boolean mi, boolean mx)
   {
      StringBuilder msg = new StringBuilder("SET query-flags ");
      appendMatchPartsDescription(msg, qi, qx, mi, mx);
      try { 
         connection.send(msg.toString());
         if (!connection.getMessage().isOK())
            throw new Exception();
      } catch (Exception e) {
         System.err.println("Failed to set options: query-flags");
      }
   }

   /**
    * Toggles between sending disambiguated and ambiguous interpretations.
    *
    * @param disamb whether to send disambiguated interpretations 
    */    
   public void setAmbiguationDisplay(boolean disamb) 
   {
      try {
         connection.send("SET disamb " + (disamb ? "1" : "0"));
         if (!connection.getMessage().isOK())
            throw new Exception();
         this.disamb = disamb;
      } catch (Exception e) {}
   }
}
