/*
 * This file is part of the Poliqarp suite.
 * 
 * Copyright (C) 2004-2009 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact
 * Michal.Ciesiolka@ipipan.waw.pl or ipi@ipipan.waw.pl for more
 * information).  Licensees holding a valid commercial license from IPI
 * PAN may use this file in accordance with that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

package ipipan.poliqarp.logic;

import java.io.*;
import java.util.HashSet;
import java.util.Set;

import ipipan.poliqarp.connection.PoliqarpConnection;
import ipipan.poliqarp.util.PoliqarpUtilities;

/**
 * A segment (also known as a position) is a single element of a corpus.
 */
public final class Segment
{
   private String orth;
   private Interpretation[] interps;

   /**
    * Constructs a <code>Segment</code> that belongs to a query result, 
    * reading its description from a connection.
    * @param connection the connection to read the segment description from.
    */
   public Segment(PoliqarpConnection connection) throws IOException
   {
      orth = connection.getStrMessage();
      int n = Integer.parseInt(connection.getStrMessage());
      interps = new Interpretation[n];
      for (int i = 0; i < n; i++)
         interps[i] = new Interpretation(connection);
   }

   /**
    * Returns this segment as a <code>String</code>.
    */
   public String toString(boolean dorth, boolean dbase, boolean dtag)
   {
      StringBuilder res = new StringBuilder(!dorth || orth == null ? "" : orth);
      if (dbase || dtag) {
          Set<String> uniq = new HashSet<String>();
         for (Interpretation i : interps) {
            String str = i.toString(dbase, dtag);
            if (!uniq.contains(str)) {
               res.append(" ");
               res.append(str);
               uniq.add(str);
            }
         }
      }
      return res.toString();
   }     
   
   /**
    * Returns this segment as a HTML-formatted <code>String</code>.
    */
   public String toHTML(boolean dorth, boolean dbase, boolean dtag, boolean bold)
   {
      StringBuilder res = new StringBuilder();
      if (dorth) {
         if (bold)
            res.append("<strong>");
         res.append(PoliqarpUtilities.htmlSpecialChars(orth));
         if (bold)
            res.append("</strong>");
      }
      if (dbase || dtag) {
          Set<String> uniq = new HashSet<String>();
         for (Interpretation i : interps) {
            String str = i.toString(dbase, dtag);
            if (!uniq.contains(str)) {
               res.append(" ");
               res.append(PoliqarpUtilities.htmlSpecialChars(str));
               uniq.add(str);
            }
         }
      }
      return res.toString();
   }     

   public String getOrth() {
      return orth;
   }

   // FIXME - change to getInterp
   public Interpretation[] getDisamb() {
      return interps;
   }
}
