/*
 * This file is part of the Poliqarp suite.
 * 
 * Copyright (C) 2004-2009 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact
 * Michal.Ciesiolka@ipipan.waw.pl or ipi@ipipan.waw.pl for more
 * information).  Licensees holding a valid commercial license from IPI
 * PAN may use this file in accordance with that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

package ipipan.poliqarp.logic;

import ipipan.poliqarp.connection.AsyncHandler;
import ipipan.poliqarp.connection.PoliqarpConnection;
import ipipan.poliqarp.connection.Message;

/**
 * A server job that may take a noticeable time.
 */
public abstract class ServerJob implements Job, AsyncHandler
{
   /**
    * The connection used to inquire about the status of this job.
    */
   protected PoliqarpConnection connection;

   /**
    * A Runnable object that is run upon completion of the job. This can
    * also be <code>null</code>, indicating a no-op.
    */
   protected Runnable notifier;

   private int lastProgress = 0;

   /**
    * Constructs a job that communicates about its result via the given
    * connection.
    */
   public ServerJob(PoliqarpConnection connection)
   {
      this.connection = connection;
      notifier = null;
      lastProgress = 0;
   }

   /**
    * Returns the progress of this job, in percents.
    * 
    * @return an integer in the range 0 through 100 inclusive, representing 
    * advance in percents of this job. If it's already completed, return 100.
    */
   public int getProgress() 
   {
      try {
         connection.send("STATUS");
         Message reply = connection.getMessage();
         if (reply.isOK()) 
            lastProgress = reply.getOKIntInfo(0);
         return lastProgress;
      } catch (Exception e) {
         return 0;
      }
   }

   /**
    * Sets a notifier for this job.
    */
   public void setNotifier(Runnable notifier)
   {
      this.notifier = notifier;
   }

   /**
    * Launches a notifier if it is non-null and sets it to null. The notifier
    * is called in the same thread as the caller.
    */
   protected void launchNotifier()
   {
      if (notifier != null)
         javax.swing.SwingUtilities.invokeLater(notifier);
   }

   /**
    * Each subclass should override this method, implementing a handler 
    * for messages sent by this job.
    * @param message the asynchronous message
    */
   public abstract void handle(String message);

   /**
    * Sets the 'direct unsure' mode for the underlying connection.
    *
    * @see PoliqarpConnection#setDirectUnsafeMode(boolean)
    */
   public void setDirectUnsafeMode(boolean val)
   {
      connection.setDirectUnsafeMode(val);
   }
}
