/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.client.activity;

import java.util.ArrayList;
import java.util.Set;

import javax.validation.ConstraintViolation;

import pl.waw.ipipan.corpcor.client.ClientFactory;
import pl.waw.ipipan.corpcor.client.place.SearchDetailsPlace;
import pl.waw.ipipan.corpcor.client.place.SearchPlace;
import pl.waw.ipipan.corpcor.client.view.SearchDetailsView;
import pl.waw.ipipan.corpcor.shared.AnnotationProxy;
import pl.waw.ipipan.corpcor.shared.AnnotationProxyUtils;
import pl.waw.ipipan.corpcor.shared.AnnotationRequest;
import pl.waw.ipipan.corpcor.shared.ResultsPage;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusSegment;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpContext;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpMetaData;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpResult;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpResult.PQResultState;

import com.google.gwt.activity.shared.AbstractActivity;
import com.google.gwt.editor.client.SimpleBeanEditorDriver;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.shared.EventBus;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.google.gwt.user.client.ui.AcceptsOneWidget;
import com.google.gwt.view.client.HasData;
import com.google.gwt.view.client.MultiSelectionModel;
import com.google.gwt.view.client.SingleSelectionModel;
import com.google.web.bindery.requestfactory.shared.Receiver;

public class SearchDetailsActivity extends AbstractActivity implements SearchDetailsView.Presenter {

    private ClientFactory clientFactory;
    private SearchDetailsPlace place;

    public SearchDetailsActivity(SearchDetailsPlace place, ClientFactory clientFactory) {
        this.clientFactory = clientFactory;
        this.place = place;
    }

    @Override
    public void start(AcceptsOneWidget panel, EventBus eventBus) {
        final SearchDetailsView view = clientFactory.getSearchDetailsView();
        view.clear();
        view.setPresenter(this);
        retrieveData();
        panel.setWidget(view.asWidget());
    }
    
    private ArrayList<CorpusSegment> storedSegments = null;

    private void retrieveData() {
        final SearchDetailsView searchDetailsView = clientFactory.getSearchDetailsView();
        clientFactory.getRpcService().getPoliqarpWideContext(place.getEntryId().resultId, new AsyncCallback<PoliqarpContext>() {
            @Override
            public void onSuccess(PoliqarpContext result) {
                searchDetailsView.setContext(result.getWideContext());
            }

            @Override
            public void onFailure(Throwable result) {
                searchDetailsView.setContext("Failed to retrieve context");
            }
        });
        clientFactory.getRpcService().getPoliqarpMetaData(place.getEntryId().resultId, new AsyncCallback<PoliqarpMetaData>() {
            @Override
            public void onSuccess(PoliqarpMetaData result) {
                searchDetailsView.setMetadata(result.toString());
            }

            @Override
            public void onFailure(Throwable result) {
                searchDetailsView.setMetadata("Failed to retrieve metadata");
            }
        });
        clientFactory.getRpcService().getCorpusSegments(place.getQuery(), place.getEntryId().resultId,
                new AsyncCallback<ArrayList<CorpusSegment>>() {
                    @Override
                    public void onSuccess(ArrayList<CorpusSegment> result) {
                        storedSegments = result;
                        searchDetailsView.setResultSegments(result);
                    }

                    @Override
                    public void onFailure(Throwable result) {
                        Window.alert("Nie pobrano segmentów: " + result);
                        storedSegments = null;
                        searchDetailsView.setResultSegments(new ArrayList<CorpusSegment>(0));
                    }
                });
    }

    @Override
    public void onCancelButtonClicked(ClickEvent e) {
        clientFactory.getPlaceController().goTo(new SearchPlace(place.getQuery()));
    }

    @Override
    public void onEditButtonClicked(ClickEvent e) {
        clientFactory.getPlaceController().goTo(new SearchPlace(place.getQuery()));
    }

    @Override
    public void onSegmentClicked(final CorpusSegment segment) {
        SearchDetailsView editCommentView = clientFactory.getSearchDetailsView();
        final SimpleBeanEditorDriver<AnnotationProxy, ?> editorDriver = editCommentView.getEditorDriver();
        AnnotationRequest request = clientFactory.getRequestFactory().annotationRequest();
        request.findLatestByPath(segment.getSegmentId(), segment.getBlockId(), segment.getTextId())
                .fire(new Receiver<AnnotationProxy>() {
                    public void onSuccess(AnnotationProxy annotationProxy) {
                        AnnotationRequest editRequest = clientFactory.getRequestFactory().annotationRequest();
                        AnnotationProxy editProxy = editRequest.create(AnnotationProxy.class);
                        if (annotationProxy == null) {
                            AnnotationProxyUtils.populateFromCorpusSegment(segment, editProxy);
                        } else {
                            AnnotationProxyUtils.populateFromAnnotationProxy(annotationProxy, editProxy);
                        }
                        editorDriver.edit(editProxy);
                    };
                });
    }

    void saveSegmentWithId(AnnotationProxy annotationProxy, String query, String textId, String blockId, String segmentId ) {
    	AnnotationRequest editRequest = clientFactory.getRequestFactory().annotationRequest();
        AnnotationProxy editProxy = editRequest.create(AnnotationProxy.class);
        AnnotationProxyUtils.cloneAnnotationProxy(annotationProxy, editProxy);
        editProxy.setBlockId(blockId);
        editProxy.setSegmentId(segmentId);
        editProxy.setTextId(textId);
        editProxy.setQuery(query);
        
        editRequest.persist(editProxy);
        
        System.out.println("Saving proxy " + textId + " " + blockId + " " + segmentId);
        editRequest.fire(new Receiver<Void>() {
            @Override
            public void onSuccess(Void arg0) {
                System.out.println("SearchDetailsActivity.saveSegmentWithId().new Receiver() {...}.onSuccess()");
                clientFactory.getSearchView().refresh();
            }
            
            @Override
            public void onConstraintViolation(Set<ConstraintViolation<?>> violations) {
                for (ConstraintViolation<?> cv : violations) {
                    Window.alert(cv.getMessage() + " (" + cv.getInvalidValue() + "): " + cv.getPropertyPath());
                }
            }
        });
    }

    @Override
    public void onSaveAll() {
    	SearchDetailsView searchDetailsView = clientFactory.getSearchDetailsView();
        SimpleBeanEditorDriver<AnnotationProxy, ?> editorDriver = searchDetailsView.getEditorDriver();
        final AnnotationProxy annotationProxy = editorDriver.flush();
        
        if (storedSegments == null) {
            System.err.println("Brak segmentow!");
            return;
        }
        
        final String query = this.place.getQuery();

        // garbage start -->
        HasData<CorpusSegment> segmentsTable = this.clientFactory.getSearchDetailsView().getSegmentsTable();
        SingleSelectionModel<CorpusSegment> selectionModel = (SingleSelectionModel<CorpusSegment>) segmentsTable.getSelectionModel();
        CorpusSegment selectedSegment = selectionModel.getSelectedObject();
        int _idxOfSelectedSegment = -1;
        for (int i = 0; i < storedSegments.size(); i++) {
            if (storedSegments.get(i).getSegmentId().equals(selectedSegment.getSegmentId()))
                _idxOfSelectedSegment = i;
        }
        if (_idxOfSelectedSegment == -1) {
            System.err.println("Nie dopasowano wybranego segmentu!");
            return;
        }
        final int idxOfSelectedSegment = _idxOfSelectedSegment;
        // <-- garbage end
        
        final int[] succeeded = new int[1];
        final int[] failed = new int[1];
        // TODO do przeniesienia na strone serwera... tutaj to bardzo nieefektywne
        MultiSelectionModel<PoliqarpResult> resultRowSelectionModel = (MultiSelectionModel<PoliqarpResult>) clientFactory.getSearchView().getTable().getSelectionModel();
        Set<PoliqarpResult> selectedResultRows = resultRowSelectionModel.getSelectedSet();
        final int total = selectedResultRows.size();
        for (PoliqarpResult pqr : selectedResultRows) {
            this.clientFactory.getRpcService().getQueryResult(query, pqr.getEntryId().resultId,
                    new AsyncCallback<PoliqarpResult>() {
                @Override
                public void onSuccess(PoliqarpResult pqresult) {
                    if (pqresult.getResultState() == PQResultState.CORPUS_UPDATE_OK) {
                        // nie wiadomo jak wyciagnac numer wybranego wiersza w celltable
                        // zeby wiedziec ktory segment jest edytowany - dlatego jest to dziwne idxOfSelectedSegment
                        CorpusSegment matchSegment = pqresult.getMatchSegments()[idxOfSelectedSegment];
                        saveSegmentWithId(annotationProxy, query, matchSegment.getTextId(), matchSegment.getBlockId(), matchSegment.getSegmentId());
                        succeeded[0]++;
                    } else {
                        failed[0]++;
                    }
                    //checkLast();
                }
                
                private void checkLast() {
                    if (succeeded[0] + failed[0] >= total) {
                        Window.alert("Operacja powiodła się dla " + succeeded[0] + " na " + (succeeded[0] + failed[0]) + " rekordów");
                        clientFactory.getSearchView().refresh();
                    }
                }

                @Override
                public void onFailure(Throwable throwable) {
                    failed[0]++;
                    checkLast();
                }
            });
        }
    }

    @Override
    public void onSaveOne() {
        SearchDetailsView searchDetailsView = clientFactory.getSearchDetailsView();
        SimpleBeanEditorDriver<AnnotationProxy, ?> editorDriver = searchDetailsView.getEditorDriver();
        final AnnotationProxy annotationProxy = editorDriver.flush();
        
        if (editorDriver.hasErrors()) {
            Window.alert("Formularz zawiera błędy.");
            return;
        }
        final String query = this.place.getQuery();
        saveSegmentWithId(annotationProxy, query, annotationProxy.getTextId(), annotationProxy.getBlockId(), annotationProxy.getSegmentId());
    }
}
