package pl.waw.ipipan.corpcor.server.corpusapi;

import java.util.ArrayList;
import java.util.ResourceBundle;
import java.util.Set;

import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.context.annotation.ScopedProxyMode;
import org.springframework.stereotype.Component;

import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEICorpus;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEICorpusText;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEIMorphoSegmentGroup;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEISegment;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEITextBlock;
import pl.waw.ipipan.corpcor.shared.ResultsPage;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusApiException;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusSegment;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusSegmentNotFoundException;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusTextBlock;
import pl.waw.ipipan.corpcor.shared.corpusapi.TextBlockSegments;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpMetaData;

import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

@Component
@Scope(value="session",proxyMode=ScopedProxyMode.TARGET_CLASS)
public class CorpusAPIBridge {
    
    private static class PQCorpusMetadata implements CorpusMetadata {
        private final PoliqarpMetaData pqmeta;

        public String[] getMetadataValues(String metadataKey) {
            return pqmeta.getMetadataValues(metadataKey);
        }

        public Set<String> getMetadataKeySet() {
            return pqmeta.getMetadataKeySet();
        }

        public PQCorpusMetadata(PoliqarpMetaData pqmeta) {
            this.pqmeta = pqmeta;
        }
        
        @Override
        public String toString() {
            return this.pqmeta.toString();
        }
    }
    
	private static final Logger LOG = LoggerFactory.getLogger(CorpusAPIBridge.class);

	private TEICorpus teiCorpus;
	private boolean open;

	public CorpusAPIBridge() {
	    ResourceBundle rb = ResourceBundle.getBundle("corpus");
		CorpusFactory factory = CorpusFactory.getInstance();
		this.teiCorpus = (TEICorpus) factory.getCorpus(rb.getString("corpus-path"), true); 
	}

	private TEICorpus getCorpus() {
		if (!open) {
			open = teiCorpus.open();
		}
		
		if (open) {
			return teiCorpus;
		} else {
			return null;
		}
	}
	
	@PreDestroy
	public void cleanup() {
		LOG.info("Doing cleanup");
		teiCorpus.close();
		teiCorpus = null;
	}

    public synchronized ArrayList<CorpusSegment> getCorpusTextBlocks(PoliqarpMetaData meta, String match, String lContext, String rContext)
            throws CorpusApiException {
        TEICorpus corpus = this.getCorpus();
        if (corpus == null) {
            throw new CorpusApiException("Could not open corpus.");
        }
        
        String leftNoWS = lContext.replaceAll("\\s", "");
        String rightNoWS = rContext.replaceAll("\\s", "");
        String matchNoWS = match.replaceAll("\\s", "");

        // get the text for metadata
        final TEICorpusText corpusText;
        if (meta.getMetadataValues("textId").length != 1) {
            LOG.error("For PQ query match: " + match + " no textId in metadata was found: " + meta + ", trying other metadata");
            corpusText = (TEICorpusText) corpus.getCorpusText(new PQCorpusMetadata(meta));
        } else {
            corpusText = (TEICorpusText) corpus.getCorpusText(meta.getMetadataValues("textId")[0]);
        }
        if (corpusText == null) {
            throw new CorpusSegmentNotFoundException("For PQ query match: " + match + " metadata was not matched among texts: " + meta);
        }
        try {
            // the corpus is browsed faster by segment links, rather by going through
            // each block and then each segment in a block
            NEXT_SEGMENT: for (Segment segment = corpusText.getFirstSegment(); 
                    segment != null; 
                    segment = segment.getNext()) {
                String orth = segment.getOrth().replaceAll("\\s", "");
                if (matchNoWS.startsWith(orth)) {
                    // check further segments if this is a total match
                    Segment firstSegmentOfMatch = segment;
                    Segment lastSegmentOfMatch = segment;
                    int segmentsInMatch = 1;
                    int position = 0 + orth.length();
                    if (position < matchNoWS.length()) {
                        // only if there is more to match - which is only the case of multiple
                        // words in a match
                        for (Segment nextSegment = segment.getNext(ContinueMode.CONTINUOUS); 
                                nextSegment != null && position < matchNoWS.length(); 
                                nextSegment = nextSegment.getNext(ContinueMode.CONTINUOUS)) {
                            String nextOrth = nextSegment.getOrth().replaceAll("\\s", "");
                            int orthLen = nextOrth.length();
                            if (matchNoWS.regionMatches(position, nextOrth, 0, orthLen)) {
                                position += orthLen;
                                lastSegmentOfMatch = nextSegment;
                                segmentsInMatch++;
                            } else {
                                // failed to match. find other segment for
                                // the main match
                                continue NEXT_SEGMENT;
                            }
                        }
                        if (position < matchNoWS.length()) {
                            // we didn't reach end of the match and there are no more segments..
                            if (lastSegmentOfMatch != null)
                                segment = segment.getNeighbor(segmentsInMatch - 1);
                            continue NEXT_SEGMENT;
                        }
                    }
                    
                    // now crawl segments to the left. match as many segments as possible
                    position = leftNoWS.length();
                    for (Segment prevSegment = firstSegmentOfMatch.getPrev(ContinueMode.CONTINUOUS); 
                            prevSegment != null && position > 0; 
                            prevSegment = prevSegment.getPrev(ContinueMode.CONTINUOUS)) {
                        String prevOrth = prevSegment.getOrth().replaceAll("\\s", "");
                        int orthLen = prevOrth.length();
                        if (leftNoWS.regionMatches(position - orthLen, prevOrth, 0, orthLen)) {
                            position -= orthLen;
                        } else {
                            // failed to match. find other segment for the main match
                            continue NEXT_SEGMENT;
                        }
                    }
                    
                    // now crawl segments to the left
                    position = 0;
                    for (Segment nextSegment = lastSegmentOfMatch.getNext(ContinueMode.CONTINUOUS); 
                            nextSegment != null && position < rightNoWS.length(); 
                            nextSegment = nextSegment.getNext(ContinueMode.CONTINUOUS)) {
                        String nextOrth = nextSegment.getOrth().replaceAll("\\s", "");
                        int orthLen = nextOrth.length();
                        if (rightNoWS.regionMatches(position, nextOrth, 0, orthLen)) {
                            position += orthLen;
                        } else {
                            // failed to match. find other segment for the main match
                            continue NEXT_SEGMENT;
                        }
                    }
                    // found exact location. whether its the right one, depends on whether
                    // the provided contexts appear just once in the whole text, but 
                    // the only way to prevent is to increase context size..
                    ArrayList<CorpusSegment> resultSegments = new ArrayList<CorpusSegment>(segmentsInMatch);

                    for (Segment resultSegment = firstSegmentOfMatch;
                            resultSegment != null; 
                            resultSegment = resultSegment.getNext()) {
                        TextBlock textBlock = segment.getTextBlock();
                        resultSegments.add(createCorpusSegment(corpusText.getTextId(), textBlock.getId(), resultSegment));
                        if (resultSegment.equals(lastSegmentOfMatch))
                            break;
                    }
                    return resultSegments;
                }
            }
        } catch (Exception e) {
            throw new CorpusApiException("Exception while reading XML.", e);
        } finally {
            corpusText.closeCorpusText();
        }
        throw new CorpusSegmentNotFoundException("For PQ query match: " + match + " exact context was not matched in text: " + corpusText.getTextId());
    }
    
    private CorpusTextBlock createCorpusTextBlock(TextBlock textBlock) throws Exception {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < textBlock.getSize(); i++) {
            Segment teiSegment;
            teiSegment = textBlock.getSegment(i);
            if (teiSegment == null) {
                break;
            }
            sb.append(teiSegment.getOrth() + " ");
        }
        CorpusTextBlock corpusTextBlock = new CorpusTextBlock(textBlock.getId(), 
                textBlock.getType().toString(), sb.toString());
        return corpusTextBlock;
    }

    private CorpusSegment createCorpusSegment(String textId, String blockId, Segment teiSegment) {
        CorpusSegment corpusSegment;
        corpusSegment = new CorpusSegment(textId, blockId, teiSegment.getId());
        TEIMorphoSegmentGroup tmsg = (TEIMorphoSegmentGroup) teiSegment.getSegmentGroup(MorphoSegmentGroup.class);
        if (tmsg != null) {
            corpusSegment.setMorpho(teiSegment.getOrth(), tmsg.getChosenInterpretation().getBase(), 
                    tmsg.getChosenInterpretation().getMsd(), 
                    tmsg.getChosenInterpretation().getCtag());
        }
        return corpusSegment;
    }

	public synchronized ResultsPage<String> getCorpusTextIds(String query, int start,
			int length) throws CorpusApiException {
		TEICorpus corpus = this.getCorpus();
		if (corpus == null) {
			throw new CorpusApiException("Could not open corpus.");
		}
        
    	Iterable<String> textIds = corpus.getCorpusTextIds();
    	
        if ((query != null) && (query.length() > 0)) {
        	textIds = Iterables.filter(textIds, Predicates.containsPattern(query));
        }
        
        ResultsPage<String> page = new ResultsPage<String>(Lists.newArrayList(Iterables.limit(Iterables.skip(textIds, start), length)),
                Iterables.size(textIds));
        
        return page;
	}

	public synchronized ResultsPage<CorpusTextBlock> getCorpusText(String id, int start,
			int length) throws CorpusApiException {
		TEICorpus corpus = this.getCorpus();
		if (corpus == null) {
			throw new CorpusApiException("Could not open corpus.");
		}
		
		ArrayList<CorpusTextBlock> textBlockList = Lists.newArrayList();
		
		int i = 0;
		try {
			TEICorpusText corpusText = corpus.getCorpusText(id);
			TEITextBlock textBlock = null;
			
			while ((textBlock = corpusText.getNextTextBlock()) != null) {
				if ((i >= start) && (i - start < length)) {
					CorpusTextBlock corpusTextBlock = createCorpusTextBlock(textBlock);
					textBlockList.add(corpusTextBlock);
				}
				i++;
			}
			corpusText.closeCorpusText();
		} catch (Exception e) {
			throw new CorpusApiException("Exception while reading XML.", e);
		}
		
		return new ResultsPage<CorpusTextBlock>(textBlockList, i);
	}

    public synchronized TextBlockSegments getCorpusTextBlock(String textId, String blockId, int start,
            int length) throws CorpusApiException {
        TEICorpus corpus = this.getCorpus();
        if (corpus == null) {
            throw new CorpusApiException("Could not open corpus.");
        }
        
        String prevBlockId = null;
        String nextBlockId = null;
        ArrayList<CorpusSegment> segmentList = Lists.newArrayList();
        try {
            TEICorpusText corpusText = corpus.getCorpusText(textId);
            TEITextBlock textBlock = null;
            while ((textBlock = corpusText.getNextTextBlock()) != null) {
                if (textBlock.getId().equals(blockId)) {
                    break;
                }
                prevBlockId = textBlock.getId();
            }
            if (textBlock == null) {
                return null;
            }
            for (int i = 0; i < textBlock.getSize(); i++) {
                TEISegment teiSegment = textBlock.getSegment(i);
                if (teiSegment == null) {
                    break;
                }
                CorpusSegment corpusSegment = createCorpusSegment(textId, blockId, teiSegment);
                segmentList.add(corpusSegment);
            }
            
            textBlock = corpusText.getNextTextBlock();
            if (textBlock != null) {
            	nextBlockId = textBlock.getId();
            }
            corpusText.closeCorpusText();
        } catch (Exception e) {
            throw new CorpusApiException("Exception while reading XML.", e);
        }
        return new TextBlockSegments(new ResultsPage<CorpusSegment>(segmentList, segmentList.size()), prevBlockId, nextBlockId);
    }

    public synchronized CorpusSegment getCorpusSegment(String textId, String blockId, String id)
            throws CorpusApiException {
        TEICorpus corpus = this.getCorpus();
        if (corpus == null) {
            throw new CorpusApiException("Could not open corpus.");
        }
        CorpusSegment corpusSegment = null;
        try {
            TEICorpusText corpusText = corpus.getCorpusText(textId);
            TEITextBlock textBlock = null;
            while ((textBlock = corpusText.getNextTextBlock()) != null) {
                if (textBlock.getId().equals(blockId))
                    break;
            }
            if (textBlock == null) {
                return null;
            }
            int i = 0;
            TEISegment teiSegment = null;
            while (i < textBlock.getSize()) {
                teiSegment = textBlock.getSegment(i);
                if (teiSegment == null) {
                    return null;
                }
                if ((teiSegment.getId() != null) && (teiSegment.getId().equals(id))) {
                    break;
                }
                i++;
            }
            corpusSegment = createCorpusSegment(textId, blockId, teiSegment);
            corpusText.closeCorpusText();
        } catch (Exception e) {
            throw new CorpusApiException("Exception while reading XML.", e);
        }
        return corpusSegment;
    }
}
