package pl.waw.ipipan.corpcor.server.corpusapi.tei.readers;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;

import org.codehaus.stax2.XMLStreamReader2;

import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEICorpusText;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEIMorphLex;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEIMorphoSegmentGroup;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.TEISegmentGroup;


/**
 * @author Mateusz Kopeć
 */
public class TEIMorphoReader extends TEIReader {
	
	/**
	 * Cache'owanie grup następuje względem id segmentów. Może być wyłączone lub
	 * ograniczone z góry przez zadaną liczbę grup.
	 */
	private HashMap<String, TEIMorphoSegmentGroup> morphoCacheSegmentId;
	
	public TEIMorphoReader(String fileName, TEICorpusText corpusText) {
		super(fileName, corpusText);
		if (cached) {
			if (cacheMaxItems == null) {
				morphoCacheSegmentId = new LinkedHashMap<String, TEIMorphoSegmentGroup>();
			} else {
				morphoCacheSegmentId = new LinkedHashMap<String, TEIMorphoSegmentGroup>() { 
					private static final long serialVersionUID = -6839392540762832442L;
					protected boolean removeEldestEntry(Map.Entry<String, TEIMorphoSegmentGroup> eldest) {
						return size() > cacheMaxItems;
					}
			    };
			}			
		}
	}
	
	/**
	 * Probuje zwolnic wykorzystywane zasoby poprzez zwolnienie pamieci
	 * zarezerwowanej na wczytany plik.
	 * 
	 * @throws XMLStreamException
	 * @throws IOException 
	 * 
	 */
	public void close() throws XMLStreamException, IOException {	
		currentSegmentId2 = null;
		lastSegmentGroup1  = null;
		lastSegmentGroup2  = null;
		if (streamReader1 != null) {
			streamReader1.close();
			streamReader1 = null;
		}
		if (streamReader2 != null) {
			streamReader2.close();
			streamReader2 = null;
		}
		if (morphoCacheSegmentId != null) {
			morphoCacheSegmentId.clear();
			morphoCacheSegmentId = null;
		}
		super.close();
	}
	
	/** Metoda fabrykująca grupę.
	 * @param id
	 * @param corpusText
	 * @param segmentIdList
	 * @param nextGroupId
	 * @param prevGroupId
	 * @return
	 */
	public TEIMorphoSegmentGroup createMorpho(String id, TEICorpusText corpusText, List<String> segmentIdList, String nextGroupId, String prevGroupId) {		
		if (cached) {
			TEIMorphoSegmentGroup morpho = null;
			if ((morpho = morphoCacheSegmentId.get(id)) == null) {
				morpho = new TEIMorphoSegmentGroup(id, corpusText, segmentIdList, nextGroupId, prevGroupId);						
				morphoCacheSegmentId.put(segmentIdList.get(0), morpho);
			}
			return morpho;
		} else {
			return new TEIMorphoSegmentGroup(id, corpusText, segmentIdList, nextGroupId, prevGroupId);
		}
	}
	
	/** Pobiera pierwsze MorphoSegmentGroup z pliku xml.
	 * @return grupa lub null, gdy nie ma
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEIMorphoSegmentGroup getFirstMorphoSegmentGroup() throws XMLStreamException, IOException {
		XMLStreamReader2 sr = createStreamReader();

		List<String> segmentIdList = new ArrayList<String>();
		String segmentId = null;

		TEIMorphoSegmentGroup result = null;

		while (sr.hasNext()) {
			int event = sr.next();					
			if (event == XMLStreamReader2.START_ELEMENT) {

				QName name = sr.getName();
				String local = name.getLocalPart();

				if ("fs".equals(local)) {
					getMorphoInfo(sr, result);
				} else if ("seg".equals(local)) {
					String corresp = sr.getAttributeValue(null, "corresp");
					String id = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
					
					if (result == null) {
						segmentId = corresp.split("#")[1];
						segmentIdList.add(segmentId);
						result = createMorpho(id, corpusText, segmentIdList, null, null);
					} else {
						result.nextGroupId = id;
						break;
					}
				}
			}
		}		
		sr.close();
		return result;
	}
	
	/* zmienne zachowujące stan streamReader1 */
	protected TEISegmentGroup lastSegmentGroup1 = null;
	protected XMLStreamReader2 streamReader1;
	
	protected boolean setStreamReaderForGroupId(String groupId) throws XMLStreamException, IOException {
		if (lastSegmentGroup1 == null || !groupId.equals(lastSegmentGroup1.nextGroupId)) {
			if (streamReader1 != null)
				streamReader1.close();
			streamReader1 = createStreamReader();
			return false;
		}
		
		return true;
	}

	/** Pobiera MorphoSegmentGroup z pliku xml, używając do jej znalezienia id tej grupy.
	 * @param morphoId
	 * @return MorphoSegmentGroup lub null, gdy nie ma
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEIMorphoSegmentGroup getMorphoSegmentGroupById(String morphoId) throws XMLStreamException, IOException {
		boolean oldStream = setStreamReaderForGroupId(morphoId);

		List<String> segmentIdList = new ArrayList<String>();
		String segmentId = null;

		TEIMorphoSegmentGroup result = null;
		String prevMorphoId = null;

		while (streamReader1.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = streamReader1.next();	
			}
			
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
					prevMorphoId = lastSegmentGroup1.getId();
				}
				
				QName name = streamReader1.getName();
				String local = name.getLocalPart();

				if ("fs".equals(local)) {
					getMorphoInfo(streamReader1, result);
				} else if ("seg".equals(local)) {
					String corresp = streamReader1.getAttributeValue(null, "corresp");
					String id = streamReader1.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");

					if (id.equals(morphoId)) {
						segmentId = corresp.split("#")[1];
						segmentIdList.add(segmentId);
						result = createMorpho(morphoId, corpusText, segmentIdList, null, prevMorphoId);
					} else {
						if (segmentId == null) {
							prevMorphoId = id;
						} else {
							result.nextGroupId = id;
							break;
						}
						streamReader1.skipElement();
					}
				}
			}
		}

		lastSegmentGroup1 = result;
		return result;
	}

	protected TEISegmentGroup lastSegmentGroup2 = null;
	protected String currentSegmentId2 = null;
	protected XMLStreamReader2 streamReader2;
	protected boolean setStreamReaderForSegmentId(String segmentId) throws XMLStreamException, IOException {
		if (lastSegmentGroup2 == null || !segmentId.equals(currentSegmentId2)) {
			if (streamReader2 != null)
				streamReader2.close();
			streamReader2 = createStreamReader();
			return false;
		}
		
		return true;
	}
	
	/** Pobiera MorphoSegmentGroup z pliku xml, używając do jej znalezienia id segmentu, który ta
	 * grupa opisuje.
	 * @param segmentId
	 * @return MorphoSegmentGroup lub null, gdy nie ma
	 * @throws XMLStreamException
	 * @throws IOException
	 */
	public TEISegmentGroup getMorphoSegmentGroupBySegmentId(String segmentId) throws XMLStreamException, IOException {
		if (cached && morphoCacheSegmentId.containsKey(segmentId)) {
			return morphoCacheSegmentId.get(segmentId);
		}
		
		boolean oldStream = setStreamReaderForSegmentId(segmentId);
		currentSegmentId2 = null;
		
		List<String> segmentIdList = new ArrayList<String>();
		segmentIdList.add(segmentId); // jedyny element

		TEIMorphoSegmentGroup result = null;
		String prevMorphoId = null;

		while (streamReader2.hasNext() || oldStream) {
			int event = 0;
			if (!oldStream) {
				event = streamReader2.next();	
			}			
			if (event == XMLStreamReader2.START_ELEMENT || oldStream) {
				if (oldStream) {
					oldStream = false;
					prevMorphoId = lastSegmentGroup2.getId();
				}				
				
				QName name = streamReader2.getName();
				String local = name.getLocalPart();

				if ("fs".equals(local)) {
					getMorphoInfo(streamReader2, result);
				} else if ("seg".equals(local)) {
					String currentSegmentId = streamReader2.getAttributeValue(null, "corresp").split("#")[1];
					String id = streamReader2.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");

					if (currentSegmentId.equals(segmentId)) {
						result = createMorpho(id, corpusText, segmentIdList, null, prevMorphoId);						
					} else {
						if (result == null) {
							prevMorphoId = id;
						} else {							
							result.nextGroupId = id;
							lastSegmentGroup2 = result;
							currentSegmentId2 = currentSegmentId;
							break;
						}
						streamReader2.skipElement();
					}
				}
			}
		}

		return result;
	}

	/** Stany automatu, oznaczaja, ze jestesmy wewnatrz danego tagu */
	private static enum State {
		OUTSIDE, 
		ORTH, 
		INTERPS, INTERPS_LEX, INTERPS_LEX_BASE, INTERPS_LEX_CTAG, INTERPS_LEX_MSD,
		DISAMB, DISAMB_TOOL_REPORT, DISAMB_TOOL, DISAMB_DATE, DISAMB_RESP
	};

	/** Pobiera z XMLStreamReader2 sr, ustawionego w odpowiednim miejscu, informacje morfosyntaktyczne.
	 * Zapisuje je do result, który powinien już być utworzony wcześniej.
	 * @param sr
	 * @param result
	 * @throws XMLStreamException
	 */
	private void getMorphoInfo(XMLStreamReader2 sr, TEIMorphoSegmentGroup result) throws XMLStreamException {

		String type = sr.getAttributeValue("", "type");
		if (type.equals("morph") && result != null) {
			// jestesmy wewnatrz wlasciwego fs typu morph

			List<TEIMorphLex> interps = new ArrayList<TEIMorphLex>();
			TEIMorphLex interpretation = null;

			State state = State.OUTSIDE;
			loop: while (sr.hasNext()) {
				
				int event = sr.next();
				if (event == XMLStreamReader2.START_ELEMENT) {
					QName name = sr.getName();
					String local = name.getLocalPart();

					switch (state) {

					case ORTH:
						if ("string".equals(local)) {
							result.orth = sr.getElementText();
							state = State.OUTSIDE;
						}
						break;

					case INTERPS:
						if ("fs".equals(local)) {
							String fstype = sr.getAttributeValue("", "type");
							if (fstype.equals("lex")) {
								state = State.INTERPS_LEX;
								String id = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
								interpretation = new TEIMorphLex(id);
							}
						}
						break;

					case INTERPS_LEX:
						if ("f".equals(local)) {
							String fname = sr.getAttributeValue("", "name");
							if (fname.equals("base")) {
								state = State.INTERPS_LEX_BASE;
							} else if (fname.equals("ctag")) {
								state = State.INTERPS_LEX_CTAG;
							} else if (fname.equals("msd")) {
								state = State.INTERPS_LEX_MSD;
							}
						}
						break;

					case INTERPS_LEX_BASE:
						if ("string".equals(local)) {
							interpretation.base = sr.getElementText();
							state = State.INTERPS_LEX;
						}
						break;

					case INTERPS_LEX_CTAG:
						if ("symbol".equals(local)) {
							interpretation.ctag = sr.getAttributeValue("", "value");
							state = State.INTERPS_LEX;
						}
						break;

					case INTERPS_LEX_MSD:
						if ("symbol".equals(local)) {
							String id = sr.getAttributeValue("http://www.w3.org/XML/1998/namespace", "id");
							String value = sr.getAttributeValue("", "value");
							interpretation.msd.put(id, value);
						}
						break;

					case DISAMB:
						if ("fs".equals(local) && sr.getAttributeValue("", "type").equals("tool_report")) {
							state = State.DISAMB_TOOL_REPORT;
						}
						break;

					case DISAMB_TOOL_REPORT:
						if ("f".equals(local)) {
							String fname = sr.getAttributeValue("", "name");
							if (fname.equals("tool")) {
								state = State.DISAMB_TOOL;
							} else if (fname.equals("date")) {
								state = State.DISAMB_DATE;
							} else if (fname.equals("resp")) {
								state = State.DISAMB_RESP;
							} else if (fname.equals("choice")) {
								String choice = sr.getAttributeValue("", "fVal");
								result.disambChoice = choice.substring(1);
							}
						}
						break;

					case DISAMB_TOOL:
						if ("string".equals(local)) {
							result.disambTool = sr.getElementText();
							state = State.DISAMB_TOOL_REPORT;
						}
						break;

					case DISAMB_DATE:
						if ("string".equals(local)) {
							result.disambDate = sr.getElementText();
							state = State.DISAMB_TOOL_REPORT;
						}
						break;

					case DISAMB_RESP:
						if ("string".equals(local)) {
							result.disambResp = sr.getElementText();
							state = State.DISAMB_TOOL_REPORT;
						}
						break;

					case OUTSIDE:
						if ("f".equals(local)) {
							String fname = sr.getAttributeValue("", "name");
							if (fname.equals("orth")) {
								state = State.ORTH;
							} else if (fname.equals("interps")) {
								state = State.INTERPS;
							} else if (fname.equals("disamb")) {
								state = State.DISAMB;
							}
						}
						break;

					default:
						break;
					}

				} else if (event == XMLStreamReader2.END_ELEMENT) {
					QName name = sr.getName();
					String local = name.getLocalPart();

					switch (state) {

					case OUTSIDE:
						if (local.equals("fs")) {
							break loop; // koniec morpho
						}
						break;

					case INTERPS:
						if (local.equals("f")) {
							state = State.OUTSIDE; // koniec wszystkich interpretacji
						}
						break;

					case INTERPS_LEX:
						if (local.equals("fs")) {
							interps.add(interpretation);
							state = State.INTERPS; // koniec danej interpretacji
						}
						break;

					case INTERPS_LEX_MSD:
						if (local.equals("f")) {
							state = State.INTERPS_LEX; // koniec msd danej interpretacji
						}
						break;

					case DISAMB:
						if (local.equals("f")) {
							state = State.OUTSIDE; // koniec disamb
						}
						break;

					case DISAMB_TOOL_REPORT:
						if (local.equals("fs")) {
							state = State.DISAMB; // koniec disamb tool
						}
						break;

					default:
						break;
					}
				}
			}

			result.setInterps(interps);		
		}
	}
}
