/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.brokers;

import ipipan.poliqarp.connection.PoliqarpConnection;
import ipipan.poliqarp.logic.Corpus;

import java.util.concurrent.Executor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.brokers.CorpusListener.Transition;

public class CorpusBroker extends AbstractBroker {

    private static final Logger LOG = LoggerFactory.getLogger(CorpusBroker.class);

    private static class TransitionRule {
        Corpus.State previous, next;
        CorpusListener.Transition target;

        public TransitionRule(Corpus.State previous, Corpus.State next, Transition target) {
            this.previous = previous;
            this.next = next;
            this.target = target;
        }
    }

    private static class TransitionStrategy {
        private TransitionRule[] transitionTable = new TransitionRule[] {
                new TransitionRule(null, Corpus.State.OPENED, CorpusListener.Transition.OPENED),
                new TransitionRule(null, Corpus.State.NOT_OPENED, CorpusListener.Transition.OPEN_FAILED),
                new TransitionRule(Corpus.State.OPENING, Corpus.State.OPENED, CorpusListener.Transition.OPENED),
                new TransitionRule(Corpus.State.OPENING, Corpus.State.NOT_OPENED, CorpusListener.Transition.OPEN_FAILED),
                new TransitionRule(Corpus.State.OPENED, Corpus.State.NOT_OPENED, CorpusListener.Transition.CLOSED),
                new TransitionRule(Corpus.State.NOT_OPENED, Corpus.State.OPENED, CorpusListener.Transition.OPENED),
                new TransitionRule(Corpus.State.OPENING, null, CorpusListener.Transition.OPEN_FAILED),
                new TransitionRule(Corpus.State.OPENED, null, CorpusListener.Transition.CLOSED) };

        CorpusListener.Transition getTransition(Corpus.State prevState, Corpus.State newState) {
            for (TransitionRule tr : transitionTable) {
                if (tr.previous == prevState && tr.next == newState)
                    return tr.target;
            }
            return CorpusListener.Transition.UNKNOWN;
        }
    }

    private final TransitionStrategy transitionStrategy = new TransitionStrategy();
    private final CorpusListener listener;

    private Corpus corpus;
    private Corpus.State previousState;

    public CorpusBroker(Executor executor, CorpusListener listener) {
        super(executor);
        this.listener = listener;
    }

    public void requestCorpusStart(final PoliqarpConnection connection, final String corpusImage) {
        Runnable startCommand = new Runnable() {

            @Override
            public void run() {
                try {
                    LOG.trace("Opening corpus on: " + corpusImage + " conn:" + connection);
                    corpus = new Corpus(connection, corpusImage, CorpusBroker.this);
                } catch (Exception e) {
                    LOG.warn("Corpus failed on open: " + corpusImage + " conn:" + connection, e);
                    Transition.OPEN_FAILED.invoke(listener, null);
                }
            }
        };
        this.executor.execute(startCommand);
    }

    public void requestCorpusClose() {
        this.executor.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    corpus.close();
                } catch (Exception e) {
                    LOG.warn("Corpus close failed", e);
                } finally {
                    Transition.CLOSED.invoke(listener, null);
                }
            }
        });
    }

    private synchronized CorpusListener.Transition getNewStateTransition(Corpus.State newState) {
        Transition transition = transitionStrategy.getTransition(this.previousState, newState);
        this.previousState = newState;
        return transition;
    }

    @Override
    public void publish() {
        Transition transition = getNewStateTransition(corpus.getState());
        transition.invoke(this.listener, corpus);
    }
}
