/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.requests;

import ipipan.poliqarp.gui.Configuration;
import ipipan.poliqarp.logic.Match;
import ipipan.poliqarp.logic.MatchList;
import ipipan.poliqarp.logic.MetaData;
import ipipan.poliqarp.logic.MetaDataValue;
import ipipan.poliqarp.logic.Query;
import ipipan.poliqarp.logic.QueryException;
import ipipan.poliqarp.logic.Segment;
import ipipan.poliqarp.logic.WideContext;
import ipipan.poliqarp.logic.results.MatchCell;
import ipipan.poliqarp.stat.StatQueryException;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.api.PQCQueryException;
import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.brokers.QueryBroker;
import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.brokers.QueryListener;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.ContextTicket;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.MetadataTicket;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.PQCRequestTicket;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.QueryTicket;
import pl.waw.ipipan.corpcor.shared.ResultsPage;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpContext;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpMetaData;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpResult;

public class QueryRequest implements Request {

    private static final Logger LOG = LoggerFactory.getLogger(QueryRequest.class);

    private static final class QueryNotifier implements QueryListener {
        private final Lock lock;
        private final Condition cond;
        private boolean fail = false;

        public QueryNotifier(Lock l) {
            this.lock = l;
            this.cond = l.newCondition();
        }

        @Override
        public void queryCompleted(Query query) {
            try {
                lock.lock();
                this.fail = query.isQueryFailed();
                cond.signalAll();
            } finally {
                lock.unlock();
            }
        }

        public boolean await() throws InterruptedException {
            try {
                lock.lock();
                boolean b = this.cond.await(5, TimeUnit.MINUTES);
                return b && !fail;
            } finally {
                lock.unlock();
            }
        }
    }

    private final RequestFactory rf;
    private final Lock notifyLock = new ReentrantLock();
    private Query query;

    public QueryRequest(RequestFactory requestFactory) {
        this.rf = requestFactory;
    }

    @Override
    public boolean canContinue(PQCRequestTicket<?> newTicket) {
        if (newTicket instanceof QueryTicket) {
            QueryTicket qt = (QueryTicket) newTicket;
            return String.valueOf(qt.getQuery()).equals(this.query.toString());
        } else if (newTicket instanceof MetadataTicket) {
            MetadataTicket mt = (MetadataTicket) newTicket;
            return query != null && !query.isQueryFailed() && query.getMatchList().count() > mt.getQueryResultId();
        } else if (newTicket instanceof ContextTicket) {
            ContextTicket ct = (ContextTicket) newTicket;
            return query != null && !query.isQueryFailed() && query.getMatchList().count() > ct.getQueryResultId();
        }
        return false;
    }

    @Override
    public void run(PQCRequestTicket<?> ticket) throws InterruptedException, IOException, PQCQueryException {
        if (ticket instanceof QueryTicket) {
            runQueryTicket((QueryTicket) ticket);
        } else if (ticket instanceof MetadataTicket) {
            runMetadataTicket((MetadataTicket) ticket);
        } else if (ticket instanceof ContextTicket) {
            runContextTicket((ContextTicket) ticket);
        }
    }

    private void runMetadataTicket(MetadataTicket ticket) throws InterruptedException, IOException {
        MetaData metaData = this.query.getMatchList().getMetaData(ticket.getQueryResultId());
        PoliqarpMetaData result = new PoliqarpMetaData();
        Set<String> keySet = metaData.keySet();
        for (String key : keySet) {
            List<MetaDataValue> list = metaData.get(key);
            List<String> convertedList = new LinkedList<String>();
            for (MetaDataValue metaDataValue : list) {
                String string = metaDataValue.toString(rf.getLocale());
                if (string != null && !string.isEmpty())
                    convertedList.add(string);
            }
            result.setMetadataValues(key, convertedList.toArray(new String[convertedList.size()]));
        }
        ticket.setResult(result);
    }

    private void runContextTicket(ContextTicket ticket) throws InterruptedException, IOException {
        WideContext wideContext = this.query.getMatchList().getWideContext(ticket.getQueryResultId());
        PoliqarpContext result = new PoliqarpContext(wideContext.toString());
        ticket.setResult(result);
    }

    private void runQueryTicket(QueryTicket ticket) throws InterruptedException, IOException, PQCQueryException {
        ResultsPage<PoliqarpResult> rp = null;
        try {
            notifyLock.lock();
            if (query == null) {
                QueryNotifier notifier = new QueryNotifier(notifyLock);
                try {
                    LOG.info("Launching a new query " + ticket.getQuery());
                    query = new Query(rf.getConnection(), ticket.getQuery());
                    query.getMatchList().createCounter(null, rf.getTagset());
                    query.launch(new QueryBroker(rf.getExecutor(), query, notifier));
                } catch (QueryException e) {
                    LOG.warn("Query " + ticket.getQuery() + " failed to create");
                    query = null;
                    throw new PQCQueryException(e, ticket.getQuery());
                } catch (StatQueryException e) {
                    LOG.warn("Query " + ticket.getQuery() + " failed to run", e);
                    query = null;
                    return;
                }
                if (!notifier.await()) {
                    LOG.warn("Query " + ticket.getQuery() + " timed out ");
                    try {
                        query.abort();
                    } catch (QueryException e) {
                        LOG.warn("Query " + ticket.getQuery() + " failed to abort", e);
                    }
                    query = null;
                    throw new IOException("Query failed to complete due to timeout or connection break");
                }
            }
            LOG.debug("Getting match list for query " + ticket.getQuery());
            MatchList matchList = query.getMatchList();
            int globalCount = matchList.count();
            Configuration config = rf.getConfig();
            ArrayList<PoliqarpResult> entries = new ArrayList<PoliqarpResult>();
            int count = matchList.count();
            int end = (ticket.getLength() == -1) ? count : ticket.getStart() + ticket.getLength();
            if (end > count) {
                LOG.info("Request range (" + ticket.getStart() + "-" + end + ") outside actual range (" + 0 + "-"
                        + count + ")");
                end = count;
            }
            for (int i = ticket.getStart(); i < end; i++) {
                Match match = matchList.getMatch(i);
                String lm = match.getCell(0).toString(config);
                String mm = match.getCell(1).toString(config);
                String rm = match.getCell(2).toString(config);
                MatchCell mc = (MatchCell) match.getCell(1);
                Segment[] segments = mc.getSegments();
                String[] orthSegments = new String[segments.length];
                for (int j = 0; j < segments.length; j++) {
                    orthSegments[j] = segments[j].getOrth();
                }
                PoliqarpResult pr = new PoliqarpResult(i, lm, rm, mm, orthSegments);
                entries.add(pr);
            }
            rp = new ResultsPage<PoliqarpResult>(entries, globalCount);
            LOG.debug("Returning result for query " + ticket.getQuery());
            ticket.setResult(rp);
        } finally {
            notifyLock.unlock();
        }
    }
}
