/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.client.lifecycle;

import java.io.IOException;
import java.net.ConnectException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.api.PQCConfig;
import pl.waw.ipipan.corpcor.server.pq.client.api.PQCQueryException;
import pl.waw.ipipan.corpcor.server.pq.client.api.PQCRequestHandler;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.PQCRequestTicket;
import pl.waw.ipipan.corpcor.server.pq.client.util.Pair;
import pl.waw.ipipan.corpcor.server.pq.common.PQObjectFactory;
import pl.waw.ipipan.corpcor.server.pq.server.lifecycle.PQDConnectionSlot;
import pl.waw.ipipan.corpcor.server.pq.server.lifecycle.PQLMConfig;
import pl.waw.ipipan.corpcor.server.pq.server.lifecycle.PQLMPool;

/**
 * @author Nestor Pawlowski
 */
public class PQCRequestHandlerPool {

    private static final Logger LOG = LoggerFactory.getLogger(PQCRequestHandlerPool.class);

    private final PQLMPool pool;
    private final PQCConfig clientConfig;
    private final PQLMConfig lmConfig;

    private ScheduledExecutorService executor;

    public PQCRequestHandlerPool(PQLMConfig lmConfig, PQLMPool pool) {
        this.lmConfig = lmConfig;
        this.pool = pool;
        try {
            clientConfig = PQObjectFactory.createClientConfig(lmConfig.getPqdPlatform(), lmConfig.getPqdVersion());
        } catch (ClassNotFoundException e) {
            LOG.error("failed to initialize PQBridge", e);
            throw new RuntimeException(e);
        }
        // minimum pool of 2 helps avoid locking when waiting for sync calls
        // executed in the same thread pool
        this.executor = new ScheduledThreadPoolExecutor(4, new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                Thread thread = new Thread(r, "PQCRequestHandlerPool thread");
                thread.setDaemon(true);
                return thread;
            }
        }, new ThreadPoolExecutor.DiscardPolicy());
    }

    private Lock poolLock = new ReentrantLock();
    private Map<String, Pair<PQCRequestHandler, PQDConnectionSlot>> handlers = new HashMap<String, Pair<PQCRequestHandler, PQDConnectionSlot>>();

    private PQCRequestHandler getRequestHandler(String clientId) throws IOException, InterruptedException {
        try {
            poolLock.lock();
            Pair<PQCRequestHandler, PQDConnectionSlot> pair = handlers.get(clientId);
            PQCRequestHandler requestHandler;
            if (pair == null) {
                PQDConnectionSlot connection = pool.newConnection();
                if (connection == null) {
                    LOG.warn("Connection not established for " + clientId);
                    return null;
                }
                PQCRequestHandler tmp = clientConfig.newRequestHandler(this.executor, connection);
                requestHandler = new PQCRequestHandlerWatchdogAdapter(executor, this, clientId, tmp, lmConfig.getSessionTimeout());
                try {
                    requestHandler.start();
                } catch (ConnectException e) {
                    LOG.error("Connection not established for " + clientId + ": server not running, closing its connection pool");
                    pool.destroyPool(connection);
                    throw e;
                } catch (IOException e) {
                    LOG.warn("Connection not established for " + clientId + " due to IO error");
                    pool.removeConnection(connection);
                    throw e;
                } catch (InterruptedException e) {
                    LOG.warn("Connection not established for " + clientId + " due to interruption", e);
                    try {
                        requestHandler.close();
                    } catch (InterruptedException e1) {
                        LOG.warn("Request handler for " + clientId + " closing interrupted", e);
                    }
                    pool.removeConnection(connection);
                    throw e;
                }
                handlers.put(clientId, new Pair<PQCRequestHandler, PQDConnectionSlot>(requestHandler, connection));
            } else {
                requestHandler = pair.first;
            }
            return requestHandler;
        } finally {
            poolLock.unlock();
        }
    }

    private void discardRequestHandler(String clientId) {
        LOG.info("Closing request handler for " + clientId);
        try {
            poolLock.lock();
            Pair<PQCRequestHandler, PQDConnectionSlot> pair = handlers.remove(clientId);
            if (pair == null) {
                LOG.warn("Request handler for " + clientId + " does not exist, cannot remove");
                return;
            }
            try {
                pair.first.close();
            } catch (InterruptedException e) {
                LOG.warn("Request handler for " + clientId + " closing interrupted", e);
            }
            pool.removeConnection(pair.second);
        } finally {
            poolLock.unlock();
        }
    }

    public void request(String clientId, PQCRequestTicket<?> ticket) throws PQRequestException, PQCQueryException {
        int attempts = 1;
        for (;;) {
            LOG.debug("Starting request: " + ticket + " for client " + clientId + ", attempt " + attempts++);
            try {
                PQCRequestHandler handler = getRequestHandler(clientId);
                handler.runRequest(ticket);
                break;
            } catch (RuntimeException e) {
                LOG.warn("Cannot realize request " + ticket + ": failed to get/run handler for " + clientId, e);
                discardRequestHandler(clientId);
                throw new PQRequestException(clientId, ticket, e);
            } catch (InterruptedException e) {
                LOG.warn("Result of " + ticket + " not acquired for " + clientId + " due to interruption");
                discardRequestHandler(clientId);
                throw new PQRequestException(clientId, ticket, e);
            } catch (PQCQueryException e) {
                LOG.warn("Query of request " + ticket + " was not accepted by server, client is " + clientId);
                discardRequestHandler(clientId);
                throw e;
            } catch (IOException e) {
                LOG.warn("IO Error when processing request " + ticket + " for " + clientId, e);
                discardRequestHandler(clientId);
                if (attempts >= 5) {
                    throw new PQRequestException(clientId, ticket, e);
                }
                continue;
            }
        }
    }

    public void closeSession(String clientId) {
        discardRequestHandler(clientId);
    }

    public void shutdown() {
        try {
            poolLock.lock();
            Set<String> clientIds = new HashSet<String>(handlers.keySet());
            for (String clientId : clientIds) {
                discardRequestHandler(clientId);
            }
        } finally {
            poolLock.unlock();
        }
        executor.shutdownNow();
    }
}
