/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.common;

import java.io.File;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.concurrent.ScheduledExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.api.PQCConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfigurator;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDDeployment;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDInstaller;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDControl;
import pl.waw.ipipan.corpcor.server.pq.server.lifecycle.PQLMConfig;

/**
 * @author Nestor Pawlowski (nestor.pawlowski@gmail.com)
 */
public class PQObjectFactory {

    private static final Logger LOG = LoggerFactory.getLogger(PQObjectFactory.class);

    private static class _StackFisher extends SecurityManager {
        @Override
        public Class<?>[] getClassContext() {
            return super.getClassContext();
        }
    }

    public static PQCConfig createClientConfig(String platform, String version) throws ClassNotFoundException {
        String s = PQConstants.PQC_PROPERTIES_FILE;
        String[] searchLocations = { PQCConfig.class.getPackage().getName() + "." + s, s };
        ResourceBundle rb = null;
        for (int i = 0; i < searchLocations.length; i++) {
            try {
                rb = ResourceBundle.getBundle(searchLocations[i]);
                break;
            } catch (MissingResourceException e) {
                continue;
            }
        }
        if (rb == null) {
            throw new MissingResourceException("PQC config not found", PQCConfig.class.getName(), searchLocations[0]);
        }
        PQCConfig config = createObject(PQCConfig.class, PQCConfig.INSTALLER_HANDLER_CLASS_NAME, platform, version);
        config.initialize(rb);
        return config;
    }

    public static PQLMConfig createLifecycleConfig() {
        String s = PQConstants.PQLM_PROPERTIES_FILE;
        String[] searchLocations = { PQLMConfig.class.getPackage().getName() + "." + s, s };
        for (int i = 0; i < searchLocations.length; i++) {
            try {
                return new PQLMConfig(searchLocations[i]);
            } catch (Exception e) {
                continue;
            }
        }
        return null;
    }

    public static PQDConfigurator createDaemonConfigurator(String platform, String version)
            throws ClassNotFoundException {
        return createObject(PQDConfigurator.class, PQDConfigurator.INSTALLER_HANDLER_CLASS_NAME, platform, version);
    }

    public static PQDDeployment createDaemonDeployment(String platform, String version, File deploymentDir,
            File executableFile, Runnable cleanup) throws ClassNotFoundException {
        PQDDeployment d = createObject(PQDDeployment.class, PQDDeployment.INSTALLER_HANDLER_CLASS_NAME, platform,
                version);
        d.initialize(deploymentDir, executableFile, cleanup);
        return d;
    }

    public static PQDInstaller createDaemonInstaller(String platform, String version) throws ClassNotFoundException {
        return createObject(PQDInstaller.class, PQDInstaller.INSTALLER_HANDLER_CLASS_NAME, platform, version);
    }

    public static PQDControl createDaemonWatchdog(String platform, String version, ScheduledExecutorService executor,
            PQDConfig dConfig, PQDConfigurator configurator) throws ClassNotFoundException {
        PQDControl watchdog = createObject(PQDControl.class, PQDControl.INSTALLER_HANDLER_CLASS_NAME, platform,
                version);
        watchdog.initialize(executor, dConfig, configurator);
        return watchdog;
    }

    public static PQDConfig createDaemonConfig(String platform, String version) {
        ClassLoader[] classLoaders = getInspectedClassLoaders();
        String v = version.replace(".", "_");
        String s = PQConstants.PQD_PROPERTIES_FILE;
        String[] discriminators = new String[] { platform + "_" + v + "." + s, "common_" + v + "." + s, s };
        String[] resourceNamesArray = getQualifiedNames(discriminators, PQDConfig.class.getPackage());
        List<String> resourceNames = new LinkedList<String>();
        resourceNames.addAll(Arrays.asList(resourceNamesArray));
        resourceNames.add(s);
        for (ClassLoader classLoader : classLoaders) {
            P: for (String resource : resourceNames) {
                try {
                    ResourceBundle bundle = ResourceBundle.getBundle(resource, Locale.getDefault(), classLoader);
                    return new PQDConfig(bundle);
                } catch (Exception e) {
                    continue P;
                }
            }
        }
        LOG.error("Failed to initialize pqd resource bundle for package postfix " + v);
        return null;
    }

    @SuppressWarnings("unchecked")
    private static <T extends PQObject> T createObject(Class<T> intf, String handlerName, String platform,
            String version) throws ClassNotFoundException {
        LOG.trace("Creating object for handler " + handlerName);
        ClassLoader[] classLoaders = getInspectedClassLoaders();
        String v = version.replace(".", "_");
        String[] discriminators = new String[] { platform + "_" + v, "common_" + v };
        String[] packageNames = getQualifiedNames(discriminators, intf.getPackage());
        for (ClassLoader classLoader : classLoaders) {
            P: for (String pkg : packageNames) {
                try {
                    String className = pkg + "." + handlerName;
                    Class<? extends PQObject> c = (Class<? extends PQObject>) classLoader.loadClass(className);
                    PQObject newInstance = c.newInstance();
                    newInstance.setDiscriminator(platform, version);
                    LOG.debug("Created object of class " + c);
                    return (T) newInstance;
                } catch (ClassNotFoundException e) {
                    continue P;
                } catch (ClassCastException e) {
                    LOG.warn("Failed to initialize object: not " + intf + " interface", e);
                    continue P;
                } catch (InstantiationException e) {
                    LOG.warn("Failed to initialize object: cannot instantiate", e);
                } catch (IllegalAccessException e) {
                    LOG.warn("Failed to initialize object: default constructor not public?", e);
                }
            }
        }
        throw new ClassNotFoundException(intf.getCanonicalName() + ", " + handlerName + " for " + platform  + "/" +  version);
    }

    private static ClassLoader getCallerClassLoader() {
        try {
            return AccessController.doPrivileged(new PrivilegedExceptionAction<ClassLoader>() {
                @Override
                public ClassLoader run() throws Exception {
                    Class<?>[] classContext = new _StackFisher().getClassContext();
                    StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
                    // index at which, and above, there are not more traces of
                    // this class in the stack
                    int idx = -1;
                    // shift to ignore java.lang.Thread.getStackTrace frames
                    int shift = stackTrace.length - classContext.length;
                    for (int i = 1; i < stackTrace.length; i++) {
                        if (stackTrace[i].getClassName().contains(PQObjectFactory.class.getSimpleName()))
                            idx = i + 1 - shift;
                    }
                    return classContext[idx].getClassLoader();
                }
            });
        } catch (PrivilegedActionException e) {
            LOG.error("Failed to initialize object: invoker classloader not accessible", e.getCause());
        }
        return null;
    }

    private static ClassLoader getContextClassLoader() {
        try {
            return AccessController.doPrivileged(new PrivilegedExceptionAction<ClassLoader>() {
                @Override
                public ClassLoader run() throws Exception {
                    return Thread.currentThread().getContextClassLoader();
                }
            });
        } catch (PrivilegedActionException e) {
            LOG.error("Failed to initialize object: context classloader not accessible", e.getCause());
        }
        return null;
    }

    private static ClassLoader[] getInspectedClassLoaders() {
        Set<ClassLoader> clsldrs = new HashSet<ClassLoader>();
        ClassLoader callerClassLoader;
        ClassLoader contextClassLoader;
        try {
            callerClassLoader = getCallerClassLoader();
            if (callerClassLoader != null) {
                clsldrs.add(callerClassLoader);
            }
            contextClassLoader = getContextClassLoader();
            if (contextClassLoader != null) {
                clsldrs.add(contextClassLoader);
            }
        } catch (Exception e) {
            LOG.warn("Failed to initialize object: failed get classloaders", e);
        }
        clsldrs.add(PQObjectFactory.class.getClassLoader());
        return clsldrs.toArray(new ClassLoader[clsldrs.size()]);
    }

    private static String[] getQualifiedNames(String[] postfixes, Package defaultPkg) {
        ArrayList<String> pkgNames = new ArrayList<String>();
        String pkgs = System.getProperty(PQConstants.PQOBJECT_HANDLERS_PACKAGES_PROPERTY);
        if (pkgs != null) {
            StringTokenizer st = new StringTokenizer(pkgs, ",");
            while (st.hasMoreTokens()) {
                String token = st.nextToken().trim();
                for (int i = 0; i < postfixes.length; i++) {
                    pkgNames.add(token + "." + postfixes[i]);
                }
            }
        }
        for (int i = 0; i < postfixes.length; i++) {
            pkgNames.add(defaultPkg.getName() + "." + postfixes[i]);
        }
        return pkgNames.toArray(new String[pkgNames.size()]);
    }
}