/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URL;
import java.security.AccessController;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivilegedAction;
import java.util.Random;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PQUtil {

    private static final Logger LOG = LoggerFactory.getLogger(PQUtil.class);

    private PQUtil() {
    }

    public static final Random random = new Random();

    public static String createTemporaryName(String prefix, String suffix) {
        long l;
        do {
            l = random.nextLong();
        } while (l == Long.MIN_VALUE);
        return prefix + String.valueOf(Math.abs(l)) + suffix;
    }

    public static File deployResource(URL resource, File deployDir, String fileName) throws IOException {
        LOG.debug("Deploying resource " + fileName + " to " + deployDir);
        File f = new File(deployDir, fileName);
        byte[] buf = new byte[512];
        if (f.exists()) {
            try {
                LOG.debug("Resource exists, check if contents are identical");
                InputStream is1 = resource.openStream();
                byte[] d1 = getISDigest(is1);
                is1.close();
                InputStream is2 = new FileInputStream(f);
                byte[] d2 = getISDigest(is2);
                is2.close();
                if (d1.length == d2.length) {
                    for (int i = 0; i < d2.length; i++) {
                        if (d1[i] != d2[i])
                            break;
                        if (i == d2.length - 1) {
                            LOG.trace("Skip deployment, already exists: " + f);
                            // identical hashes
                            return f;
                        }
                    }
                }
            } catch (NoSuchAlgorithmException e) {
                LOG.warn("Failed to get digest algorithm", e);
            }
            LOG.debug("Contents are not identical");
        }
        InputStream res = resource.openStream();
        OutputStream os = new FileOutputStream(f);
        int read;
        while ((read = res.read(buf)) >= 0) {
            os.write(buf, 0, read);
        }
        os.close();
        res.close();
        return f;
    }

    private static byte[] getISDigest(InputStream is1) throws NoSuchAlgorithmException, IOException {
        byte[] buf = new byte[512];
        MessageDigest md1 = MessageDigest.getInstance("MD5");
        int read;
        while ((read = is1.read(buf)) >= 0) {
            md1.update(buf, 0, read);
        }
        return md1.digest();
    }

    public static String getDefaultTempPath() {
        return AccessController.doPrivileged(new PrivilegedAction<String>() {
            @Override
            public String run() {
                return System.getProperty("java.io.tmpdir");
            }
        });
    }

}
