/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.daemon.common_1_3_12;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SocketChannel;
import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.util.PQDUtil;
import pl.waw.ipipan.corpcor.server.pq.common.AbstractPQObject;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfigurator;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDOption;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDControl;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.common_1_3_12.PQDResponseParser.EResponseToken;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.common_1_3_12.PQDResponseParser.Response;

public class Control extends AbstractPQObject implements PQDControl {

    static class InterruptCommand implements Runnable {
        private volatile Thread t;

        public InterruptCommand(Thread t) {
            this.t = t;
        }

        @Override
        public void run() {
            if (t != null) {
                LOG.debug("Interrupting " + t);
                t.interrupt();
            }
        }

        void cancel() {
            t = null;
        }
    }

    private static final Logger LOG = LoggerFactory.getLogger(Control.class);
    private ScheduledExecutorService executor;
    private PQDConfig dConfig;
    private PQDConfigurator configurator;

    abstract class MessageExchange<T> implements Callable<T> {
        @Override
        public final T call() throws IOException {
            SocketChannel socket = null;
            InterruptCommand command = null;
            ScheduledFuture<?> scheduledCmd = null;
            LOG.debug("Running control command " + this.toString());
            try {
                socket = newSocket();
                command = new InterruptCommand(Thread.currentThread());
                LOG.debug("Scheduling control command interrupt in " + Constants.CONST_PING_TIMEOUT);
                scheduledCmd = executor.schedule(command, Constants.CONST_PING_TIMEOUT, TimeUnit.MILLISECONDS);
                T res = perform(socket);
                return res;
            } finally {
                if (scheduledCmd != null)
                    scheduledCmd.cancel(false);
                if (command != null)
                    command.cancel();
                // clear interrupt flag in case of too late canceled watchdog
                Thread.interrupted();
                try {
                    if (socket != null) {
                        socket.close();
                        socket.socket().close();
                    }
                } catch (IOException e) {
                    LOG.debug("Error when closing SocketChannel", e);
                }
            }
        }

        public String toString() {
            return this.getClass().getSimpleName() + " to " + getHost() + ":" + getPort();
        };

        public abstract T perform(SocketChannel socket) throws IOException;
    }

    class PingCommand extends MessageExchange<Boolean> {
        @Override
        public Boolean perform(SocketChannel socket) throws IOException {
            send(socket, Constants.CMD_PING);
            String line = receive(socket);
            Response response = PQDResponseParser.getInstance().parseResponse(line);
            LOG.debug("Ping response is " + response.describe());
            return response != null && response.getResponse() == EResponseToken.PONG;
        }
    }

    class HaltCommand extends MessageExchange<Void> {
        @Override
        public Void perform(SocketChannel socket) throws IOException {
            LOG.debug("Preparing for halt, creating PQ session");
            startSession(socket);
            send(socket, Constants.CMD_HALT);
            endSession(socket);
            LOG.debug("Halt sent, PQ session ended");
            return null;
        }
    }

    public Control() {
        // keep not args
    }

    @Override
    public void initialize(ScheduledExecutorService executor, PQDConfig dConfig, PQDConfigurator configurator) {
        LOG.debug("Control initialized " + getInitializedPlatform() + "/" + getInitializedVersion());
        this.executor = executor;
        this.dConfig = dConfig;
        this.configurator = configurator;
    }

    @Override
    public boolean disconnectionCheck() {
        LOG.debug("Checking if PQD still responds to connections");
        String host = getHost();
        int port = getPort();
        for (int i = 0; i < 10; i++) {
            if (!PQDUtil.tryConnect(host, port, 250, 1))
                return true;
            try {
                Thread.sleep(250);
            } catch (InterruptedException e) {
                return false;
            }
        }
        return false;
    }
    
    @Override
    public boolean connectionCheck() {
        LOG.debug("Checking if PQD already esponds to connections");
        String host = getHost();
        int port = getPort();
        return PQDUtil.tryConnect(host, port, 500, 10);
    }

    @Override
    public boolean pingCommand() {
        LOG.debug("Ping requested to " + getHost() + ":" + getPort());
        Boolean res;
        try {
            res = new PingCommand().call();
        } catch (IOException e) {
            LOG.trace("Ping failed", e);
            return false;
        }
        return Boolean.TRUE.equals(res);
    }

    @Override
    public void haltCommand() {
        LOG.debug("Halt requested to " + getHost() + ":" + getPort());
        try {
            new HaltCommand().call();
        } catch (IOException e) {
            LOG.debug("Halt action was not completed: " + e.getLocalizedMessage());
        }
    }

    @Override
    public String getCompatibleVersion() {
        return Constants.PQ_VERSION;
    }

    private static void send(SocketChannel sc, String s) throws IOException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        OutputStreamWriter osw = new OutputStreamWriter(baos, "UTF-8");
        osw.write(s);
        osw.write('\n');
        osw.close();
        ByteBuffer buf = ByteBuffer.wrap(baos.toByteArray());
        sc.write(buf);
    }

    private void startSession(SocketChannel socket) throws IOException {
        String session = "watchdog-" + this.getPort();
        send(socket, Constants.CMD_MAKESESSION + " " + session);
        String receive = receive(socket);
        Response response = PQDResponseParser.getInstance().parseResponse(receive);
        if (response == null)
            throw new IOException("Received invalid response " + receive);
        if (response.getResponse() != EResponseToken.OK)
            throw new IOException("Received wrong message " + response.describe());
        if (response.getArguments() == null || response.getArguments().length != 1)
            throw new IOException("Did not receive session id " + response.describe());
    }

    private void endSession(SocketChannel socket) throws IOException {
        send(socket, Constants.CMD_CLOSESESSION);
        Response response = PQDResponseParser.getInstance().parseResponse(receive(socket));
        if (response.getResponse() != EResponseToken.OK)
            throw new IOException("Received wrong message " + response.describe());
    }

    private String receive(SocketChannel socket) throws IOException {
        ByteBuffer buf = ByteBuffer.allocate(256);
        StringBuilder sb = new StringBuilder();
        while (true) {
            int read = socket.read(buf);
            if (read == -1)
                break;
            String s = new String(buf.array(), "UTF-8");
            int eol = s.indexOf('\n');
            if (eol >= 0) {
                sb.append(s.substring(0, eol));
                break;
            } else {
                sb.append(s);
                buf.rewind();
            }
        }
        sb.append('\n');
        return sb.toString();
    }

    private SocketChannel newSocket() throws IOException {
        return SocketChannel.open(new InetSocketAddress(getHost(), getPort()));
    }

    private int getPort() {
        return Integer.parseInt(configurator.getPQDConfigOption(PQDOption.PORT, dConfig));
    }

    private String getHost() {
        return configurator.getPQDConfigOption(PQDOption.HOSTNAME, dConfig);
    }
}
