/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.daemon.common_1_3_12;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import pl.waw.ipipan.corpcor.server.pq.common.AbstractPQObject;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDDeployment;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDRuntimeConfig;

public class Deployment extends AbstractPQObject implements PQDDeployment {

    static class NullOutputStream extends OutputStream {
        @Override
        public void write(int b) throws IOException {
        }
    }

    static class RedirectionBroker extends Thread {

        static class Redirection {
            private final InputStream is;
            private final OutputStream os;
            private final Process p;

            public Redirection(Process p, InputStream is, OutputStream os) {
                this.p = p;
                this.is = is;
                this.os = os;
            }
        }

        private final List<Redirection> redirections = new CopyOnWriteArrayList<Redirection>();

        RedirectionBroker() {
            super("Process IO Broker");
            setDaemon(true);
            start();
        }

        @Override
        public void run() {
            while (true) {
                for (Redirection redirection : redirections) {
                    if (!transferIfValid(redirection)) {
                        redirections.remove(redirection);
                    }
                }
                try {
                    // unfortunately necessary - selection doesn't for for
                    // process pipes in java 6
                    // and we don't want a thread per process
                    Thread.sleep(250);
                } catch (InterruptedException e) {
                }
            }
        }

        private final byte[] buf = new byte[4096];

        private boolean transferIfValid(Redirection r) {
            try {
                if (r.is.available() <= 0) {
                    try {
                        // available of subprocess pipes return 0 even if
                        // process is closed (sic!), so lets try if the process
                        // really runs
                        r.p.exitValue();
                        return false;
                    } catch (IllegalThreadStateException e) {
                    }
                    return true;
                }
                int read = r.is.read(buf);
                if (read < 0) {
                    r.os.close();
                    return false;
                } else if (read == 0) {
                    return true;
                } else {
                    r.os.write(buf, 0, read);
                    return true;
                }
            } catch (IOException e) {
                try {
                    r.is.close();
                    r.os.close();
                } catch (IOException e1) {
                }
            }
            return false;
        }

        void addRedirection(Process p, InputStream is, OutputStream os) {
            redirections.add(new Redirection(p, is, os));
        }
    }

    private static RedirectionBroker instance = null;

    private static synchronized RedirectionBroker getRedirectionBroker() {
        if (instance == null) {
            instance = new RedirectionBroker();
        }
        return instance;
    }

    private Runnable cleanup;
    private File deploymentDir;
    private File executableFile;

    public Deployment() {
        // keep no args
    }

    @Override
    public void cleanup() {
        if (this.cleanup != null)
            this.cleanup.run();
    }

    @Override
    public String getCompatibleVersion() {
        return Constants.PQ_VERSION;
    }

    @Override
    public String getDeployedVersion() {
        try {
            ArrayList<String> cmd = new ArrayList<String>();
            ProcessBuilder pb = new ProcessBuilder();
            cmd.add(this.executableFile.getAbsolutePath());
            cmd.add(Constants.ARG_VERSION);
            pb.command(cmd);
            Process p = pb.start();
            BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()));
            String line = br.readLine();
            p.destroy();
            br.close();
            return line;
        } catch (IOException e) {
            return null;
        }
    }

    @Override
    public File getDeploymentDir() {
        return this.deploymentDir;
    }

    @Override
    public void initialize(File deploymentDir, File executableFile, Runnable cleanup) {
        this.deploymentDir = deploymentDir;
        this.executableFile = executableFile;
        this.cleanup = cleanup;
    }

    @Override
    public Process start(PQDRuntimeConfig runtimeConfig) throws IOException {
        ArrayList<String> cmd = new ArrayList<String>();
        ProcessBuilder pb = new ProcessBuilder();
        cmd.add(this.executableFile.getAbsolutePath());
        pb.directory(runtimeConfig.getWorkingDir());
        cmd.add(Constants.ARG_CONFIG_FILE + "=" + runtimeConfig.getConfigurationFile().getAbsolutePath());
        pb.command(cmd);
        boolean stderrMerge = false;
        if (runtimeConfig.getStdoutLogFilePath() != null
                && runtimeConfig.getStdoutLogFilePath().equals(runtimeConfig.getStderrLogFilePath())) {
            stderrMerge = true;
            pb.redirectErrorStream(true);
        }

        Process process = pb.start();
        
        // configure stdout redirection
        OutputStream os;
        if (runtimeConfig.getStdoutLogFilePath() != null) {
            os = new FileOutputStream(runtimeConfig.getStdoutLogFilePath());
        } else {
            os = new NullOutputStream();
        }
        getRedirectionBroker().addRedirection(process, process.getInputStream(), os);

        // configure stderr redirection
        if (!stderrMerge) {
            if (runtimeConfig.getStdoutLogFilePath() != null) {
                os = new FileOutputStream(runtimeConfig.getStderrLogFilePath());
            } else {
                os = new NullOutputStream();
            }
            getRedirectionBroker().addRedirection(process, process.getErrorStream(), os);
        }
        return process;
    }
}
