/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.daemon.common_1_3_12;

import java.util.ArrayList;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.nestor.interpret.parse.ChoiceParser;
import com.nestor.interpret.parse.LineScanner;
import com.nestor.interpret.parse.Match;
import com.nestor.interpret.parse.ParseEnv;
import com.nestor.interpret.parse.ParseException;
import com.nestor.interpret.parse.Parser;
import com.nestor.interpret.parse.Scanner;
import com.nestor.interpret.parse.Skip;
import com.nestor.interpret.parse.WhitespaceSkip;
import com.nestor.interpret.token.RegexToken;
import com.nestor.interpret.token.SimpleToken;
import com.nestor.interpret.token.Token;
import com.nestor.interpret.token.TokenContainer;

/**
 * TODO this class is far from complete. shall be extended in future to cover
 * all PQD response messages using a token parser. (shall replace the PQD access
 * API from GUI client). Request builder is totally optional here.
 * 
 * Ideally a version/platform-agnostic abstraction layer should be created as
 * well.
 * 
 * @author Nestor
 */
public final class PQDResponseParser {

    private static final Logger LOG = LoggerFactory.getLogger(PQDResponseParser.class);

    public static enum EResponseToken {
        R(new SimpleToken("R")), 
        PONG(new SimpleToken("PONG")),
        UNKNOWN(new SimpleToken("UNKNOWN")),
        OK(new SimpleToken("OK"));

        private final Token t;

        private EResponseToken(Token t) {
            this.t = t;
        }

        Token getToken() {
            return t;
        }
    }

    static class NonSyntaxTokens {
        static final Token TEXT = new RegexToken("Response argument text", "[^\\s]+");
        static final Token EOL = new SimpleToken("End of line", "\n");
    }

    private static TokenContainer getTokens() {
        TokenContainer tc = new TokenContainer();
        EResponseToken[] values = EResponseToken.values();
        for (EResponseToken eResponseToken : values) {
            tc.addToken(eResponseToken.getToken());
        }
        tc.addToken(NonSyntaxTokens.TEXT);
        tc.addToken(NonSyntaxTokens.EOL);
        return tc;
    }

    static class ResponseParser implements Parser<Response> {

        private final ChoiceParser<Response> responsesParser;

        public ResponseParser() {
            responsesParser = new ChoiceParser<PQDResponseParser.Response>();
            responsesParser.addParser(new NonArgResponseParser(EResponseToken.PONG));
            responsesParser.addParser(new NonArgResponseParser(EResponseToken.UNKNOWN));
            responsesParser.addParser(new OKResponseParser());
        }

        @Override
        public Response parse(Scanner scanner, ParseEnv parseenv) throws ParseException {
            scanner.required(EResponseToken.R.getToken(), "Response doesnt start with "
                    + EResponseToken.R.getToken().describe());
            return this.responsesParser.parse(scanner, parseenv);
        }

        @Override
        public String describe() {
            return EResponseToken.R.getToken().describe() + " " + this.responsesParser.describe();
        }
    }

    static abstract class ResponseBodyParser implements Parser<Response> {

        private final EResponseToken head;

        ResponseBodyParser(EResponseToken head) {
            this.head = head;
        }

        @Override
        public String describe() {
            return this.head.getToken().describe();
        }

        @Override
        public Response parse(Scanner scanner, ParseEnv env) throws ParseException {
            if (scanner.match(this.head.getToken()) == null)
                return null;
            scanner.advance();
            Response response = new Response(this.head, parseResponseArguments(scanner, env));
            return response;
        }

        abstract String[] parseResponseArguments(Scanner scanner, ParseEnv env) throws ParseException;
    }

    static class NonArgResponseParser extends ResponseBodyParser {

        public NonArgResponseParser(EResponseToken t) {
            super(t);
        }

        @Override
        String[] parseResponseArguments(Scanner scanner, ParseEnv env) throws ParseException {
            return new String[0];
        }
    }

    static class OKResponseParser extends ResponseBodyParser {

        public OKResponseParser() {
            super(EResponseToken.OK);
        }

        @Override
        String[] parseResponseArguments(Scanner scanner, ParseEnv env) throws ParseException {
            ArrayList<String> args = new ArrayList<String>();
            while (scanner.match(NonSyntaxTokens.EOL) == null) {
                args.add(scanner.matchAdvance(NonSyntaxTokens.TEXT));
            }
            return args.toArray(new String[args.size()]);
        }
    }

    public static class Response implements Match {
        private static final long serialVersionUID = 1L;
        private final String[] arguments;
        private final EResponseToken response;

        public Response(EResponseToken response, String... arguments) {
            this.response = response;
            this.arguments = arguments;
        }

        public String[] getArguments() {
            return arguments;
        }

        public EResponseToken getResponse() {
            return response;
        }

        @Override
        public String describe() {
            StringBuilder sb = new StringBuilder("[");
            for (int i = 0; arguments != null && i < arguments.length; i++) {
                sb.append(arguments[i]);
                if (i < arguments.length - 1)
                    sb.append(", ");
            }
            sb.append("]");
            return this.response.getToken().describe();
        }
    }

    private static PQDResponseParser instance = null;
    private final ParseEnv env;
    private final Skip skip;
    private final TokenContainer tc;
    private final ResponseParser parser;

    private PQDResponseParser() {
        env = new ParseEnv() {
            @Override
            public void warning(String arg0) {
                LOG.trace(arg0);
            }

            @Override
            public void log(String arg0) {
                LOG.debug(arg0);
            }

            @Override
            public void error(String arg0) {
                LOG.warn(arg0);
            }
        };
        skip = new WhitespaceSkip();
        tc = getTokens();
        parser = new ResponseParser();
    }

    public static synchronized PQDResponseParser getInstance() {
        if (instance == null)
            instance = new PQDResponseParser();
        return instance;
    }

    public synchronized Response parseResponse(String responseText) {
        LOG.debug("Parsing response " + responseText);
        LineScanner scanner = new LineScanner(env, skip, tc, responseText);
        try {
            return parser.parse(scanner, env);
        } catch (ParseException e) {
            LOG.warn("Failed to parse ", e);
            return null;
        }
    }

    public static void main(String[] args) {
        System.out.println(getInstance().parseResponse("R PONG").describe());
    }
}
