/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.lifecycle;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.common.PQObjectFactory;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfigurator;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDDeployment;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDInstaller;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDOption;

public class PQLMStarter {

    private static final Logger LOG = LoggerFactory.getLogger(PQLMStarter.class);

    private final ScheduledExecutorService executor;
    private final PQLMConfig lmConfig;

    private PQDConfigurator configurator;
    private PQDInstaller installer;
    private PQDDeployment deployment;

    public PQLMStarter(PQLMConfig lmConfig, ScheduledExecutorService executor) throws Exception {
        this.lmConfig = lmConfig;
        this.executor = executor;
        try {
            initialize();
        } catch (ClassNotFoundException e) {
            LOG.error("Cannon initialize service component", e);
            throw new Exception(e);
        } catch (IOException e) {
            LOG.error("IOException when initializing service component", e);
            throw new Exception(e);
        }
    }

    private void initialize() throws ClassNotFoundException, IOException {
        LOG.debug("Initializing PQD starter");
        configurator = PQObjectFactory.createDaemonConfigurator(lmConfig.getPqdPlatform(), lmConfig.getPqdVersion());
        installer = PQObjectFactory.createDaemonInstaller(lmConfig.getPqdPlatform(), lmConfig.getPqdVersion());
        deployment = installer.deploy(lmConfig);
        // safety checks
        List<String> versions = new ArrayList<String>();
        versions.add(configurator.getCompatibleVersion());
        versions.add(installer.getCompatibleVersion());
        versions.add(deployment.getCompatibleVersion());
        versions.add(deployment.getDeployedVersion());
        for (int i = 1; i < versions.size(); i++) {
            String v1 = versions.get(i - 1);
            if (v1 == null) {
                LOG.error("One of PQD starter components has no version information");
                return;
            }
            String v2 = versions.get(i);
            if (!v1.equals(v2)) {
                LOG.error("PQD starter component versions are not compatible: " + v1 + " and " + v2);
                return;
            }
        }
    }

    public synchronized PQDConfig newConfig() {
        return PQObjectFactory.createDaemonConfig(lmConfig.getPqdPlatform(), lmConfig.getPqdVersion());
    }

    public String getPQDConfigOption(PQDOption option, PQDConfig dConfig) {
        return this.configurator.getPQDConfigOption(option, dConfig);
    }

    public boolean setPQDConfigOption(PQDOption option, String value, PQDConfig dConfig) {
        return this.configurator.setPQDConfigOption(option, value, dConfig);
    }

    public synchronized PQLMMonitor newMonitor(PQDConfig dConfig) {
        return new PQLMMonitor(deployment, lmConfig, dConfig, configurator, executor);
    }

    public void cleanup() {
        this.deployment.cleanup();
    }
}
