/*
 * This file is part of the Poliqarp suite.
 * 
 * Copyright (C) 2004-2009 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact
 * Michal.Ciesiolka@ipipan.waw.pl or ipi@ipipan.waw.pl for more
 * information).  Licensees holding a valid commercial license from IPI
 * PAN may use this file in accordance with that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */

package ipipan.poliqarp.logic;

import java.io.IOException;
import java.util.Locale;

import ipipan.poliqarp.connection.Message;
import ipipan.poliqarp.connection.PoliqarpConnection;

/**
 * The abstract subclass for all kinds of metadata values.
 */
public abstract class MetaDataValue
{
   /**
    * An abstract constructor that creates a metadata value from
    * its string description.
    * @param description description of the value
    */
   public MetaDataValue(String description)
   {
   }
   
   /**
    * Returns the representation of this metadata value as a string.
    * A locale can be helpful when converting several types of data
    * to strings, so this method takes one as a parameter.
    */
   public abstract String toString(Locale locale);

   /**
    * Returns the representation of this metadata value as a string,
    * using the system's current locale.
    */
   public String toString()
   {
      return toString(Locale.getDefault());
   }

   /**
    * A factory method that creates the metadata value, reading its 
    * description from a connection. The value is of appropriate type.
    * @param conn the connection to read the value from
    */
   public static MetaDataValue readFromConnection(PoliqarpConnection conn)
      throws IOException
   {
      Message m = conn.getMessage();
      String mstr = m.toString();
      char type = mstr.charAt(0);
      String descr = null;
      if (type != 'U')
         descr = mstr.substring(2);
      switch (type) {
         case 'T': return new MetaDataValueText(descr);
         case 'D': return new MetaDataValueDate(descr);
         case 'U': return new MetaDataValueUndefined(descr);
      }
      return null;
   }        
}
