/*
 * This file is part of the CorpCorCorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server;

import java.util.ArrayList;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.client.CorpCorService;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusApiException;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusSegment;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpContext;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpMetaData;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpResult;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

public class CorpCorServiceCaches {

    static class GetCorpusSegmentsAction implements Callable<PoliqarpResult> {
        
        private static final Logger LOG = LoggerFactory.getLogger(GetCorpusSegmentsAction.class);

        private final CorpCorService service;
        private final String client;
        private final PoliqarpResult pqr;
        private final PoliqarpMetaData pqMetaData;
        private final PoliqarpContext pqWideContext;

        GetCorpusSegmentsAction(CorpCorService service, String client, PoliqarpResult pqr, PoliqarpMetaData pqMetaData, PoliqarpContext pqWideContext) {
            this.service = service;
            this.client = client;
            this.pqr = pqr;
            this.pqMetaData = pqMetaData;
            this.pqWideContext = pqWideContext;
        }
        
        @Override
        public PoliqarpResult call() {
            // TODO consider this when doing this action concurrently
            // String sessionId = this.client + "@" +Thread.currentThread().getName();
            try {
                LOG.info("Getting Corpus segments for query match " + this.pqr.getMatch());
                // TODO consider this when doing this action concurrently
                // CorpCorServiceUtils.temporarilyOverrideSessionId(sessionId);
                ArrayList<CorpusSegment> segments = service.getCorpusSegments(pqr, pqMetaData, pqWideContext);
                if (segments == null) {
                	LOG.warn("segments is null (from CALL)");
                }
                pqr.updatedFromCorpus(segments);
            } catch (CorpusApiException e) {
                pqr.updateFromCorpusFailed();
                LOG.warn("Update failed", e);
            } finally {
                // TODO consider this when doing this action concurrently
                //CorpCorServiceUtils.temporarilyOverrideSessionId(null);
                //PQBridge.getInstance().forwardSessionClose(sessionId);
            }
            LOG.debug("Corpus segments for query match returned " + pqr.getMatchAnnotated("not loaded", "load failed"));
            return pqr;
        }
    }
    
    static class SessionCache {
        private String query;
        final String sessionId;
        final Cache<Integer, PoliqarpResult> tier2Cache;
        
        public SessionCache(String sessionId) {
            this.sessionId = sessionId;
            this.tier2Cache = CacheBuilder.newBuilder()
                    .maximumSize(50000)
                    .expireAfterAccess(5, TimeUnit.MINUTES)
                    .concurrencyLevel(1).build();
        }

        public String getQuery() {
            return query;
        }

        public void setQuery(String query) {
            this.query = query;
        }
    }
    
    private static LoadingCache<String, SessionCache> tier1Cache;
    
    private static synchronized LoadingCache<String, SessionCache> getPQResultsCache() {
        if (tier1Cache == null) {
            tier1Cache = CacheBuilder.newBuilder()
            .maximumSize(10)
            .expireAfterAccess(15, TimeUnit.MINUTES)
            .concurrencyLevel(1)
            .build(new CacheLoader<String, SessionCache>() {
               @Override
                public SessionCache load(String sessionId) {
                    return new SessionCache(sessionId);
                } 
            });
        }
        return tier1Cache;
    }

    static SessionCache getPQResultsCache(String client) {
        return getPQResultsCache().getUnchecked(client);
    }
}
