package pl.waw.ipipan.corpcor.server;

import java.util.ArrayList;
import java.util.concurrent.ExecutionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import pl.waw.ipipan.corpcor.client.CorpCorService;
import pl.waw.ipipan.corpcor.server.CorpCorServiceCaches.GetCorpusSegmentsAction;
import pl.waw.ipipan.corpcor.server.CorpCorServiceCaches.SessionCache;
import pl.waw.ipipan.corpcor.server.corpusapi.CorpusAPIBridge;
import pl.waw.ipipan.corpcor.server.dao.AnnotationDao;
import pl.waw.ipipan.corpcor.server.model.Annotation;
import pl.waw.ipipan.corpcor.server.pq.PQBridge;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.ContextTicket;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.MetadataTicket;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.QueryTicket;
import pl.waw.ipipan.corpcor.shared.ResultsPage;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusApiException;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusSegment;
import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusTextBlock;
import pl.waw.ipipan.corpcor.shared.corpusapi.TextBlockSegments;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpContext;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpException;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpMetaData;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpResult;
import pl.waw.ipipan.corpcor.shared.pq.PoliqarpResult.PQResultState;

@Service("corpCorService")
public class CorpCorServiceImpl implements CorpCorService {

    private static final Logger LOG = LoggerFactory.getLogger(CorpCorServiceImpl.class);

    @Autowired
    private CorpusAPIBridge corpusApiBridge;

    @Autowired
    private AnnotationDao annotationDao;

    // ========================================================================================
    // == Poliqarp services
    // ========================================================================================

    public ResultsPage<PoliqarpResult> getSinglePoliqarpQueryResult(String query, int resultId) 
            throws PoliqarpException{
        QueryTicket ticket = new QueryTicket(query, resultId, -1);
        PQBridge.getInstance().forwardRequest(CorpCorServiceUtils.getSessionId(), ticket);
        try {
            return ticket.getResult();
        } catch (InterruptedException e) {
            throw new PoliqarpException(e, "Przetwarzanie zostało nieoczekiwanie przerwane (błąd serwera)");
        }
    }
            
    @Override
    public PoliqarpResult getQueryResult(String query, PoliqarpResult.PQResultItemId resultItemStart) 
            throws PoliqarpException {
        try {
            int resultId = resultItemStart == null ? 0 : resultItemStart.id;
            QueryTicket ticket = new QueryTicket(query, resultId, 1);
            PQBridge.getInstance().forwardRequest(CorpCorServiceUtils.getSessionId(), ticket);
            ResultsPage<PoliqarpResult> results = ticket.getResult();
            loadAnnotations(query, results.getEntries().get(0));
            return results.getEntries().get(0);
        } catch (InterruptedException e) {
            throw new PoliqarpException(e, "Przetwarzanie zostało nieoczekiwanie przerwane (błąd serwera)");
        }
    }

    private void loadAnnotations(String query, PoliqarpResult pqr) throws PoliqarpException {
        // invalidate result cache if query changed
        String client = CorpCorServiceUtils.getSessionId();
        SessionCache pqResultsCache = CorpCorServiceCaches.getPQResultsCache(client);
        if (!query.equals(pqResultsCache.getQuery())) {
            pqResultsCache.tier2Cache.invalidateAll();
            pqResultsCache.setQuery(query);
        }
        if (loadAnnotationFromCacheOrCorpus(pqr, true, client, pqResultsCache))
            loadAnnotationFromDB(pqr);
    }

    private boolean loadAnnotationFromCacheOrCorpus(PoliqarpResult pqr, boolean loadFromCorpus, String client, SessionCache pqResultsCache)
            throws PoliqarpException {
        PoliqarpResult pqrCached = pqResultsCache.tier2Cache.getIfPresent(pqr.getEntryId().resultId.id);
        if (pqrCached == null && loadFromCorpus) {
        	LOG.warn("pqrCached is null & loadFromCorpus");
            PoliqarpMetaData pqMetaData = getPoliqarpMetaData(pqr.getEntryId().resultId);
            PoliqarpContext pqWideContext = getPoliqarpWideContext(pqr.getEntryId().resultId);
            GetCorpusSegmentsAction task = new GetCorpusSegmentsAction(this, client, pqr, pqMetaData, pqWideContext);
            try {
                pqrCached = pqResultsCache.tier2Cache.get(pqr.getEntryId().resultId.id, task);
            } catch (ExecutionException e) {
                LOG.warn("Could not get Corpus segments from cache for " + pqr, e);
                pqr.updateFromCorpusFailed();
            }
        }
        if (pqrCached != null) {
        	LOG.warn("pqrCached is not null");
        	if (pqrCached.getMatchSegments() == null) {
        		LOG.warn("pqrCached.getMatchSegments() is null");
        	}
            pqr.updatedFromCorpus(pqrCached.getMatchSegments());
        }
        return pqrCached != null;
    }

    private boolean loadAnnotationFromDB(PoliqarpResult pqr) {
        boolean foundInDB = false;
        if (pqr.getResultState() == PQResultState.CORPUS_UPDATE_OK) {
            LOG.info("Corpus segment found for " + pqr.getMatch() + ", checking DB");
            CorpusSegment[] segments = pqr.getMatchSegments();
            for (int j = 0; j < segments.length; j++) {
                Annotation annotation = this.annotationDao.findLatestByPath(segments[j].getSegmentId(), segments[j].getBlockId(), segments[j].getTextId());
                LOG.info("Result for segment " + segments[j].getSegmentId() + " is " + annotation);
                if (annotation != null) {
                    foundInDB = true;
                    segments[j].setMorpho(annotation.getOrth(), annotation.getBase(), annotation.getMsd(), annotation.getCtag());
                }
            }
        }
        return foundInDB;
    }

    @Override
    public ResultsPage<PoliqarpResult> getQueryResults(String query, PoliqarpResult.PQViewItemId viewItemStart, int length,
            boolean loadFromCorpus, boolean loadModified) throws PoliqarpException {
        try {
            PoliqarpResult.PQViewItemId viewId = viewItemStart == null ? new PoliqarpResult.PQViewItemId(0) : viewItemStart;
            QueryTicket ticket = new QueryTicket(query, 0, -1);
            PQBridge.getInstance().forwardRequest(CorpCorServiceUtils.getSessionId(), ticket);
            ResultsPage<PoliqarpResult> results = ticket.getResult();
            loadAnnotations(query, results, viewId, length, loadFromCorpus, loadModified);
            return results;
        } catch (InterruptedException e) {
            throw new PoliqarpException(e, "Przetwarzanie zostało nieoczekiwanie przerwane (błąd serwera)");
        }
    }

    private void loadAnnotations(String query, ResultsPage<PoliqarpResult> resultPage, 
            PoliqarpResult.PQViewItemId startViewItemId, int length, 
            boolean loadFromCorpus, boolean loadModified) throws PoliqarpException {
        ArrayList<PoliqarpResult> pqrs = resultPage.getEntries();
        // invalidate result cache if query changed
        String client = CorpCorServiceUtils.getSessionId();
        SessionCache pqResultsCache = CorpCorServiceCaches.getPQResultsCache(client);
        if (!query.equals(pqResultsCache.getQuery())) {
            pqResultsCache.tier2Cache.invalidateAll();
            pqResultsCache.setQuery(query);
        }
        
        ArrayList<PoliqarpResult> resultingList = new ArrayList<PoliqarpResult>(pqrs.size());
        
        int viewItemId = 0; // 'i' and 'viewItemId' have same value quite by accident ;) keep them distinct
        PQRS: for (int i = 0; i < pqrs.size(); i++) {
            PoliqarpResult pqr = pqrs.get(i);
            if (loadAnnotationFromCacheOrCorpus(pqr, loadFromCorpus, client, pqResultsCache))
                if (loadAnnotationFromDB(pqr) && !loadModified)
                    continue PQRS;
            if ((viewItemId >= startViewItemId.id) && (length < 0 || viewItemId < startViewItemId.id + length)) {
                // enumerate the item and add to result list
                pqr.setViewItemId(viewItemId);
                resultingList.add(pqr);
            }
            viewItemId++;
        }
        resultPage.setEntries(resultingList);
        resultPage.setGlobalSize(viewItemId);
    }

    @Override
    public PoliqarpMetaData getPoliqarpMetaData(PoliqarpResult.PQResultItemId resultId) throws PoliqarpException {
        try {
            MetadataTicket ticket = new MetadataTicket(resultId.id);
            PQBridge.getInstance().forwardRequest(CorpCorServiceUtils.getSessionId(), ticket);
            return ticket.getResult();
        } catch (InterruptedException e) {
            throw new PoliqarpException(e, "Przetwarzanie zostało nieoczekiwanie przerwane (błąd serwera)");
        }
    }

    @Override
    public PoliqarpContext getPoliqarpWideContext(PoliqarpResult.PQResultItemId resultId) throws PoliqarpException {
        try {
            ContextTicket ticket = new ContextTicket(resultId.id);
            PQBridge.getInstance().forwardRequest(CorpCorServiceUtils.getSessionId(), ticket);
            return ticket.getResult();
        } catch (InterruptedException e) {
            throw new PoliqarpException(e, "Przetwarzanie zostało nieoczekiwanie przerwane (błąd serwera)");
        }
    }

    // ========================================================================================
    // == Corpus services
    // ========================================================================================

    @Override
    public ResultsPage<String> getCorpusTextIds(String query, int start, int length) throws CorpusApiException {
        return corpusApiBridge.getCorpusTextIds(query, start, length);
    }

    @Override
    public ResultsPage<CorpusTextBlock> getCorpusText(String id, int start, int length) throws CorpusApiException {
        return corpusApiBridge.getCorpusText(id, start, length);
    }

    @Override
    public TextBlockSegments getCorpusTextBlock(String textId, String blockId, int start, int length, boolean latestFromDB)
            throws CorpusApiException {
    	TextBlockSegments segments = corpusApiBridge.getCorpusTextBlock(textId, blockId, start, length);
    	
    	if (latestFromDB) {
    		for (CorpusSegment cs : segments.getSegments().getEntries()) {
    			Annotation a = annotationDao.findLatestByPath(cs.getSegmentId(), cs.getBlockId(), cs.getTextId());
    			if (a != null) {
    				cs.setMorpho(a.getOrth(), a.getBase(), a.getMsd(), a.getCtag());
    				cs.setComment(a.getComment());
    				cs.setDeleted(a.isDeleted());
    			}
    		}
    	}
    	
    	return segments;
    }

    @Override
    public CorpusSegment getCorpusSegment(String textId, String blockId, String id, boolean latestFromDB) throws CorpusApiException {
        CorpusSegment cs = corpusApiBridge.getCorpusSegment(textId, blockId, id);
        
        if (latestFromDB) {
        	Annotation a = annotationDao.findLatestByPath(cs.getSegmentId(), cs.getBlockId(), cs.getTextId());
			if (a != null) {
				cs.setMorpho(a.getOrth(), a.getBase(), a.getMsd(), a.getCtag());
				cs.setComment(a.getComment());
				cs.setDeleted(a.isDeleted());
			}
        }
        
        return cs;
    }

    @Override
    public ArrayList<CorpusSegment> getCorpusSegments(PoliqarpResult queryResult, PoliqarpMetaData meta,
            PoliqarpContext wideContext) throws CorpusApiException {
        String contextStr = wideContext.toString();
        int bestmIndex = CorpCorServiceUtils.getIndexOfMatchInContext(contextStr, queryResult);
        if (bestmIndex == -1) {
            throw new CorpusApiException("Could not find match in the context! match=" + queryResult + " context="+contextStr);
        }
        String lPart = contextStr.substring(0, bestmIndex);
        String rPart = contextStr.substring(bestmIndex + queryResult.getMatch().length());
        if (corpusApiBridge == null && CorpCorServiceUtils.isTestRun()) {
            corpusApiBridge = new CorpusAPIBridge();
        } else if (corpusApiBridge == null) {
            LOG.warn("corpusApiBridge is NULL");
        }
        return corpusApiBridge.getCorpusTextBlocks(meta, queryResult.getMatch(), lPart, rPart);
    }

    @Override
    public ArrayList<CorpusSegment> getCorpusSegments(String query, PoliqarpResult.PQResultItemId resultId) throws CorpusApiException,
            PoliqarpException {
        LOG.info("getCorpusSegments query='" + query + "' " + resultId);
        // TODO would be good to optimize this by removing the query
        ResultsPage<PoliqarpResult> results = getSinglePoliqarpQueryResult(query, resultId.id);
        PoliqarpMetaData metaData = getPoliqarpMetaData(resultId);
        PoliqarpContext context = getPoliqarpWideContext(resultId);
        return getCorpusSegments(results.getEntries().get(0), metaData, context);
    }

    public static void main(String[] args) throws Exception {
        CorpCorServiceImpl service = new CorpCorServiceImpl();
        // ResultsPage<PoliqarpResult> res =
        // service.getPoliqarpQueryResults("Formosusa i [base=postawienie]", 0,
        // -1, true);
        // ResultsPage<PoliqarpResult> res =
        // service.getPoliqarpQueryResults("Formosusa i postawienia", 0, -1,
        // true);
//        ResultsPage<PoliqarpResult> res = service.getQueryResults("kot", null, -1, true, true);
      ResultsPage<PoliqarpResult> res = service.getQueryResults("a", null, -1, true, true);
        ArrayList<PoliqarpResult> entries = res.getEntries();
        System.out.println("========================================================================================");
        for (PoliqarpResult poliqarpResult : entries) {
            System.out.println(poliqarpResult.toStringAnnotated("not loaded", "load failed"));
        }
        System.out.println("========================================================================================");
        PQBridge.getInstance().forwardShutdown();
    }
}
