package pl.waw.ipipan.corpcor.server.corpusapi;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

public abstract class CorpusTools {

	/**
	 * mapa corpusID->statystyka korpusu
	 */
	private HashMap<String, SenseStatistics> statisticsMap = new HashMap<String, SenseStatistics>();

	/**
	 * zwraca następne wystapienie danego sensu słowa wieloznacznego w korpusie
	 * 
	 * @param corpus korpus
	 * @param sense sens jak mi zostać wyszukany
	 * @return następne wystąpienie danego sensu słowa wieloznacznego w korpusie
	 */
	public abstract SenseSegmentGroup getNextSenseInstance(Corpus corpus, String sense);

	/**
	 * zwraca nastepne wystapienie dowolnego slowa wieloznacznego w korpusie
	 * 
	 * @param corpus
	 * @return nastepne wystapienie dowolnego slowa wieloznacznego w korpusie
	 */
	public abstract SenseSegmentGroup getNextSenseEntryInstance(Corpus corpus);

	/**
	 * 
	 * zwraca następne wystapienie danego słowa wieloznacznego w korpusie
	 * 
	 * @param corpus korpus
	 * @param senseEntry słowo wieloznaczne do wyszukania
	 * @return następny wystąpienie danego słowa wieloznacznego w korpusie
	 */
	public abstract SenseSegmentGroup getNextSenseEntryInstance(Corpus corpus, String senseEntry);

	/**
	 * zwraca statystyki wystąpień danego słowa wieloznacznego w korpusie (wraz
	 * z sensami tego słowa wieloznacznego)
	 * 
	 * @param corpus korpus
	 * @param senseEntry słowo wieloznaczne
	 * @return statystyki wystąpień danego słowa wieloznacznego w korpusie (wraz
	 *         z sensami tego słowa wieloznacznego)
	 */
	public SenseStatistics getSenseStatistics(Corpus corpus, String senseEntry) {
		if (!statisticsMap.containsKey(corpus.getId())
				|| !statisticsMap.get(corpus.getId()).availableSenseEntries().contains(senseEntry)) {
			// nie bylo w ogole statystyki lub nie zawierala danego slowa
			// nadpisalem equal i hashCode zeby to contains dzialalo
			// stworz statysytke dla tego slowa i dodaj ja
			SenseStatistics s = makeSenseStatistics(corpus, new HashSet<String>(Arrays
					.asList(new String[] { senseEntry }))); // nie kompletna
			// dodaj ja do mapy
			statisticsMap.put(corpus.getId(), s);
			return statisticsMap.get(corpus.getId());
		} else { // czyli: if (statisticsMap.get(corpus.getId()).isComplete() |
					// statisticsMap.get(corpus.getId()).availableSenseEntries().contains(senseEntry))
					// {//jezeli mam kompletna statystyke dla corpusu lub taka
					// ktora zawiera dany sens
			return statisticsMap.get(corpus.getId()).projectionToSenseEntries(
					Arrays.asList(new String[] { senseEntry }));
		}
	}

	/**
	 * zwraca statystyki wystąpień wszystkich słów wieloznacznych wraz z ich
	 * sensami w zadanym korpusie
	 * 
	 * @param corpus korpus
	 * @return statystyki wystąpień wszystkich słów wieloznacznych wraz z ich
	 *         sensami w zadanym korpusie
	 */
	public SenseStatistics getSenseStatistics(Corpus corpus) {
		if (!statisticsMap.containsKey(corpus.getId()) || !statisticsMap.get(corpus.getId()).isComplete()) {
			// jezeli albo nie ma statystyki wcale, albo sa w niej jedynie poszczegolne slowa
			
			// stworz statysytke dla calego korpusu, dodaj ja
			SenseStatistics s = makeSenseStatistics(corpus, null);
			// dodaj ja do mapy
			statisticsMap.put(corpus.getId(), s);
		}
		// tutaj mam kompletna statystyke dla corpusu
		return statisticsMap.get(corpus.getId());
	}

	public double getMFSfromStats(SenseStatistics stats) {		
			
		float cumulMax = 0;
		float cumulCount = 0;
			
		for (String senseId : stats.availableSenseEntries()) {
			SenseEntryStats ses = new SenseEntryStats(stats, senseId);
			cumulMax += ses.max;
			cumulCount += ses.senseEntryCount;
		}
		
		return cumulMax/cumulCount;
	}
	
	/** zwraca mape : sense entry id -> najczestszy podsens w korpusie
	 * @param stats
	 * @return
	 */
	public Map<String, String> getMFSDecisionsfromStats(SenseStatistics stats) {
		Map<String, String> decisions = new HashMap<String, String>();
		
		for (String senseId : stats.availableSenseEntries()) {
			SenseEntryStats ses = new SenseEntryStats(stats, senseId);
			decisions.put(senseId, ses.mostCommonSense);
		}
		
		return decisions;
	}
	
	/**
	 * pretty print
	 * 
	 * @param corpus
	 * @param interp
	 * @return
	 */
	// TODO przeargumentowac, co to mialo robic dokladnie???
	public abstract String format(Corpus corpus, List<List<? extends SegmentGroup>> interp);

	/**
	 * podklasy implementuja te metode. metoda jest uzywana przez
	 * getSenseStatistics(xxx) i zawiera jej zasadniczą część.
	 * 
	 * @param corpus
	 * @param senseEntries lista sensow, jesli null, to robimy dla wszystkich
	 * @return
	 */
	protected abstract SenseStatistics makeSenseStatistics(Corpus corpus, Set<String> senseEntries);
	
	public class SenseEntryStats {
		public int senseEntryCount;
		public String senseEntryId;
		public List<Integer> counts;
		public double mfsForEntry;
		public int max;
		public String mostCommonSense;
		
		public SenseEntryStats(SenseStatistics stats, String senseEntry) {
			super();
			
			senseEntryId = senseEntry;
			senseEntryCount = stats.getSenseEntryCount(senseEntryId);						
			Collection<String> subSenses = stats.getSenseEntrySenses(senseEntryId);
						
			counts = new ArrayList<Integer> (); 			
			
			max = -1;
			for (String sub : subSenses) {
				int sc = stats.getSenseCount(sub);				
				counts.add(sc);
				if (sc > max) {
					max = sc;
					mostCommonSense = sub;
				}
			}			
			
			Collections.sort(counts, Collections.reverseOrder());			
			
			mfsForEntry = 1.0*max/senseEntryCount;			
		}
		
		public void printLatex() {
			String line = senseEntryId+" & "+senseEntryCount+" & "+counts.size()+" & ";
			
			for (Integer count : counts)
				line+=count+"/";
			line = line.substring(0, line.length()-1);
			
			line+=" & "+ Math.round(mfsForEntry*1000.0) / 1000.0 + " \\\\ \\hline";
			System.out.println(line);
		}
	}
}
