package pl.waw.ipipan.corpcor.server.corpusapi.tei;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Random;
import java.util.Set;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.Attribute;
import javax.xml.stream.events.EndElement;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.corpusapi.Corpus;
import pl.waw.ipipan.corpcor.server.corpusapi.CorpusMetadata;
import pl.waw.ipipan.corpcor.server.corpusapi.CorpusText;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.pq.TEItoPQMetadata;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.readers.TEIMorphoReader;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.readers.TEIReader;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.readers.TEISegmentationReader;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.readers.TEISenseReader;
import pl.waw.ipipan.corpcor.server.corpusapi.tei.readers.TEITextReader;


public class TEICorpus implements Corpus, Serializable {
	
    private static class TEITextInfo implements Serializable {
        final Map<TEIBiblElement, List<String>> bibl = new HashMap<TEIBiblElement, List<String>>();
        final String path;
        TEITextInfo(String path) {
            this.path = path;
        }
    }
	
	private Map<String, TEITextInfo> corpusTextMap = null;
    private boolean cached = false;
    private List<String> corpusTexts = null;
	private String corpusId;
	
	protected Map<String, TEIReader> teiReaderMap = new HashMap<String, TEIReader>();
	protected transient XMLInputFactory factory = XMLInputFactory.newInstance();
	private String cachedFile;
	
	private static final Logger LOG = LoggerFactory.getLogger(TEICorpus.class);

	public TEICorpus(List<String> corpusTexts, boolean cached, String corpusId, String cachedFile) {
		this.cached = cached;
		this.corpusTexts = corpusTexts;
		this.corpusId = corpusId;
		this.cachedFile = cachedFile;
		
		LOG.info("A new TEIcorpus has been created. CorpusID: " + corpusId);
	}

	@Override
	public boolean close() {
		for (TEIReader reader : teiReaderMap.values()) {
			try {
				reader.close();
			} catch (XMLStreamException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		
		teiReaderMap.clear();
		
		return true;
	}

	@Override
	public TEICorpusHeader getCorpusHeader() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public TEICorpusText getCorpusText(String textId) {
		return new TEICorpusText(this, textId, corpusTextMap.get(textId).path);
	}
	
    @Override
    public CorpusText getCorpusText(CorpusMetadata metaSel) {
        // check each text in corpus
        CorpusText result = null;
        Set<Entry<String, TEITextInfo>> entrySet = corpusTextMap.entrySet();
        NEXT_TEXT: for (Entry<String, TEITextInfo> entry : entrySet) {
            // all must be matched or not found
            Set<String> metaKeySet = metaSel.getMetadataKeySet();
            for (String metaKey : metaKeySet) {
                TEIBiblElement teiBiblElement = TEItoPQMetadata.getForPQMetadata(metaKey);
                // check if this type of metadata is stored in TEI at all
                if (teiBiblElement != null) {
                    // and whether this text has it
                    if (entry.getValue().bibl.containsKey(teiBiblElement)) {
                        // and whether all metadata values are equal
                        if (!compareMetadataValues(entry.getValue().bibl.get(teiBiblElement), metaSel.getMetadataValues(metaKey))) {
                            continue NEXT_TEXT;
                        }
                    }
                }
            }
            // all metadata tags from selector are iterated and there is no mismatch
            result = new TEICorpusText(this, entry.getKey(), entry.getValue().path);
            break;
        }
        return result;
    }
    
	private boolean compareMetadataValues(List<String> corpusMetaValues, String[] lookupMetaValues) {
	    if (corpusMetaValues.size() == 0 || lookupMetaValues.length == 0)
	        return true;
	    // just make sure there are no conflicts
	    List<String> lookupList = Arrays.asList(lookupMetaValues);
        if (corpusMetaValues.containsAll(lookupList))
	        return true;
        if (lookupList.containsAll(corpusMetaValues))
            return true;
        return false;
    }

    @Override
	public List<String> getCorpusTextIds() {
		if (corpusTextMap == null) {
			return null;
		}
		List<String> ids = new ArrayList<String>();
		ids.addAll(corpusTextMap.keySet());
		return ids;
	}

	@Override
	public boolean open() {
		if (this.corpusTextMap == null) {
			this.corpusTextMap = getTextMap();
	
			if (this.corpusTextMap != null) {
				File cachedCorpus = new File(getCachedFile());
				if (!cachedCorpus.exists()) {
					ObjectOutputStream oos = null;
					try {
						oos = new ObjectOutputStream(new BufferedOutputStream(
								new FileOutputStream(cachedCorpus)));
						oos.writeObject(this);
					} catch (Exception e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					} finally {
						if (oos != null) {
							try {
								oos.close();
							} catch (IOException e1) {
								// TODO Auto-generated catch block
								e1.printStackTrace();
							}
						}
					}
				}
				
				return true;
			}
	
			return false;
		}
		
		return true;
	}

	@Override
	public boolean isCached() {
		return cached;
	}

	private Map<String, TEITextInfo> getTextMap() {
		Map<String, TEITextInfo> corpusTextMap = new LinkedHashMap<String, TEITextInfo>();

		XMLInputFactory factory = XMLInputFactory.newInstance();
		factory.setProperty(factory.SUPPORT_DTD, false);
		InputStreamReader reader;
		LOG.info("Start getTextMap()");

		try {
			for (String ct : corpusTexts) {
				LOG.info(ct);
			    String key = null;
			    TEITextInfo currentInfo = null;
				reader = new InputStreamReader(new FileInputStream(ct + File.separator + "header.xml"), "UTF-8");
				XMLEventReader eventReader = factory.createXMLEventReader(reader);
				while (eventReader.hasNext()) {
					XMLEvent event = eventReader.nextEvent();
					if (event.isStartElement()) {
						StartElement element = (StartElement) event;
						if (element.getName().toString().endsWith("teiHeader")) {
							Iterator it = element.getAttributes();
							while (it.hasNext()) {
								Attribute attr = (Attribute) it.next();
								if (attr.getName().getLocalPart().equalsIgnoreCase("id")) {
									key = attr.getValue();
									if (corpusTextMap.containsKey(key)) {
										System.out.println("Key already exists!");
//										System.out.println(corpusTextMap.get(key));
										System.out.println(ct);
										Random r = new Random(1);
										StringBuffer sb = new StringBuffer();
										sb.append(key);
										while (corpusTextMap.containsKey(sb.toString())) {
											sb.append(r.nextInt(10));
										}
										key = sb.toString();
									}
									corpusTextMap.put(key, new TEITextInfo(ct));
								}
							}
							// break;
						} else if (element.getName().getLocalPart().equals("bibl")) {
						    String value = getAttributeValue(element, "id");
						    if ("h_src-bibl".equals(value)) {
                                currentInfo = corpusTextMap.get(key);
						    }
						} else if (currentInfo != null) {
						    // assume this is a subelement of <bibl>, a metadata
						    String metaName = element.getName().getLocalPart();
                            String metaType = getAttributeValue(element, "type");
                            XMLEvent pcDataExpected = eventReader.nextEvent();
                            if (pcDataExpected.isCharacters()) {
                                // actual meta tag value
                                String metaValue = pcDataExpected.asCharacters().getData();
                                TEIBiblElement metaTag = new TEIBiblElement(metaName, metaType);
                                List<String> list = currentInfo.bibl.get(metaTag);
                                if (list == null) {
                                    list = new LinkedList<String>();
                                    currentInfo.bibl.put(metaTag, list);
                                }
                                list.add(metaValue);
                            }
						}
					} else if (event.isEndElement()) {
                       EndElement element = (EndElement) event;
                       if (element.getName().getLocalPart().equals("teiHeader")) {
                           key = null;
                       } if (element.getName().getLocalPart().equals("bibl")) {
                           currentInfo = null;
                       }
					}
				}
				eventReader.close();
				reader.close();
			}
		} catch (FileNotFoundException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (XMLStreamException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		
		LOG.info("Finish getTextMap()");

		return corpusTextMap;
	}
	
	private static String getAttributeValue(StartElement element, String attrLocalName) {
        @SuppressWarnings("unchecked")
        Iterator<Attribute> attributes = element.getAttributes();
        while (attributes.hasNext()) {
            Attribute attr = attributes.next();
            if (attr.getName().getLocalPart().equals(attrLocalName)) {
                return attr.getValue();
            }
        }
        return null;
	}

	@Override
	public String getId() {
		return corpusId;
	}
	
	public void closeTEIReader(String filename) throws XMLStreamException, IOException {
		TEIReader t = teiReaderMap.remove(filename);
		if (t != null) {
			t.close();
		}
	}
	
	public TEIReader getTEIReader(String fileName, TEICorpusText corpusText) {
		TEIReader teiReader = null;

		if ((teiReader = teiReaderMap.get(fileName)) == null) {
			teiReader = new TEIReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
		}

		return teiReader;
	}

	public TEISegmentationReader getTEISegmentationReader(TEICorpusText corpusText) {
		String fileName = corpusText.getSegmentationFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEISegmentationReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEISegmentationReader) teiReader;
		} else if (teiReader instanceof TEISegmentationReader) {
			return (TEISegmentationReader) teiReader;
		} else {
			return null;
		}
	}

	public TEISenseReader getTEISenseReader(TEICorpusText corpusText) {
		String fileName = corpusText.getSensesFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEISenseReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEISenseReader) teiReader;
		} else if (teiReader instanceof TEISenseReader) {
			return (TEISenseReader) teiReader;
		} else {
			return null;
		}
	}

	public TEIMorphoReader getTEIMorphoReader(TEICorpusText corpusText) {
		String fileName = corpusText.getMorphoFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEIMorphoReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEIMorphoReader) teiReader;
		} else if (teiReader instanceof TEIMorphoReader) {
			return (TEIMorphoReader) teiReader;
		} else {
			return null;
		}
	}

	public TEITextReader getTEITextReader(TEICorpusText corpusText) {
		String fileName = corpusText.getTextFilePath();
		TEIReader teiReader = teiReaderMap.get(fileName);

		if (teiReader == null) {
			teiReader = new TEITextReader(fileName, corpusText);
			teiReaderMap.put(fileName, teiReader);
			return (TEITextReader) teiReader;
		} else if (teiReader instanceof TEITextReader) {
			return (TEITextReader) teiReader;
		} else {
			return null;
		}
	}

	public XMLInputFactory getXMLInputFactory() {
		return factory;
	}

	@Override
	public String getCachedFile() {
		return cachedFile;
	}
	

	private void writeObject(ObjectOutputStream oos)
	    throws IOException {
	  oos.defaultWriteObject();
	}
	
	private void readObject(ObjectInputStream ois)
	    throws ClassNotFoundException, IOException {
	  ois.defaultReadObject();
	  factory = XMLInputFactory.newInstance();
	}
}
