/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12;

import ipipan.poliqarp.connection.PoliqarpConnection;
import ipipan.poliqarp.gui.Configuration;
import ipipan.poliqarp.logic.Corpus;
import ipipan.poliqarp.stat.Tagset;

import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.api.PQCQueryException;
import pl.waw.ipipan.corpcor.server.pq.client.api.PQCRequestHandler;
import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.brokers.CorpusBroker;
import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.brokers.CorpusListener;
import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.requests.Request;
import pl.waw.ipipan.corpcor.server.pq.client.api.common_1_3_12.requests.RequestFactory;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.PQCRequestTicket;

public class RequestHandler implements PQCRequestHandler, CorpusListener {

    private static final Logger LOG = LoggerFactory.getLogger(RequestHandler.class);

    private enum State {
        NOT_STARTED, STARTED, CLOSED;
    }

    private final Executor executor;
    private final Configuration config;
    private final String corpusImage;
    private final String id;
    private final Lock entryLock = new ReentrantLock();
    private final BlockingQueue<Transition> asyncOperationResult = new LinkedBlockingQueue<Transition>();

    private State state = State.NOT_STARTED;
    private PoliqarpConnection connection;
    private CorpusBroker broker;
    private Corpus corpus;
    private RequestFactory requestFactory;
    private Request request;

    RequestHandler(Executor executor, Configuration config, String corpusImage, String id) {
        this.executor = executor;
        this.config = config;
        this.corpusImage = corpusImage;
        this.id = id;
    }

    private void notifyTransition(Transition t) {
        LOG.debug("RequestHandler " + id + " received corpus transition " + t);
        asyncOperationResult.offer(t);
    }

    @Override
    public void opened(Corpus corpus) {
        notifyTransition(Transition.OPENED);
    }

    @Override
    public void openFailed(Corpus corpus) {
        notifyTransition(Transition.OPEN_FAILED);
    }

    @Override
    public void closed(Corpus corpus) {
        notifyTransition(Transition.CLOSED);
    }

    @Override
    public void start() throws IOException, InterruptedException {
        try {
            entryLock.lock();
            assertState(State.NOT_STARTED);
            if (broker != null)
                throw new IOException("Corpus already opened");
            String server = config.getConfigItem(Configuration.OPT_SERVER);
            int port = Integer.parseInt(config.getConfigItem(Configuration.OPT_PORT));
            connection = new PoliqarpConnection(server, port, "PQCRequestHandler-Session-" + id);
            config.updateConfig(connection);
            broker = new CorpusBroker(executor, this);
            broker.requestCorpusStart(connection, this.corpusImage);
            Transition transition = drainEvent();
            if (transition != Transition.OPENED) {
                state = State.CLOSED;
                throw new IOException("Corpus failed to open");
            }
            requestFactory = new RequestFactory(executor, config, connection, corpus, new Tagset(corpusImage));
            state = State.STARTED;
        } finally {
            entryLock.unlock();
        }
    }

    private Transition drainEvent() throws InterruptedException {
        Transition transition = asyncOperationResult.take();
        while (!asyncOperationResult.isEmpty()) {
            Transition transition2 = asyncOperationResult.poll(0, TimeUnit.SECONDS);
            if (transition2 != null)
                transition = transition2;
        }
        return transition;
    }

    @Override
    public void close() throws InterruptedException {
        try {
            entryLock.lock();
            assertState(State.STARTED);
            if (broker != null) {
                broker.requestCorpusClose();
                Transition transition = drainEvent();
                if (transition != Transition.CLOSED) {
                    LOG.warn("Unexpected transition while closing PQ connection " + transition);
                }
            }
            try {
                connection.close();
            } catch (IOException e) {
                LOG.warn("Error while closing PQ connection: " + e.getLocalizedMessage() + ", id: " + id);
            }
        } finally {
            state = State.CLOSED;
            entryLock.unlock();
        }
    }

    @Override
    public void runRequest(PQCRequestTicket<?> ticket) throws InterruptedException, IOException, PQCQueryException {
        try {
            entryLock.lock();
            assertState(State.STARTED);
            if (request == null || !request.canContinue(ticket)) {
                request = requestFactory.createRequest(ticket.getType());
            }
            request.run(ticket);
        } finally {
            entryLock.unlock();
        }
    }

    private void assertState(State s) {
        if (state != s)
            throw new IllegalStateException("RequestHandler not in state " + s);
    }
}
