/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.client.lifecycle;

import java.io.IOException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.client.api.PQCQueryException;
import pl.waw.ipipan.corpcor.server.pq.client.api.PQCRequestHandler;
import pl.waw.ipipan.corpcor.server.pq.client.tickets.PQCRequestTicket;

/**
 * @author Nestor Pawlowski
 */
public class PQCRequestHandlerWatchdogAdapter implements Runnable, PQCRequestHandler {

    private static final Logger LOG = LoggerFactory.getLogger(PQCRequestHandlerWatchdogAdapter.class);

    private final PQCRequestHandlerPool pool;
    private final ScheduledExecutorService executor;
    private final PQCRequestHandler handler;
    private final String clientId;
    private final long sessionTimeout;
    private ScheduledFuture<?> schedule;

    public PQCRequestHandlerWatchdogAdapter(ScheduledExecutorService executor, PQCRequestHandlerPool pool,
            String clientId, PQCRequestHandler handler, long sessionTimeout) {
        this.executor = executor;
        this.pool = pool;
        this.clientId = clientId;
        this.handler = handler;
        this.sessionTimeout = sessionTimeout;
    }

    @Override
    public void run() {
        cancelWatchdog();
        LOG.info("Session timeout, closing connection for " + this.clientId);
        this.pool.closeSession(this.clientId);
    }

    @Override
    public void start() throws IOException, InterruptedException {
        handler.start();
        resetWatchdog();
    }

    private synchronized void cancelWatchdog() {
        if (schedule != null) {
            LOG.debug("Cancelling session watchdog for session " + this.clientId);
            schedule.cancel(false);
        }
        schedule = null;
    }

    private synchronized void resetWatchdog() {
        cancelWatchdog();
        LOG.debug("Setting session watchog for client/session " + this.clientId + " in " + this.sessionTimeout + " secs");
        schedule = this.executor.schedule(this, this.sessionTimeout, TimeUnit.SECONDS);
    }

    @Override
    public void close() throws InterruptedException {
        cancelWatchdog();
        handler.close();
    }

    @Override
    public void runRequest(PQCRequestTicket<?> ticket) throws InterruptedException, IOException, PQCQueryException {
        try {
            handler.runRequest(ticket);
        } finally {
            resetWatchdog();
        }
    }
}
