/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.daemon.common_1_3_12;

import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import pl.waw.ipipan.corpcor.server.pq.common.AbstractPQObject;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfigurator;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDDeployment;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDOption;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDRuntimeConfig;
import pl.waw.ipipan.corpcor.server.pq.server.lifecycle.PQLMConfig;

public class Configurator extends AbstractPQObject implements PQDConfigurator {

    private static Map<String, String> processPqConfigValues(String logpath, PQDConfig dConfig) {
        HashMap<String, String> pqconfig = new HashMap<String, String>();
        pqconfig.putAll(dConfig.getPqdConfig());
        // make sure the setting exits - there is a default in the server anyway
        if (!pqconfig.containsKey(Constants.CONF_LOG_FILE)) {
            pqconfig.put(Constants.CONF_LOG_FILE, "poliqarpd.logfile.");
        }
        // check if absolute or relative path is used
        String logFile = pqconfig.get(Constants.CONF_LOG_FILE);
        if (logFile.indexOf(File.separatorChar) == -1) {
            logFile = logpath + logFile;
        }
        // add random extension
        if (logFile.endsWith(".log"))
            logFile = logFile.substring(0, logFile.length() - ".log".length());
        logFile = logFile + dConfig.getId() + ".log";
        pqconfig.put(Constants.CONF_LOG_FILE, logFile);
        return pqconfig;
    }

    private static void writeConfigFile(File configFile, Map<String, String> configMap) throws IOException {
        PrintWriter pw = new PrintWriter(configFile);
        Set<Entry<String, String>> entrySet = configMap.entrySet();
        for (Entry<String, String> entry : entrySet) {
            if (Constants.SUPPORTED_CONF_OPTIONS.contains(entry.getKey()))
                pw.println(entry.getKey() + "=" + entry.getValue());
        }
        pw.close();
    }

    public Configurator() {
        // keep no args
    }

    @Override
    public PQDRuntimeConfig configure(PQDDeployment deployment, PQLMConfig lmConfig, PQDConfig dConfig)
            throws IOException {

        final File workingDir = deployment.getDeploymentDir();

        String logpath = lmConfig.getLogRootDir().getAbsolutePath() + File.separator;
        final String stdoutLogFilePath = logpath + "pqd-stdout-" + dConfig.getId() + ".log";
        final String stderrLogFilePath = logpath + "pqd-stderr-" + dConfig.getId() + ".log";

        File configurationFile = new File(deployment.getDeploymentDir(), "poliqarpd." + dConfig.getId() + ".conf");
        if (lmConfig.isDeleteDeploymentOnExit())
            configurationFile.deleteOnExit();
        Map<String, String> processedConfig = processPqConfigValues(logpath, dConfig);
        writeConfigFile(configurationFile, processedConfig);

        return new PQDRuntimeConfig(workingDir, configurationFile, stdoutLogFilePath, stderrLogFilePath);
    }

    @Override
    public String getCompatibleVersion() {
        return Constants.PQ_VERSION;
    }

    @Override
    public String getPQDConfigOption(PQDOption option, PQDConfig dConfig) {
        String key = getOptionKey(option);
        if (key == null)
            return null;
        return dConfig.getPqdConfig().get(key);
    }

    @Override
    public boolean setPQDConfigOption(PQDOption option, String value, PQDConfig dConfig) {
        String key = getOptionKey(option);
        if (key == null)
            return false;
        dConfig.getPqdConfig().put(key, value);
        return true;
    }

    private String getOptionKey(PQDOption option) {
        String key;
        switch (option) {
        case HOSTNAME:
            key = Constants.CONF_HOSTNAME;
            break;
        case PORT:
            key = Constants.CONF_PORT;
            break;
        case LOG_FILE:
            key = Constants.CONF_LOG_FILE;
            break;
        case LOGGING:
            key = Constants.CONF_LOGGING;
            break;
        case MAX_CONNECTIONS:
            key = Constants.ECONF_MAX_CONNECTIONS;
            break;
        default:
            return null;
        }
        return key;
    }
}
