/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.lifecycle;

import java.io.File;
import java.util.ResourceBundle;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.server.PQUtil;

public final class PQLMConfig implements PQLMConstants {

    private static final Logger LOG = LoggerFactory.getLogger(PQLMConfig.class);

    private static File getDirFromBundle(ResourceBundle bundle, String dirKeyName, String defaultDir) {
        String path;
        if (bundle.containsKey(dirKeyName)) {
            defaultDir = bundle.getString(dirKeyName);
        }
        if (defaultDir.startsWith(File.separator) || (defaultDir.startsWith("."))) {
            path = defaultDir;
        } else {
            path = PQUtil.getDefaultTempPath() + File.separator + defaultDir;
        }
        File dir = new File(path);
        if (!dir.exists()) {
            dir.mkdirs();
        } else if (!dir.isDirectory()) {
            LOG.warn(dirKeyName + " directory not a directory: " + dir + ", using default");
            dir = new File(PQUtil.getDefaultTempPath());
        }
        return dir;
    }

    private final boolean deleteDeploymentOnExit;
    private final File deploymentRootDir;
    private final File logRootDir;
    private final long watchdogDelay;
    private final String pqdPlatform;
    private final String pqdVersion;
    private final int pqdMinDaemons;
    private final int pqdMaxDaemons;
    private final long sessionTimeout;

    public PQLMConfig(PQLMConfig c) {
        this.logRootDir = c.logRootDir;
        this.deploymentRootDir = c.deploymentRootDir;
        this.deleteDeploymentOnExit = c.deleteDeploymentOnExit;
        this.watchdogDelay = c.watchdogDelay;
        this.pqdMinDaemons = c.pqdMinDaemons;
        this.pqdMaxDaemons = c.pqdMaxDaemons;
        this.pqdPlatform = c.pqdPlatform;
        this.pqdVersion = c.pqdVersion;
        this.sessionTimeout = c.sessionTimeout;
    }

    public PQLMConfig(String bundleName) {
        ResourceBundle bundle = ResourceBundle.getBundle(bundleName);
        logRootDir = getDirFromBundle(bundle, LOG_DIR, "pqd-logs");
        deploymentRootDir = getDirFromBundle(bundle, DEPLOYMENT_DIR, "pqd-deployments");

        this.deleteDeploymentOnExit = getObjectFromBundle(Boolean.class, bundle, DELETE_ON_EXIT, true);
        this.pqdMaxDaemons = getObjectFromBundle(Integer.class, bundle, MAX_DAEMONS, 1);
        this.pqdMinDaemons = getObjectFromBundle(Integer.class, bundle, MIN_DAEMONS, 5);
        this.pqdPlatform = getObjectFromBundle(String.class, bundle, DAEMON_PLATFORM, "linux");
        this.pqdVersion = getObjectFromBundle(String.class, bundle, DAEMON_VERSION, "1.3.12");
        this.watchdogDelay = getObjectFromBundle(Integer.class, bundle, WATCHDOG_DELAY, 60000);
        this.sessionTimeout = getObjectFromBundle(Integer.class, bundle, SESSION_TIMEOUT, 60 * 15);
    }

    private <T> T getObjectFromBundle(Class<T> type, ResourceBundle bundle, String key, T defaultV) {
        if (bundle.containsKey(key)) {
            try {
                return type.getConstructor(String.class).newInstance(bundle.getString(key));
            } catch (Exception e) {
            }
        }
        return defaultV;
    }

    public File getDeploymentRootDir() {
        return this.deploymentRootDir;
    }

    public File getLogRootDir() {
        return this.logRootDir;
    }

    public long getWatchdogDelay() {
        return watchdogDelay;
    }

    public boolean isDeleteDeploymentOnExit() {
        return this.deleteDeploymentOnExit;
    }

    public String getPqdPlatform() {
        return pqdPlatform;
    }

    public String getPqdVersion() {
        return pqdVersion;
    }

    public int getPqdMinDaemons() {
        return pqdMinDaemons;
    }

    public int getPqdMaxDaemons() {
        return pqdMaxDaemons;
    }

    public long getSessionTimeout() {
        return sessionTimeout;
    }
}
