/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.lifecycle;

import java.io.IOException;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.common.PQObjectFactory;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfigurator;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDDeployment;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDRuntimeConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDControl;

public class PQLMMonitor implements Runnable {

    private static final Logger LOG = LoggerFactory.getLogger(PQLMMonitor.class);

    static interface MonitorListener {

        void monitorShutdown(PQLMMonitor monitor);
    }
    
    private class WatchdogKickoffCommand implements Runnable {

        private boolean cancelWatchdod = false;
        
        public void cancelWatchdog() {
            this.cancelWatchdod = true;
        }

        @Override
        public void run() {
            if (cancelWatchdod)
                return;
            if (shutdown) {
                LOG.info("Monitor shut down, ignoring ping");
                return;
            }
            boolean ping = watchdog.pingCommand();
            if (!ping) {
                LOG.info("Ping failed, restarting");
                destroyProcess();
            }
            // reschedule
            LOG.debug("Re-scheduling watchdog to kick off in " + lmConfig.getWatchdogDelay());
            watchdogExecutor.schedule(this, lmConfig.getWatchdogDelay(), TimeUnit.MILLISECONDS);
        }
    }

    private volatile Process pqd;
    private final PQDDeployment deploy;
    private final PQLMConfig lmConfig;
    private final PQDConfig dConfig;
    private final PQDConfigurator configurator;
    private volatile boolean shutdown;
    private final Thread monitorThread;
    private final ScheduledExecutorService watchdogExecutor;
    private final PQDControl watchdog;

    public PQLMMonitor(PQDDeployment deploy, PQLMConfig lmConfig, PQDConfig dConfig, PQDConfigurator configurator,
            ScheduledExecutorService watchdogExecutor) {
        this.deploy = deploy;
        this.lmConfig = lmConfig;
        this.dConfig = dConfig;
        this.configurator = configurator;
        this.watchdogExecutor = watchdogExecutor;
        PQDControl w = null;
        try {
            LOG.debug("Creating watchdog for " + deploy);
            w = PQObjectFactory.createDaemonWatchdog(deploy.getInitializedPlatform(), deploy.getInitializedVersion(),
                    this.watchdogExecutor, dConfig, configurator);
        } catch (ClassNotFoundException e) {
            LOG.warn("Failed to create watchdog", e);
        }
        this.watchdog = w;
        monitorThread = new Thread(this, "PQDMonitor await thread [" + dConfig.getId() + "]");
        monitorThread.setDaemon(true);
    }
    
    private Observable observable = new Observable();
    private static final String EVT_M_END = "M_END";

    public void addMonitorListener(final MonitorListener listener) {
        this.observable.addObserver(new Observer() {
            
            @Override
            public void update(Observable o, Object evt) {
                if (evt == EVT_M_END)
                    listener.monitorShutdown(PQLMMonitor.this);
            }
        });
    }

    private Lock lock = new ReentrantLock();
    private Condition startup = lock.newCondition();
    
    public void start() {
        lock.lock();
        try {
            LOG.debug("Starting monitor for " + this.deploy);
            monitorThread.start();

            // sync with first process start
            try {
                startup.await();
            } catch (InterruptedException e) {
                LOG.warn("start() interruped unexpectedly");
            }
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void run() {
        lock.lock();
        try {
            while (!shutdown) {
                try {
                    newProcess();
                } catch (IOException e) {
                    LOG.warn("Failed to (re)start PQD");
                    break;
                } finally {
                    startup.signal();
                }
                try {
                    try {
                        LOG.debug("Waiting until PQD process ends");
                        // clear the interruption flag to prevent exit from waitFor
                        Thread.interrupted();
                        lock.unlock();
                        pqd.waitFor();
                        LOG.debug("await thread: waitFor exited");
                    } finally {
                        lock.lock();
                    }
                } catch (InterruptedException e) {
                    destroyProcess();
                    break;
                } finally {
                    try {
                        LOG.info("PQD process ended with return code " + pqd.exitValue());
                    } catch (Exception e) {
                        LOG.warn("PQD process not ended");
                    }
                }
            }
        } finally {
            lock.unlock();
        }
        this.shutdown = true;
        this.observable.notifyObservers(EVT_M_END);
    }

    private WatchdogKickoffCommand watchdogKickoffCommand;

    private void newProcess() throws IOException {
        // cancel previous watchdog
        if (watchdogKickoffCommand != null)
            watchdogKickoffCommand.cancelWatchdog();

        // kill any existing instances
        watchdog.haltCommand();
        // wait until there are not connections possible
        if (!watchdog.disconnectionCheck())
            throw new IOException("Failed to stop existing PQD instance or port used by other application");

        LOG.debug("Creating new PQD process instance");
        PQDRuntimeConfig rtConfig = configurator.configure(this.deploy, this.lmConfig, this.dConfig);
        this.pqd = this.deploy.start(rtConfig);
        LOG.info("New PQD process instance created for id " + this.dConfig.getId());

        // wait until connections possible
        if (!watchdog.connectionCheck())
            throw new IOException("Failed to start new PQD instance: server doesn't accept test connections");
        
        // setup new watchdog. we're in critical section (newProcess() is)
        watchdogKickoffCommand = new WatchdogKickoffCommand();
        LOG.debug("Scheduling watchdog to kick off in " + lmConfig.getWatchdogDelay());
        watchdogExecutor.schedule(watchdogKickoffCommand, lmConfig.getWatchdogDelay(), TimeUnit.MILLISECONDS);
    }

    private void destroyProcess() {
        LOG.info("Destroying PQD process");
        try {
            lock.lock();
            pqd.destroy();
            pqd.waitFor();
            LOG.debug("destroy: waitFor exited");
        } catch (InterruptedException e) {
            LOG.warn("Cannot await end of process because of interrupt", e);
        } finally {
            lock.unlock();
        }
    }

    public void shutdown() {
        LOG.debug("Monitor shutdown requested");
        if (shutdown)
            return;
        this.shutdown = true;
        destroyProcess();
    }
}
