/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.server.pq.server.lifecycle;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDConfig;
import pl.waw.ipipan.corpcor.server.pq.server.daemon.PQDOption;

public class PQLMPool {

    private static final Logger LOG = LoggerFactory.getLogger(PQLMPool.class);

    private static int NEXT_CONN_ID = 0;

    private static synchronized String getNextConnId() {
        return "PQDConnectionSlot-" + String.valueOf(NEXT_CONN_ID++);
    }

    private class ConnectionPool {
        final List<PQDConnectionSlot> connections = new LinkedList<PQDConnectionSlot>();
        final PQLMMonitor monitor;
        final PQDConfig dConfig;
        final int port;

        ConnectionPool(PQLMMonitor monitor, PQDConfig dConfig, int port) {
            this.monitor = monitor;
            this.dConfig = dConfig;
            this.port = port;
        }

        boolean isAtConnectionLimit() {
            return connections.size() >= getMaxConnections();
        }

        private int getMaxConnections() {
            try {
                return Integer.valueOf(PQLMPool.this.starter.getPQDConfigOption(PQDOption.MAX_CONNECTIONS, dConfig));
            } catch (NumberFormatException e) {
                LOG.trace("Max connections not defined, fallback to default");
                return PQLMConstants.DEFAULT_MAX_CONNECTIONS;
            }
        }

        PQDConnectionSlot newConnection() {
            String hostname = PQLMPool.this.starter.getPQDConfigOption(PQDOption.HOSTNAME, dConfig);
            PQDConnectionSlot c = new PQDConnectionSlot(hostname, this.port, getNextConnId());
            this.connections.add(c);
            return c;
        }

        boolean removeConnection(PQDConnectionSlot c) {
            return this.connections.remove(c);
        }
    }

    private final int minPort;
    private final PQLMStarter starter;
    private final List<ConnectionPool> conPools = new LinkedList<PQLMPool.ConnectionPool>();
    private final PQLMConfig lmConfig;

    public PQLMPool(PQLMConfig lmConfig) throws Exception {
        this.lmConfig = lmConfig;
        
        // for monitoring purposes - pings and watchdogs
        ScheduledExecutorService executor = new ScheduledThreadPoolExecutor(3, new ThreadFactory() {
            @Override
            public Thread newThread(Runnable r) {
                Thread thread = new Thread(r, "PQLMPool monitoring pool thread");
                thread.setDaemon(true);
                return thread;
            }
        }, new ThreadPoolExecutor.DiscardPolicy());
        
        starter = new PQLMStarter(lmConfig, executor);
        PQDConfig config = starter.newConfig();
        minPort = Integer.parseInt(starter.getPQDConfigOption(PQDOption.PORT, config));

        Runtime.getRuntime().addShutdownHook(new Thread("PQLMPool shutdown hook thread") {
            @Override
            public void run() {
                LOG.info("Shutdown hook: closing monitors");
                for (ConnectionPool pool : conPools) {
                    pool.monitor.shutdown();
                }
                LOG.info("Shutdown hook: cleaning up deployment");
                starter.cleanup();
            }
        });
    }

    private ConnectionPool newPQDMonitor(int port) {
        LOG.info("New monitor spawn requested at port " + port);
        PQDConfig dConfig = this.starter.newConfig();
        dConfig.setId(String.valueOf(port));
        this.starter.setPQDConfigOption(PQDOption.PORT, String.valueOf(port), dConfig);
        PQLMMonitor monitor = this.starter.newMonitor(dConfig);
        monitor.addMonitorListener(new PQLMMonitor.MonitorListener() {
            @Override
            public void monitorShutdown(PQLMMonitor monitor) {
                LOG.info("PQ process monitor notified shutdown");
                cleanupShutdownMonitor(monitor);
            }
        });
        monitor.start();
        return new ConnectionPool(monitor, dConfig, port);
    }

    public synchronized PQDConnectionSlot newConnection() {
        LOG.debug("Creating new connection for pool");
        int limit = lmConfig.getPqdMaxDaemons();
        while (true) {
            boolean[] portsUsed = new boolean[limit];
            int totalConnections = 0;
            ConnectionPool lowestLoad = null;
            for (ConnectionPool pool : conPools) {
                totalConnections += pool.connections.size();
                portsUsed[pool.port - this.minPort] = true;
                if (pool.isAtConnectionLimit()) {
                    LOG.debug("Server at " + pool.port + " reached connection limit, trying other daemon");
                    continue;
                }
                if (lowestLoad == null || lowestLoad.connections.size() < pool.connections.size()) {
                    lowestLoad = pool;
                }
            }
            LOG.debug("Total connections: " + totalConnections);
            LOG.debug("Total servers: " + this.conPools.size());
            if (lowestLoad != null) {
                LOG.info("Creating new connection for server at " + lowestLoad.port + " current load is "
                        + lowestLoad.connections.size());
                return lowestLoad.newConnection();
            }
            // all connections used of existing servers
            if (conPools.size() >= limit) {
                LOG.info("Reached server limit, cannot create new connections");
                return null;
            }
            // find free port
            int freePort = -1;
            for (int i = 0; i < portsUsed.length; i++) {
                if (portsUsed[i] == false) {
                    freePort = this.minPort + i;
                    break;
                }
            }
            if (freePort == -1) {
                LOG.warn("Reached server limit, cannot create new connections");
                return null;
            }
            ConnectionPool pool = newPQDMonitor(freePort);
            conPools.add(pool);
        }
    }

    public synchronized void removeConnection(PQDConnectionSlot c) {
        LOG.debug("Removing connection " + c);
        for (ConnectionPool pool : conPools) {
            if (pool.removeConnection(c)) {
                if (pool.connections.isEmpty() && conPools.size() > lmConfig.getPqdMinDaemons()) {
                    pool.monitor.shutdown();
                    conPools.remove(pool);
                    return;
                }
            }
        }
    }

    public synchronized void destroyPool(PQDConnectionSlot c) {
        LOG.debug("Removing connection pool containing " + c);
        for (ConnectionPool pool : conPools) {
            if (pool.removeConnection(c)) {
                pool.monitor.shutdown();
                conPools.remove(pool);
                return;
            }
        }
    }

    private synchronized void cleanupShutdownMonitor(PQLMMonitor monitor) {
        LOG.debug("Removing connection pool of monitor " + monitor);
        for (ConnectionPool pool : conPools) {
            if (pool.monitor == monitor) {
                conPools.remove(pool);
                return;
            }
        }
    }
}
