/*
 * This file is part of the CorpCor suite.
 * 
 * Copyright (C) 2012 by Instytut Podstaw Informatyki Polskiej
 * Akademii Nauk (IPI PAN; Institute of Computer Science, Polish
 * Academy of Sciences; cf. www.ipipan.waw.pl).  All rights reserved.
 * 
 * This file may be distributed and/or modified under the terms of the
 * GNU General Public License version 2 as published by the Free Software
 * Foundation and appearing in the file gpl.txt included in the packaging
 * of this file.  (See http://www.gnu.org/licenses/translations.html for
 * unofficial translations.)
 * 
 * A commercial license is available from IPI PAN (contact 
 * ipi@ipipan.waw.pl for more information).  Licensees holding a valid 
 * commercial license from IPI PAN may use this file in accordance with 
 * that license.
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING
 * THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE.
 */
package pl.waw.ipipan.corpcor.shared.pq;

import java.io.Serializable;
import java.util.ArrayList;

import pl.waw.ipipan.corpcor.shared.corpusapi.CorpusSegment;

public class PoliqarpResult implements Serializable, Cloneable {

    public static class PQResultItemId implements Serializable {
        private static final long serialVersionUID = 1L;
        public int id = 0;
        public PQResultItemId() {
        }
        public PQResultItemId(int id) {
            this.id = id;
        }
        @Override
        public String toString() {
            return "RID=" + id;
        }
    }

    public static class PQViewItemId implements Serializable {
        private static final long serialVersionUID = 1L;
        public int id = 0;
        public PQViewItemId() {
        }
        public PQViewItemId(int id) {
            this.id = id;
        }
        @Override
        public String toString() {
            return "VID=" + id;
        }
    }

    public static class PQEntryId implements Serializable {
        private static final long serialVersionUID = 1L;
        public PQResultItemId resultId;
        public PQViewItemId viewId;
        public PQEntryId() {
            resultId = new PQResultItemId();
        }
        public PQEntryId(int resultId) {
            this.resultId = new PQResultItemId(resultId);
            this.viewId = new PQViewItemId(0);
        }
        @Override
        public String toString() {
            return "RID=" + resultId.id + " VID=" + viewId.id;
        }
    }

    public enum PQResultState {
        INVALID, PQ_QUERIED, CORPUS_UPDATE_OK, CORPUS_UPDATE_FAIL 
    }
    
    private static final long serialVersionUID = 1L;

    private PQEntryId entryId;
    private String lContext;
    private String rContext;
    private String match;
    private String[] orthSegments;
    private CorpusSegment[] corpusSegments;
    private PQResultState state = PQResultState.INVALID;

    public PoliqarpResult() {
        this(-1, "", "", "", null);
    }

    public PoliqarpResult(int pqId, String lContext, String rContext, String match, String[] orthSegments) {
        this.entryId = new PQEntryId(pqId);
        this.lContext = lContext;
        this.match = match;
        this.rContext = rContext;
        this.orthSegments = orthSegments;
        this.state = PQResultState.PQ_QUERIED;
    }

    public PQEntryId getEntryId() {
        return this.entryId;
    }

    public void setViewItemId(int viewItemId) {
        this.entryId.viewId = new PQViewItemId(viewItemId);
    }

    public String getLeftContext() {
        return lContext;
    }

    public String getRightContext() {
        return rContext;
    }

    public String getMatch() {
        return match;
    }

    public int getMatchSegmentsCount() {
        return orthSegments.length;
    }

    public String[] getMatchSegmentsOrth() {
        return orthSegments;
    }

    public CorpusSegment[] getMatchSegments() {
        return corpusSegments;
    }

    public PQResultState getResultState() {
        return state;
    }

    public void updatedFromCorpus(ArrayList<CorpusSegment> segments) {
        updatedFromCorpus(segments.toArray(new CorpusSegment[segments.size()]));
    }

    public void updatedFromCorpus(CorpusSegment[] segments) {
        if (segments.length != this.orthSegments.length)
            throw new IllegalArgumentException("Trying to put together different count of corpus segments compared to PQ segments");
        this.corpusSegments = segments;
        this.state = PQResultState.CORPUS_UPDATE_OK;
    }

    public void updateFromCorpusFailed() {
        if (this.state == PQResultState.CORPUS_UPDATE_OK)
            return;
        this.state = PQResultState.CORPUS_UPDATE_FAIL;
    }

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + entryId.resultId.id;
		result = prime * result
				+ ((lContext == null) ? 0 : lContext.hashCode());
		result = prime * result + ((match == null) ? 0 : match.hashCode());
		result = prime * result
				+ ((rContext == null) ? 0 : rContext.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		PoliqarpResult other = (PoliqarpResult) obj;
		if (entryId.resultId.id != other.entryId.resultId.id)
			return false;
		if (lContext == null) {
			if (other.lContext != null)
				return false;
		} else if (!lContext.equals(other.lContext))
			return false;
		if (match == null) {
			if (other.match != null)
				return false;
		} else if (!match.equals(other.match))
			return false;
		if (rContext == null) {
			if (other.rContext != null)
				return false;
		} else if (!rContext.equals(other.rContext))
			return false;
		return true;
	}

    public String getMatchAnnotated(String loadNeededText, String loadFailedText) {
        StringBuilder sb = new StringBuilder();
        if (state == PQResultState.PQ_QUERIED) {
            sb.append(this.match);
            sb.append(" (");
            sb.append(loadNeededText);
            sb.append(")");
        } else if (state == PQResultState.CORPUS_UPDATE_OK) {
            for (int i = 0; i < orthSegments.length; i++) {
                sb.append(orthSegments[i]);
                sb.append(corpusSegments[i].toAnnotationString());
            }
        } else if (state == PQResultState.CORPUS_UPDATE_FAIL) {
            sb.append(this.match);
            sb.append(" (");
            sb.append(loadFailedText);
            sb.append(")");
        }

        return sb.toString();
    }

    public String toStringAnnotated(String loadNeededText, String loadFailedText) {
        return "Result[" + entryId.resultId.id + "]: " + lContext + " " + getMatchAnnotated(loadNeededText, loadFailedText) + " " + rContext;
    }

    public PoliqarpResult clone()  {
        PQEntryId pqEntryId = new PQEntryId();
        pqEntryId.resultId = this.entryId.resultId != null ? new PQResultItemId(entryId.resultId.id) : null;
        pqEntryId.viewId = this.entryId.viewId != null ? new PQViewItemId(entryId.viewId.id) : null;
        CorpusSegment[] segmentsClone = null;
        if (this.corpusSegments != null) {
            segmentsClone = new CorpusSegment[this.corpusSegments.length];
            for (int i = 0; i < segmentsClone.length; i++) {
                segmentsClone[i] = corpusSegments[i].clone();
            }
        }
        PoliqarpResult c = new PoliqarpResult();
        c.corpusSegments = segmentsClone;
        c.entryId = pqEntryId;
        c.lContext = lContext;
        c.match = match;
        c.orthSegments = orthSegments;
        c.rContext = rContext;
        c.state = state;
        return c;
    }
}
