/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package KnoW;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import resources.FrequencyCounter;
import resources.RelationsPlwordnet;
import resources.SemSimPlwordnet;
import resources.SynonymDictionary;
import resources.SynonymsPlwordnet;
import resources.SynonymsSynonimyUx;
import utils.CorpusManager;
import utils.TakipiManager;
import annotation.Annotator;
import annotation.WordExpert;
import corpusapi.tei.TEICorpus;
import corpusapi.tei.TEISenseInventory;
import evaluation.AnnotationReader;
import evaluation.AnnotationStats;
import evaluation.Evaluator;
import evaluation.SingleMethodEvaluation;
import evaluation.Evaluator.Sorting;

/**
 * Main class of the application
 * 
 * @author Mateusz Kopec
 * 
 */
public class KnoW {
	private static TEICorpus corpus;
	private static AnnotationStats goldenStandard;
	private static TEISenseInventory dict;

	private static Annotator annotator;
	private static Evaluator evaluator;

	// synonyms providers
	private static SynonymsSynonimyUx synonimyUx = null;
	private static SynonymsPlwordnet plwordnetSynonyms = null;
	private static RelationsPlwordnet plwordnetRelations = null;
	private static SemSimPlwordnet plwordnetSemSim = null;

	private static FrequencyCounter frequencyCounter = null;
	private static TakipiManager takipiManager = null;

	private static final String frequencyCounterPath = "data" + File.separator + "knowledge" + File.separator
			+ "frequencies.txt";

	private static final String annotationFilename = "ann_senses_multi";

	/**
	 * Entry point to the application. Requires 5 parameters, 6th is optional.
	 * First argument is experiment description file path. Second is path to
	 * output file. Third is path to the corpus config file. Fourth is path to
	 * dictionary file. Fifth is type of output: all, compare or lexemes.
	 * Last is maximum number of methods to print
	 * 
	 * @param args
	 */
	public static void main(String[] args) {

		if (args.length > 6 || args.length < 5) {
			System.out.println("Wrong number of arguments specified.");
			System.out
					.println("Should be: KnoW.jar experimentDescriptionsPath outputPath corpusPath dictionaryPath outputType [maxMethods]");
			System.exit(1);
		}

		String experimentDescriptionsPath = args[0];
		String outputFilePath = args[1];
		String corpusPath = args[2];
		String dictPath = args[3];		
		String option = args[4];
		Integer howMany = null;
		if (args.length == 6) {
			try {
				howMany = Integer.valueOf(args[5]);
			} catch (NumberFormatException e) {
				System.out.println("Error parsing the max number of methods parameter. " + args[5]
						+ " is not proper integer.");
				System.exit(1);
			}
			if (howMany < 1) {
				System.out.println("Max number of methods parameter should be greater than 0.");
				System.exit(1);
			}
		}

		checkArguments(experimentDescriptionsPath, option, corpusPath, dictPath);

		initialize(corpusPath, dictPath);

		// loading of method descrpitions
		List<List<String>> spl = null;
		try {
			spl = getSplits(experimentDescriptionsPath);
		} catch (IOException e) {
			e.printStackTrace();
			System.exit(1);
		}

		// /////////////// annotation //////////////////////

		annotate(spl, annotationFilename);

		// /////////////// evaluation //////////////////////

		Collection<SingleMethodEvaluation> results = evaluate(spl, annotationFilename, dict);

		// selection printing to a file
		Collection<SingleMethodEvaluation> selectedResults = new ArrayList<SingleMethodEvaluation>();

		File f = new File(outputFilePath);
		try {
			BufferedWriter bw = new BufferedWriter(new FileWriter(f));

			if (option.equals("compare")) {

				bw.append("ACC\n");
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.ACCURACY, howMany, goldenStandard));
				evaluator.printEvaluationResults(bw, selectedResults, goldenStandard);

				bw.append("RARE\n");
				selectedResults.clear();
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.RARE, howMany,
						goldenStandard));
				evaluator.printEvaluationResults(bw, selectedResults, goldenStandard);

				bw.append("IMPROVE\n");
				selectedResults.clear();
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.MFS_IMPROVEMENT, howMany,
						goldenStandard));
				evaluator.printEvaluationResults(bw, selectedResults, goldenStandard);

			} else if (option.equals("all")) {
				System.out.println("ALL");
				selectedResults.clear();
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.ACCURACY, null, goldenStandard));
				evaluator.printEvaluationResults(bw, selectedResults, goldenStandard);

			} else {
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.ACCURACY, howMany, goldenStandard));
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.RARE, howMany,
						goldenStandard));
				selectedResults.addAll(evaluator.selectBestMethods(results, Sorting.MFS_IMPROVEMENT, howMany,
						goldenStandard));
				evaluator.printEvaluationResultsOnLexemes(bw, selectedResults, goldenStandard);
			}

			bw.flush();
			bw.close();

		} catch (IOException e) {
			e.printStackTrace();
			System.exit(1);
		}
		saveCaches();
		
		System.out.println("Removing annotation files...");
		for (String cid : corpus.getCorpusTextIds()) {
			File dir = new File(corpus.getCorpusText(cid).getPath());
			for (File file : dir.listFiles()) {
				if (file.isFile() && file.getName().startsWith(annotationFilename))
					file.delete();
			}
		}
		System.out.println("Done");
		
	}

	private static void checkArguments(String experimentDescriptionsPath, String option, String corpusPath, String dictPath) {
		File exp = new File(experimentDescriptionsPath);
		if (!exp.exists() || !exp.isFile()) {
			System.out.println("Experiment description file doesn't exist.");
			System.exit(1);
		}
		File corp = new File(corpusPath);
		if (!corp.exists() || !corp.isFile()) {
			System.out.println("Corpus description file doesn't exist.");
			System.exit(1);
		}
		File dict = new File(dictPath);
		if (!dict.exists() || !dict.isFile()) {
			System.out.println("Dictionary file doesn't exist.");
			System.exit(1);
		}
		if (!option.equals("compare") && !option.equals("all") && !option.equals("lexemes")) {
			System.out.println("Wrong option for output type.");
			System.exit(1);
		}
	}

	private static Collection<SingleMethodEvaluation> evaluate(List<List<String>> spl, String filename,
			TEISenseInventory dict) {
		Collection<SingleMethodEvaluation> results = new HashSet<SingleMethodEvaluation>();

		int c = 0;
		for (List<String> methods : spl) {
			System.out.println("Split nr " + (++c));

			String splitsuf = "";
			if (spl.size() > 1)
				splitsuf = "_" + c;

			AnnotationReader annotationReader = new AnnotationReader(filename + splitsuf + ".txt", methods);
			results.addAll(evaluator.evaluate(corpus, annotationReader, dict));
		}
		return results;
	}

	private static List<List<String>> getSplits(String experimentDescriptionsPath) throws IOException {
		List<String> allmethods = new ArrayList<String>();

		BufferedReader br = new BufferedReader(new FileReader(new File(experimentDescriptionsPath)));

		String line = "";
		while ((line = br.readLine()) != null)
			allmethods.add(line);

		System.out.println("Number of methods: " + allmethods.size());

		List<List<String>> spl = splitMethods(allmethods, 50);
		System.out.println("Splitted into " + spl.size() + " splits.");
		return spl;
	}

	private static void annotate(List<List<String>> spl, String filename) {
		try {
			int c = 0;
			for (List<String> methods : spl) {
				System.out.println("Split nr " + (++c));

				System.out.println("Creating wordExperts...");
				Map<String, List<WordExpert>> wordExpertsMulti = annotator.createWordExperts(dict, methods, corpus,
						takipiManager);

				System.out.println("..done.");

				String splitsuf = "";
				if (spl.size() > 1)
					splitsuf = "_" + c;

				annotator.annotateMulti(corpus, filename + splitsuf + ".txt", wordExpertsMulti);

				wordExpertsMulti.clear();
				wordExpertsMulti = null;
			}
		} catch (Exception e) {
			System.out.println("Error while annotating. Details follow.");
			e.printStackTrace();
			System.exit(1);
		}
	}

	/**
	 * Splits a list of methods into buckets of chosen size.
	 * 
	 * @param methods
	 * @param bucketSize
	 * @return list of lists of methods
	 */
	private static List<List<String>> splitMethods(List<String> methods, int bucketSize) {
		List<List<String>> result = new ArrayList<List<String>>();
		while (!methods.isEmpty()) {
			List<String> sublist = methods.subList(0, Math.min(methods.size(), bucketSize));
			List<String> bucketList = new ArrayList<String>(sublist);
			sublist.clear();
			result.add(bucketList);
		}
		return result;
	}

	/**
	 * Gets synonimyUx synonym dictionary. If not created, creates it.
	 * 
	 * @return dictionary
	 */
	public static SynonymDictionary getSynonimyUx() {
		if (synonimyUx == null)
			synonimyUx = new SynonymsSynonimyUx();
		return synonimyUx;
	}

	/**
	 * Gets Plwordnet synonym dictionary. If not created, creates it.
	 * 
	 * @return dictionary
	 */
	public static SynonymDictionary getPlwordnetSynonyms() {
		if (plwordnetSynonyms == null)
			plwordnetSynonyms = new SynonymsPlwordnet();
		return plwordnetSynonyms;
	}

	/**
	 * Gets Plwordnet with relations synonym dictionary. If not created, creates
	 * it.
	 *  
	 * @return dictionary
	 */
	public static SynonymDictionary getPlwordnetRelations() {
		if (plwordnetRelations == null)
			plwordnetRelations = new RelationsPlwordnet();
		return plwordnetRelations;
	}

	/**
	 * Gets Plwordnet semantic similarity synonym dictionary. If not created,
	 * creates it.
	 * 
	 * @return dictionary
	 */
	public static SynonymDictionary getPlwordnetSemSim() {
		if (plwordnetSemSim == null)
			plwordnetSemSim = new SemSimPlwordnet();
		return plwordnetSemSim;
	}

	/**
	 * Gets frequency statistics. If not created, creates them.
	 * 
	 * @return frequency stats
	 */
	public static FrequencyCounter getFrequencyCounter() {
		if (frequencyCounter == null)
			frequencyCounter = new FrequencyCounter(frequencyCounterPath);
		return frequencyCounter;
	}

	private static void saveCaches() {
		if (plwordnetSynonyms != null)
			plwordnetSynonyms.saveCache();

		if (plwordnetRelations != null)
			plwordnetRelations.saveCache();

		if (plwordnetSemSim != null)
			plwordnetSemSim.saveCache();

		if (takipiManager != null)
			takipiManager.saveCache();
	}

	private static void initialize(String corpusConfigFilePath, String senseInventoryPath) {
		try {
			corpus = CorpusManager.getCorpusFromConfigFile(corpusConfigFilePath);
			dict = TEISenseInventory.createFromXML(senseInventoryPath);
			goldenStandard = CorpusManager.getSenseStatisticsForCorpus(corpus, dict);
			takipiManager = new TakipiManager();
			annotator = new Annotator();
			evaluator = new Evaluator();
		} catch (Exception ex) {
			System.out.println("Errors during initialization. Details follow");
			ex.printStackTrace();
			System.exit(1);
		}
	}
}
