/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package annotation;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import utils.TakipiManager;
import basic.Context;
import basic.Gloss;
import corpusapi.Sense;

/**
 * Word expert lesk-based.
 * 
 * @author Mateusz Kopec
 * 
 */
public class WELesk extends WordExpert {

	private MatchingStrategy matchingStrategy;
	private String paramString;

	// parameters
	private int contextWindow;
	private boolean fullGloss;

	/**
	 * map: senseID -> preprocessedGloss
	 */
	private Map<String, Gloss> preprocessedGlosses = new HashMap<String, Gloss>();

	/**
	 * Constructor.
	 * 
	 * @param baseForm
	 *            word base form
	 * @param senses
	 *            senses of this word
	 * @param params
	 *            parameters of method
	 * @param takipi
	 *            takipi manager to takipize glosses
	 * @throws Exception
	 */
	public WELesk(String baseForm, Collection<? extends Sense> senses, String params, TakipiManager takipi)
			throws Exception {
		super(baseForm, senses);

		paramString = "Lesk:" + params;

		String[] splitted = params.split(":", 3);

		// input
		String cw = splitted[0];
		String gs = splitted[1];

		this.contextWindow = Integer.valueOf(cw);
		this.fullGloss = (gs.equalsIgnoreCase("full"));

		// preprocess of glosses
		for (Sense sense : senses)
			preprocessedGlosses.put(sense.getId(), preprocessGloss(sense, takipi));

		matchingStrategy = new MatchingStrategy(preprocessedGlosses, splitted[2]);
	}

	private static Comparator<Entry<String, Double>> comparator = new Comparator<Entry<String, Double>>() {

		@Override
		public int compare(Entry<String, Double> arg0, Entry<String, Double> arg1) {
			return arg1.getValue().compareTo(arg0.getValue());
		}
	};

	/*
	 * (non-Javadoc)
	 * 
	 * @see annotation.WordExpert#disambiguate(basic.Context)
	 */
	@Override
	public Sense disambiguate(Context context) {

		// preprocessing of context
		Context preprocessedContext = new Context(context);
		preprocessedContext.cutToWindow(contextWindow);

		// comparement of contexts and senses' glosses
		Sense answer = null;
		double max = Double.NEGATIVE_INFINITY;
		Map<String, Double> similarities = new HashMap<String, Double>();

		Map<String, Double> contextVector = matchingStrategy.prepareContextVector(preprocessedContext);
		for (Sense sense : senses) {
			String senseId = sense.getId();

			double similarity = matchingStrategy.computeSimilarity(contextVector, senseId);
			similarities.put(senseId, similarity);
			if (similarity > max) {
				max = similarity;
				answer = sense;
			}
		}

		// normalization and sorting of similiarities
		List<Entry<String, Double>> similaritiesSorted = new ArrayList<Entry<String, Double>>(similarities.entrySet());
		Collections.sort(similaritiesSorted, comparator);
		float total = 0;
		for (Entry<String, Double> entry : similaritiesSorted)
			total += entry.getValue();

		if (total != 0)
			for (Entry<String, Double> entry : similaritiesSorted)
				entry.setValue(entry.getValue() / total);

		return answer;
	}

	private static final Pattern pat = Pattern.compile(
			"<orth>(.*?)</orth>.*?<lex disamb=\"1\"><base>(.*?)</base><ctag>(.*?)</ctag></lex>", Pattern.DOTALL);

	private Gloss preprocessGloss(Sense sense, TakipiManager takipi) {
		Gloss gloss = new Gloss();

		String definition = null;
		StringBuffer sb = new StringBuffer();

		if (fullGloss) {
			for (String def : sense.getDefinitions())
				sb.append(def + " ");
			definition = sb.toString();
		} else {
			definition = sense.getDefinition();
		}

		String taggedGloss = takipi.takipize(definition);

		Matcher matcher = pat.matcher(taggedGloss);

		while (matcher.find()) {
			if (matcher.groupCount() != 3) {
				System.out.println("error parsing tagging!");
				System.exit(1);
			}
			String orth = matcher.group(1);
			String base = matcher.group(2);
			@SuppressWarnings("unused")
			String ctag = matcher.group(3);
			gloss.addWord(orth, base);
		}

		return gloss;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see annotation.WordExpert#getParamString()
	 */
	@Override
	public String getParamString() {
		return paramString;
	}

}
