/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package utils;

import java.io.File;
import java.rmi.RemoteException;
import java.util.HashMap;
import java.util.Map;

import pl.wroc.pwr.plwordnet.clarin.ws.takipi.DocumentFormat;
import pl.wroc.pwr.plwordnet.clarin.ws.takipi.TaggerResponse;
import pl.wroc.pwr.plwordnet.clarin.ws.takipi.TakipiProxy;

/**
 * Helper to use takipi webservice. Caches queries.
 * 
 * @author Mateusz Kopec
 * 
 */
/**
 * @author Mateusz Kopec
 * 
 */
public class TakipiManager {
	protected Map<String, String> cache;

	/**
	 * Where is the cache stored
	 * 
	 * @return path
	 */
	public String getCacheFilename() {
		return "data" + File.separator + "cache" + File.separator + "takipi";
	}

	/**
	 * Constructor
	 */
	public TakipiManager() {
		cache = loadCache();
	}

	/**
	 * Saves cache
	 */
	public void saveCache() {
		FileManager.saveObject(cache, getCacheFilename());
	}

	/**
	 * Loads cache
	 * 
	 * @return cache
	 */
	@SuppressWarnings("unchecked")
	public Map<String, String> loadCache() {
		Map<String, String> cache = (HashMap<String, String>) FileManager.loadObject(getCacheFilename());
		if (cache != null)
			return cache;
		return new HashMap<String, String>();
	}

	/**
	 * Takipize text
	 * 
	 * @param text
	 * @return takipized text
	 */
	public String takipize(String text) {
		if (cache.containsKey(text)) {
			return cache.get(text);
		}

		System.out.println("Takipizing");
		TakipiProxy proxy = new TakipiProxy();
		// Invoke service
		try {
			TaggerResponse request = proxy.tag(text, DocumentFormat.TXT, true);
			String token = request.getMsg();
			int status = request.getStatus();
			// Check whether the request was queued
			if (status == 2) {
				// Check the request status until is 2 (queued) or 3 (in
				// processing)
				do {
					status = proxy.getStatus(token);
				} while (status == 2 || status == 3);
				if (status == 1) {
					// If the status is 1 then fetch the result
					TaggerResponse result = proxy.getResult(token);
					cache.put(text, result.getMsg());
					return result.getMsg();
				}
			}
		} catch (RemoteException e) {
			e.printStackTrace();
			System.exit(1);
		}
		return text;
	}
}
