/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package evaluation;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import corpusapi.CorpusText;

/**
 * Class for reading annotation files
 * 
 * @author Mateusz Kopec
 * 
 */
public class AnnotationReader {

	private BufferedReader br = null;
	private String filename = null;
	private List<String> annotatorsNames = null;

	/**
	 * Constructor
	 * 
	 * @param annotationFilename
	 *            filename of annotation files
	 * @param annotatorsNames
	 *            methods which responses are in the files
	 */
	public AnnotationReader(String annotationFilename, List<String> annotatorsNames) {
		this.filename = annotationFilename;
		this.annotatorsNames = annotatorsNames;
	}

	/**
	 * Sets reader for annotation file of given corpus text
	 * 
	 * @param ct
	 *            corpus text
	 * @throws IOException
	 */
	public void setCurrentCorpusText(CorpusText ct) throws IOException {
		br = new BufferedReader(new FileReader(ct.getPath() + File.separator + filename));
	}

	/**
	 * Gets next annotation
	 * 
	 * @return annotation
	 * @throws IOException
	 */
	public SegmentAnnotation getNextTaggedSegment() throws IOException {
		SegmentAnnotation result = null;

		String line = br.readLine();
		if (line != null) {
			result = new SegmentAnnotation();

			String splitted[] = line.split("\t", 2);
			result.segmentId = splitted[0];

			String[] annotations = splitted[1].split("\t");

			int i = 0;
			for (String annotation : annotations)
				result.annotations.put(annotatorsNames.get(i++), annotation);

			result.lexeme = annotations[0].split("#")[1].split("\\.")[0];
		} else {
			br.close();
			br = null;
		}

		return result;
	}

	/**
	 * Gets descriptions of methods used for annotation
	 * 
	 * @return list of descriptions
	 */
	public List<String> getAnnotatorsNames() {
		return annotatorsNames;
	}

	/**
	 * Annotation of given segment
	 * 
	 * @author Mateusz Kopec
	 * 
	 */
	public static class SegmentAnnotation {
		/**
		 * Segment id
		 */
		public String segmentId;
		/**
		 * Annotations: method description -> chosen sense id
		 */
		public Map<String, String> annotations = new HashMap<String, String>();
		/**
		 * Lexeme
		 */
		public String lexeme;
	}
}
