/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package resources;

import java.io.File;
import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

import pl.wroc.pwr.ci.plwordnet.Main;
import pl.wroc.pwr.ci.plwordnet.database.dao.SynsetDAO;
import pl.wroc.pwr.ci.plwordnet.database.dto.LexicalUnitDTO;
import pl.wroc.pwr.ci.plwordnet.database.dto.SynsetDTO;
import utils.FileManager;

/**
 * Synonym dictionary for plwordnet with relations
 * 
 * @author Mateusz Kopec
 * 
 */
public class RelationsPlwordnet extends SynonymDictionary implements Serializable {

	private static final long serialVersionUID = 2145615861956784458L;

	private String cacheFilename = "data" + File.separator + "cache" + File.separator + "relations_plwordnet";

	private Main plwordnetProxy;
	private Map<String, SynonymEntry> synonyms;

	/**
	 * Constructor
	 */
	public RelationsPlwordnet() {
		synonyms = new HashMap<String, SynonymEntry>();
		plwordnetProxy = new Main();
		loadCache();
	}

	/**
	 * Saves cache in a file on disk
	 */
	public void saveCache() {
		FileManager.saveObject(synonyms, cacheFilename);
	}

	/**
	 * Loads cache from a file on disk
	 */
	@SuppressWarnings("unchecked")
	public void loadCache() {
		Map<String, SynonymEntry> cache = (HashMap<String, SynonymEntry>) FileManager.loadObject(cacheFilename);
		if (cache != null)
			synonyms = cache;
	}

	static int c = 0;

	/* (non-Javadoc)
	 * @see resources.SynonymDictionary#getSynonymsForLemma(java.lang.String, boolean)
	 */
	@Override
	public Collection<String> getSynonymsForLemma(String lemma, boolean onlyMonosemous) {
		SynonymEntry se = synonyms.get(lemma);
		if (se == null) { // not searched yet

			System.out.println(c++ + " Relations : Searching for synonyms of " + lemma);
			if (c % 1000 == 0) {
				System.out.println("Saving cache");
				this.saveCache();
			}

			se = new SynonymEntry();
			synonyms.put(lemma, se);

			Collection<LexicalUnitDTO> u = plwordnetProxy.getUnits(lemma);
			if (u.size() > 1)
				se.isPolysemous = true;

			Collection<String> rels = new HashSet<String>();
			rels.add("hiperonimia");
			rels.add("część");
			rels.add("bliskoznaczność");
			rels.add("fuzzynimia_synsetów");

			// get unique synsets
			Collection<SynsetDTO> synsets = new HashSet<SynsetDTO>();
			for (LexicalUnitDTO unit : u) {
				for (SynsetDTO synset : plwordnetProxy.getSynsets(unit)) {
					synsets.addAll(plwordnetProxy.getSynsetsInRelation(synset, rels, false));
					synsets.add(synset);
				}
			}

			// get lemmas from synsets
			for (SynsetDTO synset : synsets) {
				Collection<LexicalUnitDTO> units = SynsetDAO.dbFastGetUnits(synset);
				for (LexicalUnitDTO unit : units) {
					se.synonyms.add(unit.getLemma());
				}
			}
		}

		if (onlyMonosemous && se.isPolysemous)
			return new HashSet<String>();

		return se.synonyms;
	}
}
