/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package resources;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

/**
 * Proveides synonyms from synonimy.ux dictionary
 * 
 * @author Mateusz Kopec
 * 
 */
public class SynonymsSynonimyUx extends SynonymDictionary {

	private Map<String, SynonymEntry> synonyms;
	private File file;

	/**
	 * Constructor
	 */
	public SynonymsSynonimyUx() {
		synonyms = new HashMap<String, SynonymEntry>();
		file = new File("data" + File.separator + "knowledge" + File.separator + "synonyms.txt");
		try {
			loadDictionary();
		} catch (IOException e) {
			e.printStackTrace();
			System.exit(1);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see resources.SynonymDictionary#getSynonymsForLemma(java.lang.String,
	 * boolean)
	 */
	@Override
	public Collection<String> getSynonymsForLemma(String lemma, boolean onlyMonosemous) {
		SynonymEntry entry = synonyms.get(lemma);

		// if no synonyms found
		if (entry == null)
			return new HashSet<String>();

		// if lemma is polysemous and we want monosemous ones
		if (onlyMonosemous && entry.isPolysemous)
			return new HashSet<String>();

		return entry.synonyms;
	}

	/**
	 * Loads dictionary from a file on disk
	 * 
	 * @throws IOException
	 */
	public void loadDictionary() throws IOException {
		System.out.println("Loading synonyms ux...");

		FileInputStream fis = new FileInputStream(file);
		BufferedReader input = new BufferedReader(new InputStreamReader(fis, "iso-8859-2"));

		String line;
		while ((line = input.readLine()) != null) {
			if (!line.startsWith("#")) {

				// get the synonym group
				String[] splitted = line.split(";");
				Collection<String> synGroup = new HashSet<String>();
				for (String s : splitted)
					synGroup.add(s);

				// add synonyms to each words from group
				for (String s : synGroup) {
					SynonymEntry entry = synonyms.get(s);

					if (entry == null) {
						entry = new SynonymEntry();
						synonyms.put(s, entry);
					} else {
						// at least two synonym groups
						entry.isPolysemous = true;
					}
					Collection<String> without = new HashSet<String>(synGroup);
					without.remove(s);

					entry.synonyms.addAll(without);
				}
			}
		}

		input.close();
		System.out.println("Done.");
	}
}
