/*
 * 
 *  Copyright (C) 2011 Mateusz Kopec
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see http://www.gnu.org/licenses/.
 *
 */
package evaluation;

import java.util.HashSet;
import java.util.Set;

import corpusapi.tei.TEISenseInventory;

/**
 * Evaluation of single method
 * 
 * @author Mateusz Kopec
 * 
 */
public class SingleMethodEvaluation {

	/**
	 * Description of method evaluated
	 */
	public String methodId;

	// maps: lexeme -> counter of correct answers for each subsense
	private AnnotationStats correctAnnotations;

	// maps: lexeme -> counter of incorrect answers for each subsense
	private AnnotationStats incorrectAnnotations;

	/**
	 * Segment tagged as unknown sense by the method
	 */
	public int segmentsTaggedAsUnknown = 0;

	/**
	 * Constructor
	 * 
	 * @param methodName
	 *            method id
	 * @param dict
	 *            dictionary of senses
	 */
	public SingleMethodEvaluation(String methodName, TEISenseInventory dict) {
		this.methodId = methodName;
		this.correctAnnotations = new AnnotationStats(dict);
		this.incorrectAnnotations = new AnnotationStats(dict);
	}

	/**
	 * Prints statistics about evaluated method.
	 */
	public void printStats() {
		System.out.println();
		System.out.println(methodId);
		System.out.println();

		int totalCorrect = correctAnnotations.getAllCount();
		int totalIncorrect = incorrectAnnotations.getAllCount();
		double totalAccuracy = 1.0 * totalCorrect / (totalIncorrect + totalCorrect);

		System.out.println("Segments - tagged correctly      : " + totalCorrect);
		System.out.println("Segments - tagged incorrectly    : " + totalIncorrect);
		System.out.println("Segments - tagged as unknown     : " + segmentsTaggedAsUnknown);
		System.out.println();
		System.out.println("Total accuracy                 : " + totalAccuracy);
		System.out.println();
		System.out.println("Done.");
	}

	/**
	 * Returns total accuracy of the method evaluated
	 * 
	 * @return result
	 */
	public Double getTotalAccuracy() {
		int totalCorrect = correctAnnotations.getAllCount();
		int totalIncorrect = incorrectAnnotations.getAllCount();
		return 1.0 * totalCorrect / (totalIncorrect + totalCorrect);
	}

	/**
	 * Returns the improvement agains mfs baseline for given lexeme
	 * 
	 * @param goldenStandard
	 * @param lexeme
	 * @return result
	 */
	public double getLexemeImprovement(AnnotationStats goldenStandard, String lexeme) {
		double baseAcc = goldenStandard.getMfsForLexeme(lexeme);
		double methAcc = this.getLexemeAccuracy(lexeme);
		return methAcc - baseAcc;
	}

	/**
	 * Gets number of annotations for lexeme
	 * 
	 * @param lexeme
	 * @return result
	 */
	public int getLexemeCount(String lexeme) {
		int correct = correctAnnotations.getCountForLexeme(lexeme);
		int incorrect = incorrectAnnotations.getCountForLexeme(lexeme);

		return correct + incorrect;
	}

	/**
	 * Returns accuracy for lexeme if the sense distribution would be changed to
	 * equal
	 * 
	 * @param lexeme
	 * @return result
	 */
	public Double getEqualDistLexemeAccuracy(String lexeme) {
		double sum = 0;
		int count = 0;

		for (String sub : getSubsenseSet(lexeme)) {
			Double s = getSubsenseAccuracy(lexeme, sub);
			if (s == null)
				continue;

			sum += s;
			count++;
		}

		return sum / count;
	}

	/**
	 * Returns accuracy for a lexeme
	 * 
	 * @param lexeme
	 * @return result
	 */
	public Double getLexemeAccuracy(String lexeme) {
		int correct = correctAnnotations.getCountForLexeme(lexeme);
		int incorrect = incorrectAnnotations.getCountForLexeme(lexeme);

		return 1.0 * correct / (incorrect + correct);
	}

	/**
	 * Returns accuracy for a subsense
	 * 
	 * @param lexeme
	 * @param subsenseId
	 * @return result
	 */
	public Double getSubsenseAccuracy(String lexeme, String subsenseId) {
		int correct = correctAnnotations.getCountForSubsense(lexeme, subsenseId);
		int incorrect = incorrectAnnotations.getCountForSubsense(lexeme, subsenseId);

		if (correct + incorrect == 0)
			return null;
		return 1.0 * correct / (incorrect + correct);
	}

	/**
	 * Gets method description
	 * 
	 * @return description
	 */
	public String getMethodId() {
		return methodId;
	}

	/**
	 * Calculates the number of lexemes for which this method was better than
	 * mfs (and which are of given part of speech)
	 * 
	 * @param pos
	 *            part of speech
	 * @param goldenStandard
	 *            golden standard for mfs calculation
	 * @return result
	 */
	public int getNumberOfImprovements(String pos, AnnotationStats goldenStandard) {
		int better = 0;
		for (String lexeme : goldenStandard.getLexemeSet(pos))
			if (getLexemeAccuracy(lexeme) > goldenStandard.getMfsForLexeme(lexeme))
				better++;
		return better;
	}

	/**
	 * Gets RARE result of method for given part of speech
	 * 
	 * @param pos
	 *            part of speech
	 * @param goldenStandard
	 * @return result
	 */
	public double getWeightedRank(String pos, AnnotationStats goldenStandard) {
		double result = 0;

		for (String lexeme : goldenStandard.getLexemeSet(pos))
			for (String subsense : goldenStandard.getSubsensesList(lexeme)) {
				int correct = correctAnnotations.getCountForSubsense(lexeme, subsense);
				double weight = goldenStandard.getSenseWeight(lexeme, subsense);
				result += weight * correct;
			}

		return result;
	}

	/**
	 * Adds correct annotation to evaluation
	 * 
	 * @param lexeme
	 * @param senseId
	 */
	public void addCorrectAnnotation(String lexeme, String senseId) {
		correctAnnotations.addAnnotation(lexeme, senseId);
	}

	/**
	 * Adds incorrect annotation to evaluation
	 * 
	 * @param lexeme
	 * @param senseId
	 */
	public void addIncorrectAnnotation(String lexeme, String senseId) {
		incorrectAnnotations.addAnnotation(lexeme, senseId);
	}

	/**
	 * Returns set of all annotated lexemes
	 * 
	 * @return set
	 */
	public Set<String> getLexemeSet() {
		Set<String> result = new HashSet<String>(correctAnnotations.getLexemeSet());
		result.addAll(incorrectAnnotations.getLexemeSet());
		return result;
	}

	/**
	 * Returns set of annotated subsenses of given lexeme
	 * 
	 * @param lexeme
	 * @return set
	 */
	public Set<String> getSubsenseSet(String lexeme) {
		Set<String> result = new HashSet<String>(correctAnnotations.getSubsensesList(lexeme));
		result.addAll(incorrectAnnotations.getSubsensesList(lexeme));
		return result;
	}

	/**
	 * Gets total accuracy for given part of speech
	 * 
	 * @param pos
	 *            part of speech
	 * @param goldenStandard
	 * @return result
	 */
	public double getTotalAccuracy(String pos, AnnotationStats goldenStandard) {
		int correct = 0;
		int incorrect = 0;
		for (String lexeme : goldenStandard.getLexemeSet(pos)) {
			correct += correctAnnotations.getCountForLexeme(lexeme);
			incorrect += incorrectAnnotations.getCountForLexeme(lexeme);
		}
		return 1.0 * correct / (correct + incorrect);
	}
}