package dendrarium.trees.disamb;

import dendrarium.trees.AnswerType;
import dendrarium.trees.Forest;
import dendrarium.trees.NonterminalNode;

/** dezambiguator kolizyjny usprawnia proces rozstrzygania kolizji -
 * automatycznie wybiera warianty, przy których dendrolodzy byli zgodni itp. -
 * szczegóły działania w metodzie isSuggested (a jeśli zupełnie nie wiadomo
 * o co chodzi - w saturate Disambiguatora)
 * @author ks
 */
public class CollisionDisambiguator extends NKJPDisambiguator {

    private Forest left, right;
    private String leftName, rightName;
    private AnswerType leftSpecial, rightSpecial;
    
    public CollisionDisambiguator(Forest forest,
            Forest left,
            Forest right, String leftName, String rightName, AnswerType leftSpecial, AnswerType rightSpecial) {
        super(forest);
        this.left = left;
        this.right = right;
        this.leftName = leftName;
        this.rightName = rightName;
        this.leftSpecial = leftSpecial;
        this.rightSpecial = rightSpecial;
    }

    /** dany wariant jest sugerowany, jeśli co najmniej jeden z dendrologów
     * wybrał ten wariant, lub jeśli żaden z dendrologów nie wybrał dziecka dla
     * danego wierzchołka (już wcześniej poszli inną ścieżką) a wariant jest
     * sugerowany z innych powodów (np. przez NKJP)
     */
    @Override
    public boolean isSuggested(int nodeId, int childrenNo) {
        NonterminalNode l = (NonterminalNode) left.nodeById(nodeId);
        NonterminalNode r = (NonterminalNode) right.nodeById(nodeId);

        return (l.getChosenChildrenNo() != null &&
                    l.getChosenChildrenNo() == childrenNo) ||
                (r.getChosenChildrenNo() != null &&
                    r.getChosenChildrenNo() == childrenNo) ||
                (l.getChosenChildrenNo() == null &&
                    r.getChosenChildrenNo() == null &&
                    super.isSuggested(nodeId, childrenNo));
    }

    /** jeśli dany wariant został wybrany przez jakiegoś dendrologa, zaznaczamy
     * to w jego opisie, podając nazwisko tego dendrologa
     */
    @Override
    public String decorate(int nodeId, int childrenNo) {
        NonterminalNode l = (NonterminalNode) left.nodeById(nodeId);
        NonterminalNode r = (NonterminalNode) right.nodeById(nodeId);

        String ret = "";
        if (leftSpecial == AnswerType.FULL && l.getChosenChildrenNo() != null &&
                l.getChosenChildrenNo() == childrenNo)
            ret += "::" + leftName + "::    ";
        if (rightSpecial == AnswerType.FULL && r.getChosenChildrenNo() != null &&
                    r.getChosenChildrenNo() == childrenNo)
            ret += "::" + rightName + "::    ";
        return ret + super.decorate(nodeId, childrenNo);
    }

    @Override
    public String decorate(AnswerType type) {
        String ret = "";
        if (leftSpecial == type)
            ret += "::" + leftName + "::    ";
        if (rightSpecial == type)
            ret += "::" + rightName + "::    ";
        return ret + super.decorate(type);
    }

    public boolean hasTypeCollision() {
        return leftSpecial != rightSpecial;
    }

}
