package termopl;

import java.io.*;
import java.util.LinkedList;

public abstract class Parser 
{
	
	public static final int EOF     =  -1;
	public static final int EOL     =  10;
	public static final int SPACE   =  32;
	public static final int EXCLM   =  33;
	public static final int QUOTE   =  34;
	public static final int HASH    =  35;
	public static final int DOLLAR  =  36;
	public static final int LPAR    =  40;
	public static final int RPAR    =  41;
	public static final int STAR    =  42;
	public static final int PLUS    =  43;
	public static final int COMMA   =  44;
	public static final int SLASH   =  47;
	public static final int COLON   =  58;
	public static final int SCOLON  =  59;
	public static final int LANGLE  =  60;
	public static final int EQ      =  61;
	public static final int RANGLE  =  62;
	public static final int QMARK   =  63;
	public static final int LBRCT   =  91;
	public static final int BSLASH  =  92;
	public static final int RBRCT   =  93;
	public static final int CIRCMFX =  94;
	public static final int VBAR    = 124;
	public static final int TILDE   = 126;

	public enum Symbol {
		EMPTY, IDENTIFIER, STRING, NUMBER, HASH, DOLLAR, CIRCMFX, 
		QMARK, STAR, PLUS, OR, COMMA, COLON, SCOLON, EQ, NEQ, MATCH, 
		NMATCH, LPAR, RPAR, LBRCT, RBRCT, LANGLE, RANGLE, UNKNOWN
	};

	private   Object owner;
	protected int lineCount;
	protected int charCount;
	protected int scanPos;
	protected char[] line;
	protected boolean loadNewLine;
	protected Symbol symbol;
	protected String identifier;
	protected String string;
	protected BufferedReader reader;
	protected StringBuffer buffer;
	protected LinkedList<ParserError> errors;

	public abstract void parse();
	public abstract void getSymbol() throws ParserException;
	
	public Parser(Object owner)
	{
		string = null;
		buffer = new StringBuffer();
		errors = new LinkedList<ParserError>();
		line = null;
		lineCount = 0;
		charCount = 0;
		scanPos = 0;
		loadNewLine = true;
		this.owner = owner;
	}
	
	public int getNextChar() throws ParserException
	{
		try {
			if (loadNewLine) {
				String l = reader.readLine();
				
				if (l == null) return EOF;
				else if (l.isEmpty()) return EOL;
				else if (l.startsWith("#")) return EOL;
				else {
					line = l.toCharArray();
					lineCount++; charCount = 0;
					loadNewLine = false;
				}
			}
			else if (line.length <= charCount) {
				loadNewLine = true;
				return EOL;
			}
			
			int ch = line[charCount++];
			
			if (ch == SLASH) {
				int nch = line[charCount];
				if (nch == SLASH) {
					String l = reader.readLine();
					
					if (l == null) return EOF;
					else {
						line = l.toCharArray();
						lineCount++; charCount = 0;
						return EOL;
					}
				}
			}
			return ch;
		}
		catch (IOException exception) {
			error(exception.getMessage());
		}
		return EOF;
	}
	
	public void getString() throws ParserException
	{
		int ch;
		
		buffer.setLength(0);
		do {
			ch = getNextChar();
			if (ch != QUOTE) buffer.append((char)ch);
			if (ch == BSLASH) {
				ch = getNextChar();
				if (ch == QUOTE) buffer.setCharAt(buffer.length() - 1, (char)ch);
				else if (ch == EOL) {
					buffer.setLength(buffer.length() - 1);
					ch = SPACE;
				}
				else buffer.append((char)ch);
			}
		} while (ch != QUOTE && ch != EOL && ch != EOF);
		if (ch == QUOTE) {
			string = buffer.toString();
			symbol = Symbol.STRING;
		}
		else {
			string = null;
			symbol = Symbol.UNKNOWN;
			error(ParserError.END_OF_STRING_EXPECTED, line, lineCount, charCount);
			skip(Symbol.RBRCT);
		}
	}
	
	public boolean isLetter(int ch)
	{
		return (ch >= 65 && ch <= 90) || (ch >= 97 && ch <= 122);
	}
	
	public boolean isDigit(int ch)
	{
		return ch >= 48 && ch <= 57;
	}
	
	public boolean isIdentifierChar(int ch)
	{
		return isLetter(ch) || isDigit(ch) || ch == '_' || ch == '-';
	}
	
	public void skip(Symbol... list) throws ParserException
	{
		while (!belongs(symbol, list) && symbol != Symbol.EMPTY) getSymbol();
	}
	
	public boolean belongs(Symbol symbol, Symbol[] list)
	{
		for (Symbol s : list) {
			if (s == symbol) return true;
		}
		return false;
	}
	
	public void error(int err, char[] line, int lineCount, int scanPos) throws ParserException
	{
		if (symbol == Symbol.EMPTY) {
			errors.add(new ParserError(ParserError.ERR_EOF));
			throw new ParserException();
		}
		else {
			for (ParserError e : errors) {
				if (e.lineNo == lineCount && e.pos == scanPos) return;
			}
			errors.add(new ParserError(err, line, lineCount, scanPos));
		}
	}
	
	public void error(String msg) throws ParserException
	{
		errors.add(new ParserError(msg));
		throw new ParserException();
	}
	
	public void error(int err) throws ParserException
	{
		error(err, line, lineCount, scanPos);
	}
	
	public boolean noErrors()
	{
		return errors.size() == 0;
	}
	
	public boolean tooManyErrors()
	{
		int n = errors.size();
		
		if (n >= 10) {
			if (n == 10) errors.add(new ParserError(ParserError.TOO_MANY_ERRORS));
			return true;
		}
		return false;
	}
	
	public void reportErrors()
	{
		if (!noErrors()) {
			ErrorViewer viewer = null;
			
			if (owner != null) {
				viewer = new ErrorViewer(owner);
			}
			for (ParserError err : errors) {
				if (owner == null) err.print();
				else viewer.showError(err.toString());
			}
			if (owner != null) viewer.setVisible(true);
		}
	}

}
