package termopl;

import java.awt.*;
import java.io.*;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.util.*;

@SuppressWarnings("serial")
public class Preferences implements Serializable, Cloneable
{

	public static final int MIN_FONT_SIZE = 11;
	public static final int MAX_FONT_SIZE = 15;
	public static final int[] DEFAULT_SORT_PREFS = { 1, 1, 1, -1, -1, -1, -1, -1, -1 };
	public static final String DEFAULT_WORKSPACE = 
		System.getProperty("user.home") + File.separator + "TermoPLWorkspace";
	public int sortedColumn;
	public int maxResults;
	public int minLength;
	public int maxLength;
	public int contrastiveRankingMethod;
	public int NPMIMethod;
	public int NPMIfactor;
	public int detRatio;
	public int cntxMethod;
	public int detHandling;
	public int minfrq;
	public int fontSize;
	public int[] sortPrefs;
	public double mincvalue;
	public double alpha;
	public double beta;
	public boolean reuseTaggedFiles;
	public boolean trimResults;
	public boolean filterResults;
	public boolean multiWordTermsOnly;
	public boolean checkStopWords;
	public boolean makeGroups;
	public boolean useWordNet;
	public boolean mergeSW;
	public boolean removeCompoundPreps;
	public boolean mergeCP;
	public boolean removeCommonTerms;
	public boolean mergeCT;
	public boolean showSimplifiedForms;
	public boolean useUD;
	public boolean useCustomTagset;
	public boolean useCustomGrammar;
	public boolean calculateBaseForms;
	public boolean collectAllForms;
	public boolean makeIndex;
	public boolean ignoreCase;
	public boolean useNPMIMethod;
	public boolean trimFromLeftToRight;
	public boolean detectDeterminers;
	public boolean applyContrastiveRanking;
	public boolean applyContrastiveRankingForTopRankedTerms;
	public boolean applyContrastiveRankingForFrequentTerms;
	public boolean useCValues;
	public boolean saveCount;
	public boolean saveRank;
	public boolean saveSF;
	public boolean saveBF;
	public boolean saveCV;
	public boolean saveComp;
	public boolean saveLen;
	public boolean saveFreqs;
	public boolean saveFreqin;
	public boolean saveContext;
	public boolean extract;
	public boolean compare;
	public boolean reloadContrastiveTerms;
	public boolean caseSensitive;
	public boolean entireWord;
	public boolean regEx;
	public String language;
	public String workSpace;
	public String filePath;
	public String tagsetPath;
	public String grammarPath;
	public String pythonPath;
	public String wordNetPath;
	public String contrastiveDataPath;
	public LinkedList<String> stopWords;
	public LinkedList<String> compPreps;
	public LinkedList<CommonTerm> commonTerms;
	public LinkedList<String> recentFiles;
	public HashMap<Integer, Rectangle> sizeAndLocation;
	
	public transient boolean modified;
	public transient boolean repaint;
	public transient boolean recalculate;
	public transient Tagset tagset;
	public transient Template template;
	public transient Template compoundPrepositions;
	public transient Font plainFont;
	public transient Font boldFont;
	public transient Set<String> stopWordSet;
	public transient Set<String> commonTermSet;
	
	public Preferences()
	{
		modified = false;
	}

	public Object clone()
	{
		try {
			Preferences cloned = (Preferences)super.clone();
			
			return cloned;
		}
		catch (CloneNotSupportedException exception) {
			return null;
		}
	}
	
	public void setDefaults()
	{
		sortedColumn = 1;
		maxResults = 1000;
		minLength = 2;
		maxLength = -1;
		contrastiveRankingMethod = 1;
		NPMIMethod = 3;
		NPMIfactor = 120;
		detRatio = 20;
		cntxMethod = 1;
		detHandling = 1;
		minfrq = 1;
		fontSize = MIN_FONT_SIZE;
		sortPrefs = new int[DEFAULT_SORT_PREFS.length];
		System.arraycopy(DEFAULT_SORT_PREFS, 0, sortPrefs, 0, DEFAULT_SORT_PREFS.length);
		mincvalue = 0.0;
		alpha = 0.9;
		beta = 0.3;
		reuseTaggedFiles = true;
		trimResults = false;
		filterResults = false;
		multiWordTermsOnly = false;
		checkStopWords = true;
		makeGroups = true;
		mergeSW = false;
		removeCompoundPreps = true;
		mergeCP = false;
		removeCommonTerms = false;
		mergeCT = false;
		showSimplifiedForms = false;
		useUD = false;
		useCustomTagset = false;
		useCustomGrammar = false;
		calculateBaseForms = true;
		collectAllForms = false;
		makeIndex = false;
		ignoreCase = true;
		useNPMIMethod = true;
		trimFromLeftToRight = false;
		detectDeterminers = true;
		applyContrastiveRanking = false;
		applyContrastiveRankingForTopRankedTerms = true;
		applyContrastiveRankingForFrequentTerms = true;
		useCValues = true;
		saveCount = true;
		saveRank = true;
		saveSF = true;
		saveBF = true;
		saveCV = true;
		saveComp = true;
		saveLen = true;
		saveFreqs = true;
		saveFreqin = true;
		saveContext = true;
		repaint = false;
		recalculate = false;
		useWordNet = false;
		language = "pl";
		workSpace = DEFAULT_WORKSPACE;
		filePath = DEFAULT_WORKSPACE;
		tagsetPath = null;
		grammarPath = null;
		pythonPath = null;
		wordNetPath = null;
		contrastiveDataPath = null;
		stopWords = null;
		compPreps = null;
		reloadContrastiveTerms = false;
		caseSensitive = false;
		entireWord = true;
		regEx = false;
		commonTerms = null;
		tagset = null;
		template = null;
		stopWordSet = null;
		commonTermSet = null;
		compoundPrepositions = null;
		extract = false;
		compare = false;
		recentFiles = new LinkedList<String>();
		sizeAndLocation = new HashMap<Integer, Rectangle>();
		createFonts();
		setModified(true);
	}
	
	public static Preferences createPreferences(boolean finalize)
	{
		String path = System.getProperty("user.home") + File.separator + ".TermoPL-8";
		File file = new File(path);
		ObjectInputStream input;
		Preferences preferences;
		
		try {
			input = new ObjectInputStream(new FileInputStream(file));
			preferences = (Preferences)input.readObject();
			input.close();
		}
		catch (Exception exception) {
			preferences = new Preferences();
			preferences.setDefaults();
		}
		preferences.finalizeCreate(finalize);
		return preferences;
	}
	
	public void finalizeCreate(boolean finalize)
	{
		if (finalize) {
			File ws = new File(workSpace);
			File f;
			String path;
			
			if (!ws.exists()) {
				if (!workSpace.equals(DEFAULT_WORKSPACE)) workSpace = DEFAULT_WORKSPACE;
				ws = new File(workSpace);
				ws.mkdirs();
			}
			createFonts();
			if (filePath != null) {
				f = new File(filePath);
				if (!f.exists()) filePath = workSpace;
			}
			if (contrastiveDataPath != null) {
				path = resolvePath(contrastiveDataPath);
				f = new File(path);
				if (f.exists()) contrastiveDataPath = path;
				else contrastiveDataPath = null;
			}
			if (tagsetPath != null) {
				path = resolvePath(tagsetPath);
				f = new File(path);
				if (f.exists()) tagsetPath = path;
				else tagsetPath = null;
			}
			if (useCustomTagset) readCustomTagset(TermoPL.dialogOwner);
			else tagset = Tagset.createDefaultTagset();
			if (grammarPath != null) {
				path = resolvePath(grammarPath);
				f = new File(path);
				if (f.exists()) grammarPath = path;
				else grammarPath = null;
			}
			if (useCustomGrammar) readCustomGrammar(TermoPL.dialogOwner);
			else template = Template.NPP();
			if (checkStopWords) readStopWords(TermoPL.dialogOwner);
			if (removeCompoundPreps) readCompoundPreps(TermoPL.dialogOwner);
			else compoundPrepositions = null;
		}
	}
	
	public void save() 
	{
		String path = System.getProperty("user.home") + File.separator + ".TermoPL-8";
		
		save(new File(path));
	}
	
	public void save(File file)
	{
		try {
			ObjectOutputStream output = new ObjectOutputStream(new FileOutputStream(file));
			
			save(output);
			output.flush();
			output.close();
		}
		catch (Exception exception) {
		}
	}
	
	public void save(ObjectOutputStream output)
	{
		String tp = tagsetPath;
		String gp = grammarPath;
		String dp = contrastiveDataPath;

		if (tagsetPath != null)
			tagsetPath = getRelativizedPath(tagsetPath);
		if (grammarPath != null)
			grammarPath = getRelativizedPath(grammarPath);
		if (contrastiveDataPath != null)
			contrastiveDataPath = getRelativizedPath(workSpace, contrastiveDataPath, "WORKSPACE");
		try {
			output.writeObject(this);
		}
		catch (Exception exception) {
		}
		tagsetPath = tp;
		grammarPath = gp;
		contrastiveDataPath = dp;
	}
	
	public Tagset getTagset()
	{
		return tagset;
	}
	
	public void readCustomTagset(Component owner)
	{
		if (tagsetPath != null) {
			File tfile = new File(tagsetPath);
			
			if (tfile.exists()) {
				TParser parser = new TParser(tfile, owner);
				Tagset tset;
				
				parser.parse();
				tset = parser.getTagset();
				if (tset != null) tagset = tset;
				else {
					tagsetPath = null;
					useCustomTagset = false;
					tagset = Tagset.createDefaultTagset();
				}
			}
			else {
				tagsetPath = null;
				useCustomTagset = false;
				tagset = Tagset.createDefaultTagset();
			}
		}
		else {
			useCustomTagset = false;
			tagset = Tagset.createDefaultTagset();
		}
	}
	
	public void readCustomGrammar(Component owner)
	{
		if (grammarPath != null) {
			File gfile = new File(grammarPath);
			
			if (gfile.exists()) {
				GParser parser = new GParser(gfile, tagset, owner);
				Template tmp;
				
				parser.parse();
				tmp = parser.getTemplate();
				if (tmp != null) template = tmp;
				else {
					grammarPath = null;
					tagsetPath = null;
					useCustomGrammar = false;
					useCustomTagset = false;
					template = Template.NPP();
				}
			}
			else {
				grammarPath = null;
				tagsetPath = null;
				useCustomGrammar = false;
				useCustomTagset = false;
				template = Template.NPP();
			}
		}
		else {
			useCustomGrammar = false;
			useCustomTagset = false;
			template = Template.NPP();
		}
	}
	
	public void readStopWords(Component owner)
	{
		LinkedList<String> sw = stopWords;
		
		if (sw == null) {
			String path = System.getProperty("user.dir") + File.separator + "termopl_sw.txt";
			File file = new File(path);
			
			if (file.exists()) {
				try {
					BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF8"));
					String line;
					
					do {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							if (!line.isEmpty()) {
								if (sw == null) sw = new LinkedList<String>();
								if (!sw.contains(line)) sw.add(line);
							}
						}
					} while (line != null);
					reader.close();
					stopWords = sw;
				}
				catch (IOException exception) {
					sw = null;
				}
			}
		}
	}
	
	public void readCompoundPreps(Component owner)
	{
		LinkedList<String> cp = compPreps;
		
		if (cp == null) {
			String path = System.getProperty("user.dir") + File.separator + "termopl_cp.txt";
			File file = new File(path);
			
			if (file.exists()) {
				try {
					BufferedReader reader = new BufferedReader(new InputStreamReader(new FileInputStream(file), "UTF8"));
					String line;
					
					do {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							if (!line.isEmpty()) {
								if (cp == null) cp = new LinkedList<String>();
								if (!cp.contains(line)) cp.add(line);
							}
						}
					} while (line != null);
					reader.close();
				}
				catch (IOException exception) {
					cp = null;
				}
			}
		}
		if (cp != null) {
			PParser parser = new PParser(cp, owner);
			Template tmp;
			
			parser.parse();
			tmp = parser.getTemplate();
			if (tmp != null) {
				compoundPrepositions = tmp;
				compPreps = cp;
			}
			else compoundPrepositions = null;
		}
		else compoundPrepositions = null;
	}
	
	public void setModified(boolean f)
	{
		modified = f;
	}
	
	public boolean isModified()
	{
		return modified;
	}
	
	public Rectangle getBounds(int windowID)
	{
		Rectangle rect = sizeAndLocation.get(windowID);
		
		if (rect == null) rect = new Rectangle();
		return rect;
	}
	
	public void setBounds(int windowID, Rectangle bounds)
	{
		sizeAndLocation.put(windowID, bounds);
		setModified(true);
	}
	
	public void createFonts()
	{
		plainFont = new Font(Font.SANS_SERIF, Font.PLAIN, fontSize);
		boldFont = new Font(Font.SANS_SERIF, Font.BOLD, fontSize);
	}
	
	public String resolvePath(String path)
	{
		int index = path.lastIndexOf('+');
		String p;
		
		if (index < 0) return path;
		p = path.substring(index + 1);
		
		if (path.startsWith("WORKSPACE+")) p = workSpace + File.separator + p;
		else if (path.startsWith("APPLOC+")) p = TermoPL.appLocation + File.separator + p;
		return p;
	}
	
	public String resolveFilePath(String path)
	{
		File file = new File(path);
		String filePath = null;
		
		if (file.getAbsolutePath().equals(path)) {
			if (file.exists()) filePath = path;
		}
		else {
			filePath = workSpace + File.separator + path;
			file = new File(filePath);
			if (!file.exists()) {
				filePath = TermoPL.appLocation + File.separator + path;
				file = new File(filePath);
				if (!file.exists()) filePath = null;
			}
		}
		return filePath;
	}
	
	public static String resolveFilePath(String workSpace, String path)
	{
		File file = new File(path);
		String fPath = file.getAbsolutePath(); 
		
		if (path.equals(fPath) || workSpace == null) return fPath;
		return workSpace + File.separator + path;
	}
	
	public String getRelativizedPath(String path)
	{
		Path wpath = FileSystems.getDefault().getPath(workSpace);
		Path ppath = FileSystems.getDefault().getPath(path);
		
		if (wpath.getRoot().compareTo(ppath.getRoot()) == 0)
			return "WORKSPACE+" + wpath.relativize(ppath).toString();
		wpath = FileSystems.getDefault().getPath(TermoPL.appLocation);
		if (wpath.getRoot().compareTo(ppath.getRoot()) == 0)
			return "APPLOC+" + wpath.relativize(ppath).toString();
		return "ABSOLUTE+" + path;
	}
	
	public String getRelativizedPath(String workSpace, String path, String label)
	{
		String p;

		if (workSpace.isEmpty()) p = "ABSOLUTE+" + path;
		else {
			Path wpath = FileSystems.getDefault().getPath(workSpace);
			Path ppath = FileSystems.getDefault().getPath(path);
			
			if (wpath.getRoot().compareTo(ppath.getRoot()) == 0)
				p = label + "+" + wpath.relativize(ppath).toString();
			else p = "ABSOLUTE+" + path;
		}
		return p;
	}
	
}