package termopl;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.util.*;
import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.*;

@SuppressWarnings("serial")
public class SelectedFiles extends JPanel 
{

	public static final int OK = 1;
	public static final String[] HEADERS = { "#", "File Path" };
	
	private TermoPLDocument doc;
	private File[] files;
	private File[] oldFiles;
	private File[] newFiles;
	private JLabel missingFiles;
	private JTextField corpusName;
	private JTable table;
	private JScrollPane sp;
	private PictButton addButton;
	private PictButton removeButton;
	private JCheckBox reuseBox;
	private JDialog dialog = null;
	private int numOfOldFiles = 0;
	private int numOfNewFiles = 0;
	private int answer;
	
	public SelectedFiles(TermoPLDocument doc)
	{
		super(new BorderLayout());
		
		this.doc = doc;
		oldFiles = doc.getOldFiles();
		newFiles = doc.getNewFiles();
		
		int s;
		
		if (oldFiles != null) numOfOldFiles = oldFiles.length;
		if (newFiles != null) numOfNewFiles = newFiles.length;
		files = new File[numOfOldFiles + numOfNewFiles];
		s = 0;
		if (oldFiles != null) {
			for (int i = 0; i < numOfOldFiles; i++) {
				files[s++] = oldFiles[i];
			}
		}
		if (newFiles != null) {
			for (int i = 0; i < numOfNewFiles; i++) {
				files[s++] = newFiles[i];
			}
		}
		table = new JTable();
		table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
		setBorder(BorderFactory.createEmptyBorder(16, 16, 16, 16));
		arrangeComponents();
		initTable();
		if (numOfOldFiles + numOfNewFiles == 0) addNewFiles();
	}
	
	public void arrangeComponents()
	{
		JButton acceptButton = new JButton("Accept");
		JButton cancelButton = new JButton("Cancel");
		JButton workspaceButton = new JButton("Workspace");
		JPanel topPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
		Box vbox = Box.createVerticalBox();
		Box hbox = Box.createHorizontalBox();
		
		reuseBox = new JCheckBox("Reuse already tagged text files.");
		reuseBox.setSelected(doc.getPreferences().reuseTaggedFiles);
		vbox.add(Box.createVerticalStrut(2));
		addButton = new PictButton("plus.png", true);
		addButton.setBorder(BorderFactory.createLineBorder(Color.lightGray));
		hbox.add(addButton);
		removeButton = new PictButton("minus.png", true);
		removeButton.setEnabled(false);
		removeButton.setBorder(BorderFactory.createLineBorder(Color.lightGray));
		hbox.add(removeButton);
		hbox.add(Box.createHorizontalGlue());
		missingFiles = new JLabel("There are missing file(s)!");
		missingFiles.setForeground(Color.red);
		hbox.add(missingFiles);
		vbox.add(hbox);
		hbox = Box.createHorizontalBox();
		hbox.add(reuseBox);
		hbox.add(Box.createHorizontalStrut(8));
		hbox.add(Box.createHorizontalGlue());
		hbox.add(workspaceButton);
		hbox.add(Box.createHorizontalStrut(4));
		hbox.add(cancelButton);
		hbox.add(Box.createHorizontalStrut(4));
		hbox.add(acceptButton);
		vbox.add(hbox);
		
		sp = new JScrollPane(table);
		sp.setBorder(BorderFactory.createLineBorder(Color.lightGray));
		
		topPanel.add(new JLabel("Corpus name: "));
		corpusName = new JTextField(20);
		if (doc.getCorpusName() != null) corpusName.setText(doc.getCorpusName());
		topPanel.add(corpusName);
		
		add(topPanel, BorderLayout.NORTH);
		add(sp, BorderLayout.CENTER);
		add(vbox, BorderLayout.SOUTH);
		
		addButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) 
			{
				addNewFiles();
			}
		});
		removeButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) 
			{
				removeFiles();
			}
		});
		acceptButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) 
			{
				if (isFileNameValid(corpusName.getText())) {
					answer = OK;
					dialog.setVisible(false);
				}
				else {
					JOptionPane.showMessageDialog(dialog, "Enter a valid corpus name.", "Error", JOptionPane.ERROR_MESSAGE);
					corpusName.selectAll();
					corpusName.requestFocus();
				}
			}
		});
		cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) 
			{
				dialog.setVisible(false);
			}
		});
		workspaceButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) 
			{
				doc.selectWorkspace();
				((AbstractTableModel)table.getModel()).fireTableDataChanged();
			}
		});
		showMissingFiles();
	}
	
	public void initTable()
	{
		FontMetrics fm = getFontMetrics(TermoPL.preferences.boldFont);
		
		table.setModel(new SFTableModel());
		table.setBorder(BorderFactory.createCompoundBorder(
			BorderFactory.createEmptyBorder(-1, -1 , 0, -1),
			BorderFactory.createLineBorder(Color.lightGray)));
		table.setBackground(CommonResources.LIGHT_GRAY);
		table.setGridColor(Color.lightGray);
		table.setShowHorizontalLines(false);
		table.setShowVerticalLines(true);
		table.setRowHeight(fm.getHeight() + 4);
		
		TableColumnModel cmodel = table.getColumnModel();
		TableCellRenderer cellRenderer = new RowRenderer();
		TableHeader tableHeader = new TableHeader(cmodel);
		
		table.setTableHeader(tableHeader);
		for (int i = 0; i < cmodel.getColumnCount(); i++) {
			TableColumn column = cmodel.getColumn(i);
			
			column.setHeaderValue(HEADERS[i]);
			column.setCellRenderer(cellRenderer);
			if (i == 0) {
				column.setPreferredWidth(40);
				column.setMinWidth(40);
				column.setMaxWidth(40);
			}
			else column.setPreferredWidth(sp.getViewport().getPreferredSize().width - 20);
		}
		table.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent event) 
			{
				int row = table.getSelectedRow();
				
				removeButton.setEnabled(row >= 0);
			}
		});
	}
	
	public void showMissingFiles()
	{
		for (File f : files) {
			if (!f.exists()) {
				missingFiles.setVisible(true);
				return;
			}
		}
		missingFiles.setVisible(false);
	}
	
	public void addNewFiles()
	{
		File[] newList = doc.selectSourceFiles();
		
		if (newList != null) {
			LinkedList<File> list = new LinkedList<File>();
			
			for (File f : files) list.add(f);
			for (File f : newList) {
				String fpath = f.getPath();
				boolean found = false;
				
				for (File e : list) {
					String epath = e.getPath();
					
					if (epath.equals(fpath)) {
						found = true;
						break;
					}
				}
				if (!found) list.add(f);
			}
			if (list.size() > files.length) {
				files = list.toArray(files);
				((AbstractTableModel)table.getModel()).fireTableDataChanged();
				if (corpusName.getText().equals("")) {
					String str = TermoPL.getFileName(files[0]);
					
					corpusName.setText(str);
				}
			}
		}
	}
	
	public void removeFiles()
	{
		int[] selected = table.getSelectedRows();
		File[] list = new File[files.length - selected.length];
		
		for (int i = 0, j = 0, n = 0; i < files.length; i++) {
			if (n < selected.length) {
				if (i == selected[n]) {
					if (i < numOfOldFiles) numOfOldFiles--;
					else numOfNewFiles--;
					n++;
				}
				else list[j++] = files[i];
			}
			else list[j++] = files[i];
		}
		files = list;
		((AbstractTableModel)table.getModel()).fireTableDataChanged();
	}
	
	public boolean isFileNameValid(String str)
	{
		File f = new File(str);
		
		try {
			if (f.getCanonicalFile().getName().equals(str)) return true;
			return false;
		}
		catch (IOException e) {
			return false;
		}
	}
	
	public File[] getFiles()
	{
		return files;
	}
	
	public File[] getNewFiles()
	{
		LinkedList<File> newf = new LinkedList<File>();
		
		for (File f1 : files) {
			boolean found = false;
			
			if (oldFiles != null) {
				for (File f2 : oldFiles) {
					if (f1.getAbsolutePath().equals(f2.getAbsolutePath())) {
						found = true;
						break;
					}
				}
			}
			if (!found) newf.add(f1);
		}
		if (newf.isEmpty()) return null;
		return newf.toArray(new File[0]);
	}
	
	public File[] getOldFiles()
	{
		LinkedList<File> oldf = new LinkedList<File>();
		
		for (File f1 : files) {
			if (oldFiles != null) {
				for (File f2 : oldFiles) {
					if (f1.getAbsolutePath().equals(f2.getAbsolutePath())) {
						oldf.add(f1);
						break;
					}
				}
			}
		}
		if (oldf.isEmpty()) return null;
		return oldf.toArray(new File[0]);
		
	}
	
	public String getCorpusName()
	{
		return corpusName.getText();
	}
	
	// 0 -- rebuilding is not necessary
	// 1 -- rebuilding is necessary, only new files must be processed
	// 2 -- rebuilding is necessary, all files must be processed
	// 3 -- rebuilding is impossible; files must be selected
	public int rebuild()
	{
		File[] oldf = getOldFiles();
		File[] newf = getNewFiles();
		File[] selectedFiles = doc.getSelectedFiles();
		int len1, len2;
		
		if (oldf == null) len1 = 0;
		else len1 = oldf.length;
		if (newf == null) len2 = 0;
		else len2 = newf.length;
		
		if (selectedFiles != null) {
			boolean subset = true;
			
			for (File f1 : selectedFiles) {
				boolean found = false;
				
				if (oldf != null) {
					for (File f2 : oldf) {
						if (f1.getAbsolutePath().equals(f2.getAbsolutePath())) {
							found = true;
							break;
						}
					}
				}
				if (!found && newf != null) {
					for (File f2 : newf) {
						if (f1.getAbsolutePath().equals(f2.getAbsolutePath())) {
							found = true;
							break;
						}
					}
				}
				if (!found) {
					subset = false;
					break;
				}
			}
			if (subset) {
				if (selectedFiles.length == (len1 + len2)) return 0;
				return 1;
			}
			return 2;
		}
		else {
			if (oldf != null || newf != null) return 2;
		}
		return 3;
	}
	
	public boolean reuse()
	{
		return reuseBox.isSelected();
	}
	
	public void dispose()
	{
		if (dialog != null) dialog.dispose();
	}
	
	public int doDialog()
	{
		answer = -1;
		dialog = new JDialog(TermoPL.dialogOwner);
		dialog.setModal(true);
		dialog.setDefaultCloseOperation(JDialog.HIDE_ON_CLOSE);
		dialog.setTitle("Selected Files");
		dialog.setResizable(true);
		dialog.getContentPane().add(this);
		dialog.pack();
		dialog.setLocationRelativeTo(TermoPL.dialogOwner);
		dialog.setVisible(true);
		return answer;
	}
	
	private class SFTableModel extends AbstractTableModel
	{
		
		public int getColumnCount()
		{
			return 2;
		}
		
		public int getRowCount()
		{
			if (files == null) return 0;
			return files.length;
		}
		
		public boolean isCellEditable(int row, int column)
		{
			return false;
		}
		
		public Object getValueAt(int row, int column)
		{
			Path workspace = FileSystems.getDefault().getPath(doc.getPreferences().workSpace);
			
			if (files == null) return 0;
			if (column == 0) return row + 1;
			
			Path p = FileSystems.getDefault().getPath(files[row].getAbsolutePath());
			
			if (p.startsWith(workspace)) return workspace.relativize(p).toString();
			return p.toString();
		}
		
	}
	
	private class TableHeader extends JTableHeader
	{
		
		public TableHeader(TableColumnModel model)
		{
			super(model);
			setDefaultRenderer(new TableHeaderRenderer());
			setReorderingAllowed(false);
		}
		
		public Dimension getPreferredSize()
		{
			Dimension dim = super.getPreferredSize();
			FontMetrics fm = getFontMetrics(TermoPL.preferences.boldFont);
			
			dim.height = fm.getHeight() + 4;
			return dim;
		}
		
	}

	private class TableHeaderRenderer implements TableCellRenderer
	{
		
		private JLabel renderer;
		
		public TableHeaderRenderer()
		{
			super();
			initRenderer();
		}
		
		public void initRenderer()
		{
			Border headerBorder = UIManager.getBorder("TableHeader.cellBorder");
			
			renderer = new JLabel();
			renderer.setFont(TermoPL.preferences.boldFont);
			renderer.setBorder(headerBorder);
			renderer.setBackground(UIManager.getColor("TableHeader.background"));
			renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
		}
		
		public Component getTableCellRendererComponent(JTable table, Object value,
				boolean isSelected, boolean hasFocus, int row, int column)
		{
			renderer.setText((String)value);
			if (column == 0) renderer.setHorizontalAlignment(JLabel.CENTER);
			else renderer.setHorizontalAlignment(JLabel.LEFT);
			return renderer;
		}

	}
	
	private class RowRenderer extends DefaultTableCellRenderer
	{
		
		private final Border border = BorderFactory.createEmptyBorder(0, 4, 0, 4);
		private final ImageIcon missingIcon = new ImageIcon(ClassLoader.getSystemClassLoader().getResource("deleted.png"));
		private final ImageIcon warningIcon = new ImageIcon(ClassLoader.getSystemClassLoader().getResource("warning-small.png"));
		
		public RowRenderer()
		{
			super();
		}
		
		public Component getTableCellRendererComponent(JTable table, Object value,
				boolean isSelected, boolean hasFocus, int row, int column)
		{
			setForeground(Color.black);
			if (isSelected) {
				setForeground(UIManager.getColor("Table.selectionForeground"));
				setBackground(UIManager.getColor("Table.selectionBackground"));
			}
			else {
				setForeground(Color.black);
				if (row < numOfOldFiles) setBackground(CommonResources.PALE_ORANGE);
				else setBackground(Color.white);
			}
			setValue(value);
			setFont(TermoPL.preferences.plainFont);
			setBorder(border);
			if (column == 0) {
				setHorizontalAlignment(JLabel.CENTER);
				if (files[row].exists()) {
					Path workspace = FileSystems.getDefault().getPath(doc.getPreferences().workSpace);
					Path p = FileSystems.getDefault().getPath(files[row].getPath());
					
					if (p.startsWith(workspace)) setIcon(null);
					else setIcon(warningIcon);
				}
				else setIcon(missingIcon);
			}
			else {
				setHorizontalAlignment(JLabel.LEFT);
				setIcon(null);
			}
			return this;
		}
		
	}
	
}
