package termopl;

import java.awt.*;
import java.awt.event.*;
import java.net.*;
import java.text.*;
import java.util.regex.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.table.*;

@SuppressWarnings("serial")
public class TermsView extends JPanel 
{

	public static final String SELECT_WARNING = "No file selected for term extraction.";
	public static final String EXTRACT_WARNING = "Extract terms.";
	public static final String COMPARE_WARNING = "Compare terms.";
	public static final String CANCEL_WARNING = "Cancelled.";
	public static final String SEARCH_FAILED = "No terms found.";
	
	private static final String[] HEADERS = { "#", "Rank", "Term", "C-value", "?", "Length", "Freq_s", "Freq_in", "Context #" };
	private static final String[] CMETHODS = { "LL", "TFITF", "CSmw", "TW" };
	private static final int[] SIZES = { 70, 70, 360, 100, 80, 100, 100, 100, 100 };
	
	private TermoPLDocument doc;
	private JScrollPane tableScrollPane;
	private TermTable table;
	private CntrlPanel1 cntrlPanel1;
	private CntrlPanel2 cntrlPanel2;
	private InfoPanel infoPanel;
	private Component bottomComponent;
	private Term selectedTerm;
	private String searchString;
	private Runnable scrollToSelectedRow;
	private int selectedTermIndex;
	private int matchingTermIndex;
	private int matchingTermCounter;
	private int numOfMatchingTerms;
	private boolean patternError;
	private boolean columnsAreVisible;
	
	public TermsView(TermoPLDocument doc)
	{
		this.doc = doc;
		searchString = "";
		selectedTerm = null;
		selectedTermIndex = -1;
		matchingTermIndex = -1;
		matchingTermCounter = 0;
		patternError = false;
		numOfMatchingTerms = 0;
		setLayout(new BorderLayout());
		arrangeComponents();
		if (doc.getTerms() != null) table.showColumns(true);
		setTableDimensions();
		scrollToSelectedRow = new Runnable() {
			public void run() 
			{
				int row = table.getSelectedRow();
				
				table.scrollRectToVisible(table.getCellRect(row,  0,  true));
			}
		};
	}
	
	public void arrangeComponents()
	{
		PlaceHolder ph = new PlaceHolder();
		
		table = new TermTable(ph);
		tableScrollPane = new JScrollPane(table);
		tableScrollPane.getViewport().setBackground(CommonResources.LIGHT_GRAY);
		tableScrollPane.setCorner(JScrollPane.UPPER_RIGHT_CORNER, ph);
		tableScrollPane.setBorder(BorderFactory.createCompoundBorder(
			BorderFactory.createEmptyBorder(-1, -1, 0, -1),
			BorderFactory.createLineBorder(Color.lightGray)));
		add(tableScrollPane, BorderLayout.CENTER);
		cntrlPanel2 = new CntrlPanel2();
		infoPanel = new InfoPanel();
		add(infoPanel, BorderLayout.SOUTH);
		bottomComponent = infoPanel;
		if (doc.getSelectedFiles() == null && doc.getNewFiles() == null) {
			infoPanel.setWarning(SELECT_WARNING);
			infoPanel.setAccessory(doc.selectAction);
		}
		if (doc.isLoaded()) finishCreate();
	}
	
	public void finishCreate()
	{
		cntrlPanel1 = new CntrlPanel1();
		add(cntrlPanel1, BorderLayout.NORTH);
		revalidate();
	}
	
	public void setTableDimensions()
	{
		int s = 0;
		
		for (int i = 0; i < SIZES.length; i++) s += SIZES[i];
		if (!doc.getPreferences().applyContrastiveRanking) s -= SIZES[TermoPLConsts.CSVALUE];
		s += tableScrollPane.getVerticalScrollBar().getPreferredSize().width;
		table.setPreferredScrollableViewportSize(new Dimension(s, 800));
	}
	
	public void resetResults()
	{
		table.resetResults();
	}
	
	public void resetSearch()
	{
		setNumOfMatches();
		if (numOfMatchingTerms > 0 && !doc.getPreferences().filterResults) {
			findNext();
		}
	}
	
	public void clearSearch()
	{
		matchingTermIndex = -1;
		matchingTermCounter = 0;
		numOfMatchingTerms = 0;
		cntrlPanel1.numOfMatches(-1);
	}
	
	public void setSearchString(String str)
	{
		searchString = str;
	}
	
	public String getSearchString()
	{
		return searchString;
	}
	
	public void search(String str)
	{
		Term[] terms = doc.getTerms();
		Preferences prefs = doc.getPreferences();
		
		if (terms != null) {
			if (str.equals("")) {
				for (int i = 0; i < terms.length; i++) {
					terms[i].setMatching(false);
				}
			}
			else {
				String srchPattern = str;
				Pattern patt;
				Matcher matcher;
				int i, n = 0, popt = Pattern.UNICODE_CASE;
				
				
				if (!prefs.caseSensitive) popt |= Pattern.CASE_INSENSITIVE;
				if (!prefs.regEx) {
					srchPattern = Pattern.quote(srchPattern);
					if (prefs.entireWord) srchPattern = "(\\s|^)" + srchPattern + "(\\s|$)";
				}
				
				try {
					patt = Pattern.compile(srchPattern, popt);
					patternError = false;
				}
				catch (PatternSyntaxException exception) {
					patt = null;
					patternError = true;
				}
				if (!patternError) {
					for (i = 0; i < terms.length; i++) {
						matcher = patt.matcher(terms[i].str);
						if (matcher.find()) {
							terms[i].setMatching(true);
							n++;
						}
						else terms[i].setMatching(false);
					}
				}
				numOfMatchingTerms = n;
				matchingTermCounter = 0;
				matchingTermIndex = -1;
				searchString = str;
				cntrlPanel1.numOfMatches(n);
				cntrlPanel1.updateNavButtons();
				if (n > 0 && !prefs.filterResults) findNext();
			}
		}
		searchString = str;
	}
	
	public void search()
	{
		setSearchString("");
		cntrlPanel1.doSearch();
	}
	
	public void setNumOfMatches()
	{
		Term[] terms = doc.getTerms();
		int n = 0;
		
		for (int i = 0; i < doc.getTableSize(); i++) {
			if (terms[i].isMatching()) n++;
		}
		numOfMatchingTerms = n;
		matchingTermCounter = 0;
		matchingTermIndex = -1;
		cntrlPanel1.numOfMatches(n);
		cntrlPanel1.updateNavButtons();
	}
	
	public void findNext()
	{
		if (matchingTermCounter < numOfMatchingTerms) {
			int i, n = matchingTermIndex + 1;
			Term[] terms = doc.getTerms();
			
			for (i = n; i < doc.getTableSize(); i++) {
				if (terms[i].isMatching()) {
					matchingTermIndex = i;
					matchingTermCounter++;
					break;
				}
			}
			table.clearSelection();
			selectedTermIndex = matchingTermIndex;
			table.addRowSelectionInterval(selectedTermIndex, selectedTermIndex);
			cntrlPanel1.updateNavButtons();
			showSelected();
		}
	}
	
	public void findPrevious()
	{
		if (matchingTermCounter > 0) {
			int i, n = matchingTermIndex - 1;
			Term[] terms = doc.getTerms();
			
			for (i = n; i >= 0; i--) {
				if (terms[i].isMatching()) {
					matchingTermIndex = i;
					matchingTermCounter--;
					break;
				}
			}
			table.clearSelection();
			selectedTermIndex = matchingTermIndex;
			table.addRowSelectionInterval(selectedTermIndex, selectedTermIndex);
			cntrlPanel1.updateNavButtons();
			showSelected();
		}
	}
	
	public void showSelected()
	{
		EventQueue.invokeLater(scrollToSelectedRow);
	}
	
	public void showProgress(boolean show)
	{
		showProgress(show, true);
	}
	
	public void showProgress(boolean show, boolean allowCancel)
	{
		if (show) {
			if (bottomComponent == infoPanel) {
				remove(bottomComponent);
				add(cntrlPanel2, BorderLayout.SOUTH);
				bottomComponent = cntrlPanel2;
				cntrlPanel2.allowCancel(allowCancel);
				revalidate();
				repaint();
			}
		}
		else {
			if (bottomComponent == cntrlPanel2) {
				remove(bottomComponent);
				add(infoPanel, BorderLayout.SOUTH);
				bottomComponent = infoPanel;
				revalidate();
				repaint();
			}
		}
	}
	
	public void changeProgress(int type)
	{
		cntrlPanel2.changeProgress(type);
	}
	
	public void reportTagging(String fname)
	{
		cntrlPanel2.reportTagging(fname);
	}
	
	public void reportPreprocessing(String fname)
	{
		cntrlPanel2.reportPreprocessing(fname);
	}
	
	public void report(float value)
	{
		cntrlPanel2.report(value);
	}
	
	public void report(String fileName, int count)
	{
		cntrlPanel2.report(fileName, count);
	}
	
	public void report(int count, float value)
	{
		cntrlPanel2.report(count, value);
	}
	
	public void report(int processed, int total, float value)
	{
		cntrlPanel2.report(processed, total, value);
	}
	
	public void finalize()
	{
		cntrlPanel2.finalize();
	}
	
	public void pleaseWait()
	{
		cntrlPanel2.pleaseWait();
	}
	
	public void setInfo(String info)
	{
		infoPanel.setInfo(info);
	}
	
	public void setWarning(String warning)
	{
		infoPanel.setWarning(warning);
	}
	
	public void setAccessory(AbstractAction action)
	{
		infoPanel.setAccessory(action);
	}
	
	public void refreshColumns()
	{
		table.showColumns(false);
		table.showColumns(true);
	}
	
	public TermTable getTermTable()
	{
		return table;
	}
	
	public Term getSelectedTerm()
	{
		return selectedTerm;
	}
	
	public void setSelectedTerm(Term term)
	{
		Term[] terms = doc.getTerms();
		
		for (int i = 0; i < terms.length; i++) {
			if (terms[i] == term) {
				table.setRowSelectionInterval(i, i);
				showSelected();
			}
		}
	}
	
	public void changeFontSize()
	{
		table.changeFontSize();
	}
	
	private class TermTable extends JTable
	{
		
		private PlaceHolder placeHolder;
		private TableColumn[] tableColumns;
		
		public TermTable(PlaceHolder placeHolder)
		{
			super();
			this.placeHolder = placeHolder;
			initTable();
		}
		
		public void initTable()
		{
			TableViewModel model = new TableViewModel();
			FontMetrics fm = getFontMetrics(TermoPL.preferences.boldFont);

			setModel(model);
			setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createEmptyBorder(-1, -1 , 0, -1),
					BorderFactory.createLineBorder(Color.lightGray)));
			setBackground(CommonResources.LIGHT_GRAY);
			setGridColor(Color.lightGray);
			setAutoResizeMode(AUTO_RESIZE_OFF);
			setShowHorizontalLines(false);
			setRowHeight(fm.getHeight() + 4);
			
			TableColumnModel columnModel = getColumnModel();
			TableCellRenderer cellRenderer = new RowRenderer();
			TableHeader tableHeader = new TableHeader(columnModel);
			
			setTableHeader(tableHeader);
			tableColumns = new TableColumn[HEADERS.length];

			for (int i = 0; i < columnModel.getColumnCount(); i++) {
				TableColumn column = columnModel.getColumn(i);
				
				tableColumns[i] = column;
				column.setHeaderValue(HEADERS[i]);
				column.setCellRenderer(cellRenderer);
				column.setPreferredWidth(SIZES[i]);
			}
			changeHeader();
			columnsAreVisible = true;
			showColumns(false);

			getSelectionModel().addListSelectionListener(new ListSelectionListener() {
				public void valueChanged(ListSelectionEvent event) 
				{
					if (!event.getValueIsAdjusting()) {
						int row = getSelectedRow();
						
						if (row >= 0) {
							selectedTerm = doc.getTerms()[row];
							selectedTermIndex = row;
						}
						else {
							selectedTerm = null;
							selectedTermIndex = -1;
						}
						if (doc.getPreferences().collectAllForms) {
							FormsView fv = doc.getFormsView();
							
							if (fv != null) fv.setData(selectedTerm);
						}
						if (doc.getPreferences().makeIndex) {
							SentencesView sv = doc.getSentencesView();
							
							if (sv != null) sv.setData(selectedTerm);
						}
						if (doc.getPreferences().makeGroups) {
							GroupsView gv = doc.getGroupsView();
							
							if (gv != null) gv.setData((TermEx)selectedTerm);
						}
					}
				}
			});
		}
		
		public void changeFontSize()
		{
			FontMetrics fm = getFontMetrics(TermoPL.preferences.plainFont);
			TableColumnModel columnModel = getColumnModel();
			TableHeader tableHeader = new TableHeader(columnModel);
			Rectangle r = getVisibleRect();
			Point p = r.getLocation();
			int h = fm.getHeight() + 4;
			int x = rowAtPoint(p);
			
			setRowHeight(h);
			setTableHeader(tableHeader);
			for (int i = 0; i < columnModel.getColumnCount(); i++) {
				TableColumn column = columnModel.getColumn(i);
				column.setHeaderRenderer(tableHeader.getDefaultRenderer());
			}
			r.y = x * h;
			scrollRectToVisible(r);
			tableScrollPane.revalidate();
			tableScrollPane.repaint();
		}
		
		public boolean isCellEditable(int row, int column)
		{
			return false;
		}
		
		public void resetResults()
		{
			if (doc.getTerms() == null) showColumns(false);
			else {
				showColumns(true);
				clearSelection();
				revalidate();
				scrollRectToVisible(CommonResources.nullRect);
				repaint();
			}
			getTableHeader().repaint();
		}
		
		public void sortTable(int column)
		{
			doc.sortTable(column);
			
			Term[] terms = doc.getTerms();
			int i;

			if (selectedTerm != null) {
				for (i = 0; i < terms.length; i++) {
					if (terms[i] == selectedTerm) break;
				}
				clearSelection();
				addRowSelectionInterval(i, i);

			}
			if (selectedTerm != null) showSelected();
			repaint();
			getTableHeader().repaint();
		}
		
		public void showColumns(boolean show)
		{
			if (columnsAreVisible != show) {
				columnsAreVisible = show;
				for (int i = 0; i < tableColumns.length; i++) {
					if (show) {
						if (i == TermoPLConsts.CSVALUE) {
							if (doc.getPreferences().applyContrastiveRanking && doc.getCTerms() != null) addColumn(tableColumns[i]);
						}
						else addColumn(tableColumns[i]);
					}
					else removeColumn(tableColumns[i]);
				}
				if (show) {
					TableColumnModel columnModel = getColumnModel();
					
					for (int i = 0; i < columnModel.getColumnCount(); i++) {
						TableColumn column = columnModel.getColumn(i);
						
						column.setHeaderRenderer(tableHeader.getDefaultRenderer());
					}
				}
			}
			placeHolder.showBottomLine(show);
		}
		
		public void changeHeader()
		{
			tableColumns[TermoPLConsts.CSVALUE].setHeaderValue(CMETHODS[doc.getPreferences().contrastiveRankingMethod - 1]);
			getTableHeader().repaint();
		}
		
	}
	
	private class TableViewModel extends AbstractTableModel
	{
		
		private DecimalFormat format = new DecimalFormat("#.##");
		
		public TableViewModel()
		{
			super();
		}
		
		public int getRowCount()
		{
			return doc.getTableSize();
		}
		
		public int getColumnCount()
		{
			return HEADERS.length;
		}
		
		public Object getValueAt(int rowIndex, int columnIndex) 
		{
			if (doc.getTerms() != null) {
				Term t = doc.getTerms()[rowIndex];
				
				switch (columnIndex) {
					case TermoPLConsts.NUM :
						return Integer.toString(rowIndex + 1);
					case TermoPLConsts.RANK :
						return Integer.toString(t.rank);
					case TermoPLConsts.TERM :
						return t.str;
					case TermoPLConsts.CVALUE :
						return format.format(t.cvalue);
					case TermoPLConsts.CSVALUE : 
						if (t.getStatus() == Term.IGNORE_STATUS) return("n/a");
						if (t.contrast == Double.POSITIVE_INFINITY || t.contrast == Double.NEGATIVE_INFINITY) return("-");
						return format.format(t.contrast);
					case TermoPLConsts.LENGTH :
						return Integer.toString(t.len);
					case TermoPLConsts.FREQS :
						return Integer.toString(t.freq_s);
					case TermoPLConsts.FREQIN :
						return Integer.toString(t.freq_in);
					case TermoPLConsts.CONTEXT :
						return Integer.toString(t.lk);
				}
			}
			return null;
		}
		
	}
	
	private class RowRenderer extends DefaultTableCellRenderer
	{
		
		private final Border border = BorderFactory.createEmptyBorder(0, 4, 0, 4);
		
		public RowRenderer()
		{
			super();
		}
		
		public Component getTableCellRendererComponent(JTable table, Object value,
				boolean isSelected, boolean hasFocus, int row, int column)
		{
			if (isSelected) {
				 setForeground(table.getSelectionForeground());
				 setBackground(table.getSelectionBackground());
			}
			else {
				setForeground(Color.black);
				if (table.getColumnCount() == HEADERS.length) {
					setBackground(doc.getTerms()[row].getColor());
				}
				else {
					if (row % 2 == 0) setBackground(CommonResources.VLIGHT_BLUE);
					else setBackground(Color.white);
				}
			}
			setValue(value);
			setFont(TermoPL.preferences.plainFont);
			setBorder(border);
			if (column <= TermoPLConsts.RANK) setHorizontalAlignment(JLabel.CENTER);
			else setHorizontalAlignment(JLabel.LEFT);
			return this;
		}
		
	}
	
	private class TableHeader extends JTableHeader
	{
		
		private PlaceHolder corner = new PlaceHolder();
		private boolean initialized = false;
		
		public TableHeader(TableColumnModel model)
		{
			super(model);
			setDefaultRenderer(new TableHeaderRenderer());
			setReorderingAllowed(false);
			corner.setLocation(0, 0);
			corner.showBottomLine(columnsAreVisible);
			add(corner);
			addMouseListener(new MouseAdapter() 
			{
				public void mouseClicked(MouseEvent event)
				{
					TableColumnModel tcm = getColumnModel();
					int column = tcm.getColumnIndexAtX(event.getX());
					
					if (column > TermoPLConsts.NUM) {
						if (!doc.getPreferences().applyContrastiveRanking || doc.getCTerms() == null) {
							if (column > TermoPLConsts.CVALUE) column++;
						}
						((TermTable)getTable()).sortTable(column);
					}
				}
			});
		}
		
		public Dimension getPreferredSize()
		{
			Dimension dim = super.getPreferredSize();
			FontMetrics fm = getFontMetrics(TermoPL.preferences.boldFont);
			
			dim.height = fm.getHeight() + 4;
			return dim;
		}

		public void setBounds(int x, int y, int width, int height)
		{
			super.setBounds(x, y, width, height);
			moveCorner();
			initialized = true;
		}
		
		public void moveCorner()
		{
			TableColumnModel tcm = getColumnModel();
			Rectangle	rect;
			
			rect = getHeaderRect(tcm.getColumnCount() - 1);
			corner.setLocation(rect.x + rect.width, 0);
			corner.setSize(getWidth() - rect.x - rect.width, getHeight());
		}
		
		public void resizeAndRepaint()
		{
			TableColumnModel model = getColumnModel();
			
			super.resizeAndRepaint();
			if (initialized) {
				moveCorner();
				corner.showBottomLine(model.getColumnCount() > 0);
			}
		}
		
	}
	
	private class TableHeaderRenderer implements TableCellRenderer
	{
		
		private JLabel renderer;
		private ImageIcon down1, down2, up1, up2;

		public TableHeaderRenderer()
		{
			super();
			initIcons();
			initRenderer();
		}
		
		private void initIcons()
		{
			URL url = ClassLoader.getSystemClassLoader().getResource("down_1.gif");
			
			down1 = new ImageIcon(url);
			url = ClassLoader.getSystemClassLoader().getResource("down_2.gif");
			down2 = new ImageIcon(url);
			url = ClassLoader.getSystemClassLoader().getResource("up_1.gif");
			up1 = new ImageIcon(url);
			url = ClassLoader.getSystemClassLoader().getResource("up_2.gif");
			up2 = new ImageIcon(url);
		}
		
		private void initRenderer()
		{
			Border headerBorder = UIManager.getBorder("TableHeader.cellBorder");
			
			renderer = new JLabel();
			renderer.setFont(TermoPL.preferences.boldFont);
			renderer.setBorder(headerBorder);
			renderer.setBackground(UIManager.getColor("TableHeader.background"));
			renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
		}
		
		public Component getTableCellRendererComponent(JTable table, Object value,
			boolean isSelected, boolean hasFocus, int row, int column)
		{
			Preferences prefs = doc.getPreferences();
			
			renderer.setText((String)value);
			if (column > 0) {
				if (!prefs.applyContrastiveRanking || doc.getCTerms() == null) {
					if (column > TermoPLConsts.CVALUE) column++;
				}
				if (column == prefs.sortedColumn) {
					if (prefs.sortPrefs[column] > 0) renderer.setIcon(up1);
					else renderer.setIcon(down1);
				}
				else {
					if (prefs.sortPrefs[column] > 0) renderer.setIcon(up2);
					else renderer.setIcon(down2);
				}
			}
			else renderer.setIcon(null);
			if (column <= 1) renderer.setHorizontalAlignment(JLabel.CENTER);
			else renderer.setHorizontalAlignment(JLabel.LEFT);
			switch (column) {
				case TermoPLConsts.NUM :
					if (prefs.saveCount) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.RANK :
					if (prefs.saveRank) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.TERM :
					renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					break;
				case TermoPLConsts.CVALUE :
					if (prefs.saveCV) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.CSVALUE :
					if (prefs.saveComp) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.LENGTH :
					if (prefs.saveLen) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.FREQS :
					if (prefs.saveFreqs) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.FREQIN :
					if (prefs.saveFreqin) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
					break;
				case TermoPLConsts.CONTEXT :
					if (prefs.saveContext) {
						renderer.setForeground(UIManager.getColor("TableHeader.foreground"));
					}
					else {
						renderer.setForeground(Color.blue);
					}
			}
			return renderer;
		}
		
	}
	
	private class CntrlPanel1 extends JPanel
	{
		private JPanel mainPanel;
		private JPanel searchOptionsPanel;
		private JCheckBox showTopRanked;
		private JCheckBox showMultiWordTermsOnly;
		private JCheckBox filter;
		private JCheckBox caseSensitive;
		private JCheckBox entireWord;
		private JCheckBox regEx;
		private JLabel show;
		private JLabel topRanked;
		private JLabel numOfMatches;
		private JLabel searchLabel;
		private JTextField searchField;
		private JTextField numTopRanked;
		private JTextField minLengthField;
		private JTextField maxLengthField;
		private PictButton prevButton;
		private PictButton nextButton;
		private PictButton searchButton;
		
		public CntrlPanel1()
		{
			Dimension dim;
			Action action;
			Preferences prefs = doc.getPreferences();
			JLabel label = new JLabel(" - ");
			
			setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
			mainPanel = new JPanel();
			mainPanel.setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createCompoundBorder(
						BorderFactory.createEmptyBorder(-1, -1, 0, -1),
						BorderFactory.createLineBorder(Color.lightGray)),
					BorderFactory.createEmptyBorder(5, 8, 3, 8)));
			mainPanel.setBackground(CommonResources.GRAY);
			mainPanel.setLayout(new BoxLayout(mainPanel, BoxLayout.X_AXIS));
			showTopRanked = new JCheckBox();
			showTopRanked.setOpaque(false);
			showTopRanked.setSelected(prefs.trimResults);
			mainPanel.add(showTopRanked);
			show = new JLabel("Show ");
			mainPanel.add(show);
			numTopRanked = new JTextField(Integer.toString(prefs.maxResults), 4);
			dim = numTopRanked.getPreferredSize();
			numTopRanked.setMaximumSize(dim);
			mainPanel.add(numTopRanked);
			topRanked = new JLabel(" top-ranked terms");
			mainPanel.add(topRanked);
			mainPanel.add(Box.createHorizontalStrut(16));
			showMultiWordTermsOnly = new JCheckBox("Multi-word terms of length ");
			showMultiWordTermsOnly.setOpaque(false);
			showMultiWordTermsOnly.setSelected(prefs.multiWordTermsOnly);
			mainPanel.add(showMultiWordTermsOnly);
			minLengthField = new JTextField(Integer.toString(prefs.minLength), 2);
			dim = minLengthField.getPreferredSize();
			minLengthField.setMaximumSize(dim);
			maxLengthField = new JTextField(2);
			if (prefs.maxLength > 0) maxLengthField.setText(Integer.toString(prefs.maxLength));
			dim = maxLengthField.getPreferredSize();
			maxLengthField.setMaximumSize(dim);
			mainPanel.add(minLengthField);
			mainPanel.add(label);
			mainPanel.add(maxLengthField);
			mainPanel.add(Box.createHorizontalStrut(16));
			mainPanel.add(Box.createHorizontalGlue());
			numOfMatches = new JLabel();
			numOfMatches.setVisible(false);
			mainPanel.add(numOfMatches); 
			mainPanel.add(Box.createHorizontalStrut(16));
			prevButton = new PictButton("left.png", "left_pressed.png", "left_rollover.png");
			prevButton.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createLineBorder(Color.lightGray),
				BorderFactory.createEmptyBorder(3, 3, 3, 3)));
			prevButton.setBorderPainted(true);
			prevButton.setContentAreaFilled(true);
			mainPanel.add(prevButton);
			prevButton.setVisible(false);
			mainPanel.add(Box.createHorizontalStrut(4));
			nextButton = new PictButton("right.png", "right_pressed.png", "right_rollover.png");
			nextButton.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createLineBorder(Color.lightGray),
				BorderFactory.createEmptyBorder(3, 3, 3, 3)));
			nextButton.setBorderPainted(true);
			nextButton.setContentAreaFilled(true);
			nextButton.setVisible(false);
			mainPanel.add(nextButton);
			mainPanel.add(Box.createHorizontalStrut(16));
			searchButton = new PictButton("search.png", "search_pressed.png", "search_rollover.png");
			mainPanel.add(searchButton);
			mainPanel.add(Box.createHorizontalStrut(4));
			searchLabel = new JLabel("Search: ");
			mainPanel.add(searchLabel);
			searchField = new JTextField(10);
			dim = searchField.getPreferredSize();
			searchField.setMaximumSize(dim);
			mainPanel.add(searchField);
			filter = new JCheckBox("Filter results");
			filter.setOpaque(false);
			filter.setSelected(prefs.filterResults);
			mainPanel.add(filter);
			add(mainPanel);
			searchOptionsPanel = new JPanel();
			searchOptionsPanel.setVisible(false);
			searchOptionsPanel.setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createCompoundBorder(
						BorderFactory.createEmptyBorder(-1, -1, 0, -1),
						BorderFactory.createLineBorder(Color.lightGray)),
					BorderFactory.createEmptyBorder(2, 8, 1, 8)));
			searchOptionsPanel.setLayout(new BoxLayout(searchOptionsPanel, BoxLayout.X_AXIS));
			searchOptionsPanel.add(Box.createHorizontalGlue());
			caseSensitive = new JCheckBox("Case sensitive");
			caseSensitive.setSelected(prefs.caseSensitive);
			caseSensitive.putClientProperty("JComponent.sizeVariant", "small");
			searchOptionsPanel.add(caseSensitive);
			entireWord = new JCheckBox("Entire word");
			entireWord.setSelected(prefs.entireWord);
			entireWord.setEnabled(!prefs.regEx);
			entireWord.putClientProperty("JComponent.sizeVariant", "small");
			searchOptionsPanel.add(entireWord);
			regEx = new JCheckBox("Regular expression");
			regEx.setSelected(prefs.regEx);
			regEx.putClientProperty("JComponent.sizeVariant", "small");
			searchOptionsPanel.add(regEx);
			add(searchOptionsPanel);
			
			action = new AbstractAction() {
				public void actionPerformed(ActionEvent event) 
				{
					doc.trim(showTopRanked.isSelected(), showMultiWordTermsOnly.isSelected(), filter.isSelected());
				}
			};
			numTopRanked.addActionListener(action);
			minLengthField.addActionListener(action);
			maxLengthField.addActionListener(action);
			showTopRanked.addActionListener(action);
			showMultiWordTermsOnly.addActionListener(action);
			
			numTopRanked.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					updateMaxResults();
				}
			});
			
			numTopRanked.addFocusListener(new FocusListener() {
				public void focusGained(FocusEvent e) 
				{
				}

				public void focusLost(FocusEvent e) 
				{
					updateMaxResults();
				}
			});
			
			minLengthField.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					updateMinLength();
				}
			});
			
			minLengthField.addFocusListener(new FocusListener() {
				public void focusGained(FocusEvent e) 
				{
				}

				public void focusLost(FocusEvent e) 
				{
					updateMinLength();
				}
			});
			
			maxLengthField.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					updateMaxLength();
				}
			});
			
			maxLengthField.addFocusListener(new FocusListener() {
				public void focusGained(FocusEvent e) 
				{
				}

				public void focusLost(FocusEvent e) 
				{
					updateMaxLength();
				}
			});
			
			searchButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					searchOptionsPanel.setVisible(!searchOptionsPanel.isVisible());
				}
			});
			
			action = new AbstractAction() {
				public void actionPerformed(ActionEvent event) 
				{
					searchField.selectAll();
				}
			};
			searchField.addActionListener(action);
			searchField.getDocument().addDocumentListener(new DocumentListener() {
				public void insertUpdate(DocumentEvent e) 
				{
					doSearch();
				}

				public void removeUpdate(DocumentEvent e) 
				{
					doSearch();
				}

				public void changedUpdate(DocumentEvent e) 
				{
				}
			});
			nextButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					findNext();
				}
			});
			prevButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					findPrevious();
				}
			});
			filter.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					if (doc.getTerms() == null) {
						prevButton.setVisible(false);
						nextButton.setVisible(false);
						numOfMatches.setVisible(false);
					}
					else {
						doc.trim(showTopRanked.isSelected(), showMultiWordTermsOnly.isSelected(), filter.isSelected());
	
						boolean f = !filter.isSelected() && numOfMatchingTerms > 0;
						
						prevButton.setVisible(f);
						nextButton.setVisible(f);
					}
				}
			});
			caseSensitive.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e)
				{
					Preferences prefs = doc.getPreferences();
					
					if (prefs.caseSensitive != caseSensitive.isSelected()) {
						prefs.caseSensitive = !prefs.caseSensitive;
						TermoPL.preferences.caseSensitive = prefs.caseSensitive;
						TermoPL.preferences.setModified(true);
						doSearch(true);
					}
				}
			});
			entireWord.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e)
				{
					Preferences prefs = doc.getPreferences();
					
					if (prefs.entireWord != entireWord.isSelected()) {
						prefs.entireWord = !prefs.entireWord;
						TermoPL.preferences.entireWord = prefs.entireWord;
						TermoPL.preferences.setModified(true);
						doSearch(true);
					}
				}
			});
			regEx.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e)
				{
					Preferences prefs = doc.getPreferences();
					
					if (prefs.regEx != regEx.isSelected()) {
						prefs.regEx = !prefs.regEx;
						TermoPL.preferences.regEx = prefs.regEx;
						TermoPL.preferences.setModified(true);
						entireWord.setEnabled(!regEx.isSelected());
						doSearch(true);
					}
				}
			});
		}
		
		public void doSearch()
		{
			doSearch(false);
		}
		
		public void doSearch(boolean enforce)
		{
			String str = searchField.getText();
			
			if (!str.equals(searchString) || enforce) {
				numOfMatches(-1);
				search(str);
				if (filter.isSelected()) {
					doc.trim(showTopRanked.isSelected(), showMultiWordTermsOnly.isSelected(), filter.isSelected());
				}
			}
		}
		
		public void updateNavButtons()
		{
			prevButton.setEnabled(matchingTermCounter > 1);
			nextButton.setEnabled(matchingTermCounter < numOfMatchingTerms);
		}
		
		public void updateMaxResults()
		{
			Preferences prefs = doc.getPreferences();
			
			try {
				int m = Integer.parseInt(numTopRanked.getText());
				
				if (m != prefs.maxResults) {
					prefs.maxResults = m;
					TermoPL.preferences.maxResults = m;
					TermoPL.preferences.setModified(true);
					if (prefs.trimResults) {
						doc.trim(prefs.trimResults, prefs.multiWordTermsOnly, prefs.filterResults);
					}
				}
			}
			catch (NumberFormatException exception) {
				Toolkit.getDefaultToolkit().beep();
				numTopRanked.setText(Integer.toString(prefs.maxResults));
			}
			numTopRanked.selectAll();
		}
		
		public void updateMinLength()
		{
			Preferences prefs = doc.getPreferences();
			
			try {
				prefs.minLength = Integer.parseInt(minLengthField.getText());
				TermoPL.preferences.minLength = prefs.minLength;
				TermoPL.preferences.setModified(true);
				if (prefs.trimResults) {
					doc.trim(prefs.trimResults, prefs.multiWordTermsOnly, prefs.filterResults);
				}
			}
			catch (NumberFormatException exception) {
				Toolkit.getDefaultToolkit().beep();
				minLengthField.setText(Integer.toString(prefs.minLength));
			}
			minLengthField.selectAll();
		}
		
		public void updateMaxLength()
		{
			Preferences prefs = doc.getPreferences();
			
			try {
				String str = maxLengthField.getText().trim();
				
				if (str.isEmpty()) {
					if (prefs.maxLength > 0) {
						prefs.maxLength = -1;
						TermoPL.preferences.maxLength = -1;
						TermoPL.preferences.setModified(true);
						if (prefs.trimResults) {
							doc.trim(prefs.trimResults, prefs.multiWordTermsOnly, prefs.filterResults);
						}
					}
				}
				else {
					int len = Integer.parseInt(str);
					
					if (len != prefs.maxLength) {
						prefs.maxLength = len;
						TermoPL.preferences.maxLength = len;
						TermoPL.preferences.setModified(true);
						if (prefs.trimResults) {
							doc.trim(prefs.trimResults, prefs.multiWordTermsOnly, prefs.filterResults);
						}
					}
				}
			}
			catch (NumberFormatException exception) {
				Toolkit.getDefaultToolkit().beep();
				if (prefs.maxLength < 0) maxLengthField.setText("");
				else maxLengthField.setText(Integer.toString(prefs.maxLength));
			}
			maxLengthField.selectAll();
		}
		
		public void numOfMatches(int n)
		{
			if (n < 0) {
				nextButton.setVisible(false);
				prevButton.setVisible(false);
				numOfMatches.setVisible(false);
			}
			else {
				if (n == 0) {
					if (searchString.equals("")) {
						numOfMatches.setText("");
						numOfMatches.setVisible(false);
					}
					else {
						numOfMatches.setText("No matches");
						numOfMatches.setVisible(true);
					}
					nextButton.setVisible(false);
					prevButton.setVisible(false);
				}
				else if (n == 1) {
					numOfMatches.setText("1 match");
					nextButton.setVisible(false);
					prevButton.setVisible(false);
					numOfMatches.setVisible(true);
				}
				else {
					boolean f = !searchString.equals("") && !filter.isSelected();
					
					if (searchString.equals("")) numOfMatches.setText("");
					else numOfMatches.setText(n + " matches");
					
					nextButton.setVisible(f);
					prevButton.setVisible(f);
					numOfMatches.setVisible(true);
				}
			}
		}
		
	}
	
	private class CntrlPanel2 extends JPanel
	{
		
		private JProgressBar progress;
		private JButton cancelButton;
		private JLabel progressLabel;
		private JLabel extracted;
		private JLabel fileNameLabel;
		private JLabel fileName;
		private JLabel normalized;
		private Component hspace;
		private Dimension progressDim;
		private Dimension indeterminateProgressDim;
		
		public CntrlPanel2()
		{
			setBackground(CommonResources.GRAY);
			setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
			setBorder(BorderFactory.createEmptyBorder(8, 8, 8, 8));
			progress = new JProgressBar();
			progress.setMinimum(0);
			progress.setMaximum(100);
			progress.setIndeterminate(true);
			if (TermoPL.isMacOS) progress.putClientProperty("JProgressBar.style", "circular");
			indeterminateProgressDim = progress.getPreferredSize();
			progressDim = new Dimension();
			progressDim.width = 80;
			progressDim.height = indeterminateProgressDim.height;
			if (!TermoPL.isMacOS) indeterminateProgressDim.width = 80;
			progress.setMaximumSize(progressDim);
			add(progress);
			add(Box.createHorizontalStrut(16));
			fileNameLabel = new JLabel();
			add(fileNameLabel);
			fileName = new JLabel();
			add(fileName);
			hspace = Box.createHorizontalStrut(16);
			add(hspace);
			progressLabel = new JLabel("Matched maximal phrases: ");
			add(progressLabel);
			extracted = new JLabel();
			add(extracted);
			normalized = new JLabel();
			normalized.setVisible(false);
			add(normalized);
			add(Box.createHorizontalGlue());
			cancelButton = new JButton("Cancel");
			cancelButton.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) 
				{
					doc.cancel();
				}
			});
			add(cancelButton);
		}
		
		public void allowCancel(boolean f)
		{
			cancelButton.setVisible(f);
		}
		
		public void finalize()
		{
			progress.setVisible(false);
			hspace.setVisible(false);
			fileName.setVisible(false);
			fileNameLabel.setVisible(false);
			extracted.setVisible(false);
			normalized.setVisible(false);
			progressLabel.setText("Finalizing...");
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
		public void pleaseWait()
		{
			progress.setIndeterminate(true);
			progress.setPreferredSize(indeterminateProgressDim);
			progress.setVisible(true);
			hspace.setVisible(false);
			fileName.setVisible(false);
			fileNameLabel.setVisible(false);
			extracted.setVisible(false);
			normalized.setVisible(false);
			progressLabel.setText("Wait...");
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
		public void changeProgress(int type)
		{
			progress.setValue(0);
			progress.setVisible(true);
			if (type == 0) {
				extracted.setVisible(false);
				normalized.setVisible(false);
				fileName.setVisible(false);
				fileNameLabel.setVisible(false);
				hspace.setVisible(false);
				progress.setIndeterminate(true);
				progress.setPreferredSize(progressDim);
				progressLabel.setText("Loading Contrastive Set of Terms...");
			}
			else if (type == 1) {
				allowCancel(true);
				extracted.setVisible(true);
				normalized.setVisible(false);
				fileName.setVisible(true);
				fileNameLabel.setVisible(true);
				fileNameLabel.setText("File: ");
				hspace.setVisible(true);
				progressLabel.setVisible(true);
				progressLabel.setText("Matched maximal phrases: ");
				progress.setIndeterminate(true);
				progress.setPreferredSize(indeterminateProgressDim);
			}
			else if (type == 2) {
				allowCancel(true);
				extracted.setVisible(true);
				normalized.setVisible(false);
				fileName.setVisible(true);
				fileNameLabel.setVisible(true);
				fileNameLabel.setText("File: ");
				hspace.setVisible(true);
				progressLabel.setVisible(true);
				progressLabel.setText("Extracted terms: ");
				progress.setIndeterminate(true);
				progress.setPreferredSize(indeterminateProgressDim);
			}
			else if (type == 3) {
				extracted.setVisible(false);
				normalized.setVisible(false);
				fileName.setVisible(false);
				fileNameLabel.setVisible(false);
				hspace.setVisible(false);
				progressLabel.setText("Merging...");
				progress.setIndeterminate(false);
				progress.setPreferredSize(progressDim);
			}
			else if (type >= 6 && type <= 8) {
				progress.setVisible(false);
				extracted.setVisible(false);
				normalized.setVisible(false);
				fileName.setVisible(false);
				fileNameLabel.setVisible(false);
				hspace.setVisible(false);
				if (type == 6) {
					progressLabel.setText("Building term hierarchy...");
					progress.setIndeterminate(false);
					progress.setPreferredSize(progressDim);
				}
				else if (type == 7) {
					progressLabel.setText("Loading WordNet...");
					progress.setIndeterminate(true);
					progress.setPreferredSize(indeterminateProgressDim);
				}
				else {
					progressLabel.setText("Analysing terms with WordNet...");
					progress.setIndeterminate(false);
					progress.setPreferredSize(progressDim);
				}
				progress.setVisible(true);
			}
			else {
				extracted.setVisible(true);
				normalized.setVisible(false);
				fileName.setVisible(false);
				fileNameLabel.setVisible(false);
				hspace.setVisible(false);
				progress.setIndeterminate(false);
				progress.setPreferredSize(progressDim);
				if (type == 4) {
					progressLabel.setText("Extracted terms: ");
				}
				else {
					extracted.setVisible(false);
					normalized.setVisible(true);
					normalized.setText("");
					if (doc.getPreferences().calculateBaseForms) progressLabel.setText("Base Forms: ");
					else progressLabel.setText("Simplified Forms: ");
				}
			}
			repaint();
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
		public void reportTagging(String fName)
		{
			fileNameLabel.setText("Parsing file " + fName + "...");
			fileName.setVisible(false);
			progressLabel.setVisible(false);
			extracted.setVisible(false);
			allowCancel(true);
		}
		
		public void reportPreprocessing(String fName)
		{
			fileNameLabel.setText("Preprocessing file " + fName + "...");
			fileName.setVisible(false);
			progressLabel.setVisible(false);
			extracted.setVisible(false);
			allowCancel(true);
		}
		
		public void report(float value)
		{
			progress.setValue((int)(100 * value));
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
		public void report(String name, int count)
		{
			if (!fileName.getText().equals(name)) fileName.setText(name);
			extracted.setText(Integer.toString(count));
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
		public void report(int count, float value)
		{
			extracted.setText(Integer.toString(count));
			progress.setValue((int)(100 * value));
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
		public void report(int processed, int total, float value)
		{
			progress.setValue((int)(100 * value));
			normalized.setText(Integer.toString(processed) + "/" + Integer.toString(total));
			try {
				Thread.sleep(1);
			}
			catch (InterruptedException exception) {
			}
		}
		
	}

	public class PlaceHolder extends JPanel 
	{

		private boolean show;
		
		public PlaceHolder()
		{
			super();
			show = false;
			setBackground(CommonResources.LIGHT_GRAY);
		}
		
		public void showBottomLine(boolean f)
		{
			if (show != f) {
				show = f;
				repaint();
			}
		}
		
		public void paintComponent(Graphics graphics)
		{
			super.paintComponent(graphics);
			if (show) {
				graphics.setColor(Color.lightGray);
				if (TermoPL.isMacOS) {
					graphics.drawLine(0,  getHeight() - 2,  getWidth(), getHeight() - 2);
					graphics.setColor(Color.white);
					graphics.drawLine(0,  getHeight() - 1,  getWidth(), getHeight() - 1);
				}
				else graphics.drawLine(0,  getHeight() - 1,  getWidth(), getHeight() - 1);
			}
		}
		
	}
	
}
