package termopl;

import java.awt.*;
import java.awt.event.*;
import java.util.*;

import javax.swing.*;

@SuppressWarnings("serial")
public class GroupsView extends JPanel 
{

	private static final int ROOT   = 0;
	private static final int PARENT = 1;
	private static final int CHILD  = 2;
	
	private TermoPLDocument doc;
	private TermComparator comp;
	private SimpleComparator simpleComp;
	private JSplitPane splitPane;
	private HierarchyPanel hierarchyPanel;
	private RelatedPanel relatedPanel;
	private InfoPanel infoPanel;
	private JLabel relatedLabel;
	
	public GroupsView(TermoPLDocument document)
	{
		doc = document;
		comp = new TermComparator();
		simpleComp = new SimpleComparator();
		setLayout(new BorderLayout());
		arrangeComponents();
		
		TermsView tv = doc.getTermsView();
		
		if (tv != null) setData((TermEx)tv.getSelectedTerm());
	}
	
	public void arrangeComponents()
	{
		JScrollPane sp;
		
		hierarchyPanel = new HierarchyPanel();
		sp = new JScrollPane(hierarchyPanel);
		sp.getViewport().setBackground(CommonResources.LIGHT_GRAY);
		sp.setBorder(BorderFactory.createCompoundBorder(
			BorderFactory.createEmptyBorder(-1, -1, 0, -1),
			BorderFactory.createLineBorder(Color.lightGray)));
		if (doc.getPreferences().useWordNet) {
			JPanel container = new JPanel(new BorderLayout());
			
			relatedLabel = new JLabel();
			relatedLabel.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createEmptyBorder(0, -1, -1, -1),
				BorderFactory.createCompoundBorder(
					BorderFactory.createLineBorder(Color.lightGray),
					BorderFactory.createEmptyBorder(2, 4, 2, 4))));
			container.add(relatedLabel, BorderLayout.NORTH);
			splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
			splitPane.setDividerSize(6);
			splitPane.setBorder(BorderFactory.createEmptyBorder());
			splitPane.setTopComponent(sp);
			relatedPanel = new RelatedPanel();
			sp = new JScrollPane(relatedPanel);
			sp.getViewport().setBackground(CommonResources.LIGHT_GRAY);
			sp.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createEmptyBorder(0, -1, 0, -1),
				BorderFactory.createLineBorder(Color.lightGray)));
			container.add(sp, BorderLayout.CENTER);
			splitPane.setBottomComponent(container);
			add(splitPane, BorderLayout.CENTER);
			EventQueue.invokeLater(new Runnable() {
				public void run() {
					splitPane.setDividerLocation(0.75);
				}
			});
		}
		else {
			relatedPanel = null;
			splitPane = null;
			add(sp, BorderLayout.CENTER);
		}
		infoPanel = new InfoPanel();
		add(infoPanel, BorderLayout.SOUTH);
	}
	
	public void requestData(TermEx term)
	{
		TermsView tv = doc.getTermsView();
		
		if (tv != null) tv.setSelectedTerm(term);
	}
	
	public void setData(TermEx term)
	{
		hierarchyPanel.setData(term);
		if (relatedPanel != null) relatedPanel.setData(term);
		if (term != null) infoPanel.setInfo("Term: " + term.str);
		else infoPanel.setWarning("Term is not selected.");
	}
	
	public void sort(LinkedList<String> terms, boolean simple)
	{
		if (simple) Collections.sort(terms, simpleComp);
		else Collections.sort(terms, comp);
	}
	
	public void reset()
	{
		setData(null);
	}
	
	public void changeFontSize()
	{
		hierarchyPanel.changeFont();
		if (relatedPanel != null) relatedPanel.changeFont();
		invalidate();
		repaint();
	}
	
	private class HierarchyPanel extends JPanel
	{
		
		private TreeNode root;
		private int offsetY, maxWidth;
		
		public HierarchyPanel()
		{
			super(null);
		}
		
		public void setData(TermEx term)
		{
			setVisible(false);
			removeAll();
			offsetY = maxWidth = 0;
			if (term != null) arrangeNodes(term);
			else root = null;
			EventQueue.invokeLater(new Runnable() {
				public void run()
				{
					scrollRectToVisible(CommonResources.nullRect);
					calcSize();
					revalidate();
					repaint();
				}
			});
			setVisible(true);
		}
		
		public void changeFont()
		{
			if (root != null) {
				TermEx term = root.term;
				
				setData(term);
			}
		}
		
		public void arrangeNodes(TermEx term)
		{
			offsetY = 16;
			arrangeNodes(null, term);
		}
		
		public void arrangeNodes(TreeNode parent, TermEx child)
		{
			TreeNode node = new TreeNode(child, (parent == null ? ROOT : CHILD));
			LinkedList<String> children = child.collectChildren(doc.getTermMap());
			LinkedList<String> parents = child.collectParents(doc.getTermMap());
			LinkedList<TermEx> visited;
			Dimension dim;
			int x, y, dy;
			
			add(node);
			if (parents != null) sort(parents, false);
			dim = node.getPreferredSize();
			node.setBounds(16, offsetY, dim.width, dim.height);
			dy = dim.height;
			if (parent == null) {
				root = node;
				if (maxWidth < dim.width + 16) maxWidth = dim.width + 16;
				if (parents != null) {
					ArrowPanel ap = new ArrowPanel();
					
					x = node.getX() + node.getWidth();
					y = node.getY();
					add(ap);
					dim = ap.getPreferredSize();
					ap.setBounds(x, y, dim.width, dim.height);
					x += dim.width;
					visited = new LinkedList<TermEx>();					
					for (String t : parents) {
						TermEx term = (TermEx)doc.getTermMap().get(t);
						boolean found = false;
						
						for (TermEx v : visited) {
							LinkedList<String> eq = v.getEquivalentTerms();
							
							if (eq != null && eq.contains(t)) {
								found = true;
								break;
							}
						}
						if (!found) {
							TreeNode p = new TreeNode(term, PARENT);
							
							add(p);
							dim = p.getPreferredSize();
							p.setBounds(x, y, dim.width, dim.height);
							if (dy < dim.height) dy = dim.height;
							x += dim.width + 4;
							visited.add(term);
						}
					}
					if (maxWidth < x) maxWidth = x;
				}
			}
			else {
				boolean drawArrow = true;
				
				parent.addChild(node);
				x = node.getX() + node.getWidth();
				y = node.getY();
				visited = new LinkedList<TermEx>();	
				if (parents != null) {
					for (String t : parents) {
						TermEx term = (TermEx)doc.getTermMap().get(t);
						boolean found = false;
						
						for (TermEx v : visited) {
							LinkedList<String> eq = v.getEquivalentTerms();
							
							if (eq != null && eq.contains(t)) {
								found = true;
								break;
							}
						}
						if (!root.term.isDescendant(term, doc.getTermMap()) && !found) {
							if (drawArrow) {
								ArrowPanel ap = new ArrowPanel();
								
								add(ap);
								dim = ap.getPreferredSize();
								ap.setBounds(x, y, dim.width, dim.height);
								x += dim.width;
								drawArrow = false;
							}
							
							TreeNode p = new TreeNode(term, PARENT);
							
							add(p);
							dim = p.getPreferredSize();
							if (dy < dim.height) dy = dim.height;
							p.setBounds(x, y, dim.width, dim.height);
							x += dim.width + 4;
							visited.add(term);
						}
					}
				}
				if (maxWidth < x) maxWidth = x;
			}
			offsetY = node.getY() + dy + 8;
			if (children != null) {
				sort(children, false);
				for (String t : children) {
					TermEx term = (TermEx)doc.getTermMap().get(t);
					
					arrangeNodes(node, term);
				}
			}
		}
		
		public void calcSize()
		{
			setPreferredSize(new Dimension(maxWidth + 16, offsetY + 8));
		}
		
		public void paintComponent(Graphics graphics)
		{
			Graphics2D g = (Graphics2D)graphics;
			g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
			g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
			
			super.paintComponent(g);
			if (root != null) {
				g.setColor(Color.gray);
				root.drawEdges(g);
			}
		}
		
		private class TreeNode extends JPanel
		{
			
			private TermEx term;
			private int type;
			private LinkedList<TreeNode> children = new LinkedList<TreeNode>();
			
			public TreeNode(TermEx term, int type)
			{
				super(new GridLayout(0, 1));
				setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createEmptyBorder(0, 0, -1, 0),
					BorderFactory.createLineBorder(Color.lightGray)));
				this.term = term;
				this.type = type;
				arrangeComponents();
			}
			
			public void arrangeComponents()
			{
				LinkedList<String> equiv = term.getEquivalentTerms();
				String skipID = null;
				boolean addType = true;
				
				if (type == ROOT) {
					addLabel(term, TermoPL.preferences.boldFont, type);
					skipID = term.id;
					addType = false;
				}
				else if (equiv == null) addLabel(term, TermoPL.preferences.boldFont, type);
				if (equiv != null) {
					sort(equiv, true);
					for (String e : equiv) {
						if (e != skipID) {
							if (addType) {
								addLabel((TermEx)doc.getTermMap().get(e), TermoPL.preferences.boldFont, type);
								addType = false;
							}
							else addLabel((TermEx)doc.getTermMap().get(e));
						}
					}
				}
			}
			
			public void addLabel(TermEx term)
			{
				TLabel label = new TLabel(term);
				add(label);
			}
			
			public void addLabel(TermEx term, Font font, int type)
			{
				Color color;
				
				switch (type) {
					case ROOT:
						color = CommonResources.LIGHT_BLUE;
						break;
					case PARENT:
						color = CommonResources.PALE_ORANGE;
						break;
					case CHILD:
						color = CommonResources.LIGHT_GREEN;
						break;
					default:
						color = CommonResources.CREAM;
				}
				TLabel label = new TLabel(term, font, color);
				add(label);
			}
			
			public void addChild(TreeNode node)
			{
				Dimension dim = node.getPreferredSize();
				
				children.add(node);
				node.setBounds(getX() + 32, offsetY, dim.width, dim.height);
				offsetY += dim.height + 8;
				if (maxWidth < getX() + dim.width + 32) maxWidth = getX() + dim.width + 32;
			}
			
			public void drawEdges(Graphics2D graphics)
			{
				for (TreeNode tn : children) {
					drawEdgeToChild(tn, graphics);
					tn.drawEdges(graphics);
				}
			}
			
			public void drawEdgeToChild(TreeNode tn, Graphics2D graphics)
			{
				int x1 = getX() + 8;
				int y1 = getY() + getHeight();
				int x2 = tn.getX();
				int y2 = tn.getY() + tn.getHeight() / 2;
		        Polygon arrowHead = new Polygon();
				
				graphics.drawLine(x1, y1, x1, y2 - 8);
		        graphics.drawArc(x1, y2 - 16, 16, 16, -90, -90);
				graphics.drawLine(x1 + 8, y2, x2, y2);
		        arrowHead.addPoint(x2, y2);
		        arrowHead.addPoint(x2 - 8, y2 + 4);
		        arrowHead.addPoint(x2 - 6, y2);
		        arrowHead.addPoint(x2 - 8, y2 - 4);
		        graphics.fill(arrowHead);
			}
			
		}
		
	}
	
	private class RelatedPanel extends JPanel
	{
		
		private TermEx term;
		private LinkedList<ReplacementsPanel> components;
		
		public RelatedPanel()
		{
			super(new GridBagLayout());
		}
		
		public void setData(TermEx term)
		{
			setVisible(false);
			this.term = term;
			removeAll();
			components = new LinkedList<ReplacementsPanel>();
			installRelated();
			if (term != null) {
				LinkedList<String> equiv = term.getEquivalentTerms();
				StringBuffer buffer = new StringBuffer();
				
				buffer.append("Terms related to: ");
				buffer.append("'");
				buffer.append(term.str);
				buffer.append("'");
				if (equiv != null) {
					for (String e : equiv) {
						if (e != term.id) {
							TermEx t = (TermEx)doc.getTermMap().get(e);
							
							buffer.append(" & ");
							buffer.append("'");
							buffer.append(t.str);
							buffer.append("'");
						}
					}
				}
				relatedLabel.setText(buffer.toString());
			}
			else relatedLabel.setText("Related terms");
			EventQueue.invokeLater(new Runnable() {
				public void run()
				{
					doLayout();
					revalidate();
					repaint();
				}
			});
			setVisible(true);
		}
		
		public void changeFont()
		{
			doLayout();
			revalidate();
			repaint();
		}
		
		public void installRelated()
		{
			if (term != null) {
				LinkedList<Pair<String, LinkedList<WordReplacement>>> relatedTerms = term.collectRelated(doc.getTermMap());
				GridBagConstraints constr = new GridBagConstraints();
				
				constr.anchor = GridBagConstraints.NORTHWEST;
				constr.fill = GridBagConstraints.HORIZONTAL;
				constr.weightx = 1.0;
				constr.weighty = 0.0;
				constr.gridx = 0;
				constr.gridy = 0;
				
				if (relatedTerms != null) {
					Color color = Color.white;
					
					for (Pair<String, LinkedList<WordReplacement>> p : relatedTerms) {
						ReplacementsPanel panel = new ReplacementsPanel(p);
						
						panel.setBackground(color);
						if (color == Color.white) color = CommonResources.VLIGHT_BLUE;
						else color = Color.white;
						add(panel, constr);
						components.add(panel);
						constr.gridy++;
					}
				}
				
				constr.fill = GridBagConstraints.VERTICAL;
				constr.weighty = 1.0;
				add(Box.createGlue(), constr);
			}
		}
		
		public void doLayout()
		{
			calcSize();
			super.doLayout();
		}
		
		public void calcSize()
		{
			for (ReplacementsPanel p : components) p.calcSize();
		}
		
		private class ReplacementsPanel extends JPanel
		{
			
			private Pair<String, LinkedList<WordReplacement>> replacements;
			
			public ReplacementsPanel(Pair<String, LinkedList<WordReplacement>> replacements)
			{
				this.replacements = replacements;
				setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createEmptyBorder(-1, -1, 0, -1),
					BorderFactory.createLineBorder(Color.lightGray)));
				addMouseListener(new MouseAdapter() {
					public void mouseClicked(MouseEvent event)
					{
						if (event.getClickCount() > 1) {
							TermEx t = (TermEx)doc.getTermMap().get(replacements.first);
							
							requestData(t);
						}
					}
				});
			}
			
			public void calcSize()
			{
				Graphics graphics = getGraphics();
				Graphics2D g = (Graphics2D)graphics;
				FontMetrics fm;
				String str;
				TermEx t;
				ListIterator<Integer> it;
				double fx = 4, fdx;
				int i, y = 2, w = getWidth() - 4;
				boolean app, reversed;
			
				g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
				g.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
				graphics.setColor(Color.black);
				graphics.setFont(TermoPL.preferences.boldFont);
				fm = graphics.getFontMetrics();
				t = (TermEx)doc.getTermMap().get(replacements.first);
				str = t.str + ":";
				fx += fm.getStringBounds(str, g).getWidth() + 16;
				graphics.setFont(TermoPL.preferences.plainFont);
				app = false;
				for (WordReplacement wr : replacements.second) {
					if (wr.relTypes != null) {
						if (wr.relTypes.getFirst() < 0) reversed = true;
						else reversed = false;
						if (app) {
							str = "; ";
							fdx = fm.getStringBounds(str, g).getWidth();
							if (fx + fdx > w) {
								fx = 32; y += fm.getHeight();
							}
							fx += fdx;
						}
						else app = true;
						fdx = fm.getStringBounds(wr.word, g).getWidth();
						if (fx + fdx > w) {
							fx = 32; y += fm.getHeight();
						}
						fx += fdx;
						
						if (reversed) str = " \u2190 ";
						else str = " \u2192 ";
						fdx = fm.getStringBounds(str, g).getWidth();
						if (fx + fdx > w) {
							fx = 32; y += fm.getHeight();
						}
						fx += fdx;
						
						fdx = fm.getStringBounds(wr.expression, g).getWidth();
						if (fx + fdx > w) {
							fx = 32; y += fm.getHeight();
						}
						fx += fdx;
						i = 0;
						it = wr.relTypes.listIterator();
						while (it.hasNext()) {
							int relType = it.next();
							
							if (i > 0) {
								str = ", ";
								fdx = fm.getStringBounds(str, g).getWidth();
								if (fx + fdx > w) {
									fx = 32; y += fm.getHeight();
								}
								fx += fdx;
								str = "";
							}
							else str = "[";
							if (relType < 0) relType = -relType;
							str = WordNet.getName(relType);
							if (i == wr.relTypes.size() - 1) str += "]";
							fdx = fm.getStringBounds(str, g).getWidth();
							if (fx + fdx > w) {
								fx = 32; y += fm.getHeight();
							}
							fx += fdx;
							i++;
						}
					}
				}
				y += fm.getHeight();
				setPreferredSize(new Dimension(0, y + 2));
			}
			
			public void paintComponent(Graphics graphics)
			{
				super.paintComponent(graphics);
				
				Graphics2D g = (Graphics2D)graphics;
				FontMetrics fm;
				String str;
				TermEx t;
				ListIterator<Integer> it;
				double fx = 4, fdx;
				int i, y = 2, w = getWidth() - 4;
				boolean app, reversed;
			
				g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
				g.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
				graphics.setColor(Color.black);
				graphics.setFont(TermoPL.preferences.boldFont);
				fm = graphics.getFontMetrics();
				t = (TermEx)doc.getTermMap().get(replacements.first);
				str = t.str + ":";
				g.drawString(str, (int)fx, y + fm.getAscent());
				fx += fm.getStringBounds(str, g).getWidth() + 16;
				graphics.setFont(TermoPL.preferences.plainFont);
				app = false;
				for (WordReplacement wr : replacements.second) {
					if (wr.relTypes != null) {
						if (wr.relTypes.getFirst() < 0) reversed = true;
						else reversed = false;
						if (app) {
							str = "; ";
							fdx = fm.getStringBounds(str, g).getWidth();
							if (fx + fdx > w) {
								fx = 32; y += fm.getHeight();
							}
							g.drawString(str, (int)fx, y + fm.getAscent());
							fx += fdx;
						}
						else app = true;
						fdx = fm.getStringBounds(wr.word, g).getWidth();
						if (fx + fdx > w) {
							fx = 32; y += fm.getHeight();
						}
						g.drawString(wr.word, (int)fx, y + fm.getAscent());
						fx += fdx;
						
						if (reversed) str = " \u2190 ";
						else str = " \u2192 ";
						fdx = fm.getStringBounds(str, g).getWidth();
						if (fx + fdx > w) {
							fx = 32; y += fm.getHeight();
						}
						g.drawString(str, (int)fx, y + fm.getAscent());
						fx += fdx;
						
						fdx = fm.getStringBounds(wr.expression, g).getWidth();
						if (fx + fdx > w) {
							fx = 32; y += fm.getHeight();
						}
						g.drawString(wr.expression, (int)fx, y + fm.getAscent());
						fx += fdx;
						i = 0;
						it = wr.relTypes.listIterator();
						while (it.hasNext()) {
							int relType = it.next();
							
							if (i > 0) {
								str = ", ";
								fdx = fm.getStringBounds(str, g).getWidth();
								if (fx + fdx > w) {
									fx = 32; y += fm.getHeight();
								}
								g.drawString(str, (int)fx, y + fm.getAscent());
								fx += fdx;
								str = "";
							}
							else str = "[";
							if (relType < 0) relType = -relType;
							str += WordNet.getName(relType);
							if (i == wr.relTypes.size() - 1) str += "]";
							fdx = fm.getStringBounds(str, g).getWidth();
							if (fx + fdx > w) {
								fx = 32; y += fm.getHeight();
							}
							g.drawString(str, (int)fx, y + fm.getAscent());
							fx += fdx;
							i++;
						}
					}
				}
			}
			
		}
		
	}
	
	private class TLabel extends JLabel
	{
		
		private TermEx term;
		
		public TLabel()
		{
			super();
			setBorder(BorderFactory.createCompoundBorder(
					BorderFactory.createEmptyBorder(-1, -1, 0, -1),
					BorderFactory.createCompoundBorder(
						BorderFactory.createLineBorder(Color.lightGray),
						BorderFactory.createEmptyBorder(2, 4, 2, 4))));
			setOpaque(true);
			addMouseListener(new MouseAdapter() {
				public void mouseClicked(MouseEvent event)
				{
					if (event.getClickCount() > 1) requestData(term);
					else if (relatedPanel != null) relatedPanel.setData(term);
				}
			});
		}
		
		public TLabel(TermEx term)
		{
			this(term, TermoPL.preferences.plainFont, CommonResources.CREAM);
		}
		
		public TLabel(TermEx term, Font font, Color background)
		{
			this();
			this.term = term;
			this.setText(term.str);
			setFont(font);
			setBackground(background);
		}
		
	}
	
	private class ArrowPanel extends JPanel
	{
		
		public ArrowPanel()
		{
			super();
			
			FontMetrics fm = getFontMetrics(TermoPL.preferences.boldFont);
			Dimension dim = new Dimension(32, fm.getHeight() + 5);
			
			setPreferredSize(dim);
		}
		
		public void paintComponent(Graphics graphics)
		{
			Graphics2D g = (Graphics2D)graphics;
			int y = getHeight() / 2;
	        Polygon arrowHead = new Polygon();
			
			g.setColor(Color.gray);
			g.drawLine(0, y, 32, y);
	        arrowHead.addPoint(0, y);
	        arrowHead.addPoint(8, y - 4);
	        arrowHead.addPoint(6, y);
	        arrowHead.addPoint(8, y + 4);
	        g.fill(arrowHead);
		}
		
	}
	
	private class SimpleComparator implements Comparator<String>
	{

		public int compare(String s1, String s2) 
		{
			double v1 = doc.getTermMap().get(s1).cvalue;
			double v2 = doc.getTermMap().get(s2).cvalue;
			
			if (v1 < v2) return 1;
			if (v1 > v2) return -1;
			return 0;
		}
		
	}

	private class TermComparator implements Comparator<String>
	{

		public int compare(String s1, String s2) 
		{
			TermEx t1 = (TermEx)doc.getTermMap().get(s1);
			TermEx t2 = (TermEx)doc.getTermMap().get(s2);
			double v1, v2;
			
			if (t1.getEquivalentTerms() == null) v1 = t1.cvalue;
			else {
				v1 = 0.0;
				for (String s : t1.getEquivalentTerms()) {
					TermEx t = (TermEx)doc.getTermMap().get(s);
					
					if (t.cvalue > v1) v1 = t.cvalue;
				}
			}
			if (t2.getEquivalentTerms() == null) v2 = t2.cvalue;
			else {
				v2 = 0.0;
				for (String s : t2.getEquivalentTerms()) {
					TermEx t = (TermEx)doc.getTermMap().get(s);
					
					if (t.cvalue > v2) v2 = t.cvalue;
				}
			}
			if (v1 < v2) return 1;
			if (v1 > v2) return -1;
			return 0;
		}
		
	}
	
}
