package termopl;

import java.awt.*;
import java.io.Serializable;
import java.util.*;

@SuppressWarnings("serial")
public class Term implements Serializable
{
	
	public static final byte DEFAULT_STATUS = -1;
	public static final byte IGNORE_STATUS  = 10;
	public static final byte UNIQUE_STATUS  = 11;
	
	private static final Color[] TERM_COLORS = {
		new Color(139, 169, 82),
		new Color(165, 194, 106),
		new Color(194, 214,	155),
		new Color(214, 227,	188),
		new Color(233, 240, 220),
		new Color(255, 255,	204),
		new Color(255, 255,	153),
		new Color(255, 255,	102),
		new Color(255, 217,	97),
		new Color(255, 169,	65),
		new Color(221, 221, 221),
		Color.white
	};

	private LinkedList<Context> contexts;
	private LinkedList<Form> forms;
	private LinkedList<DocFreq> docFreq;
	private LinkedList<SentenceRef> sentenceRef;
	private HashMap<String,Integer> formsWithTAG;
	private Term cterm;
	private byte status;
	private byte matching;
	
	public String str;
	public int freq_s;
	public int freq_in;
	public int lk;
	public int len;
	public int rank;
	public double cvalue;
	public double contrast;
	
	public Term()
	{
	}
	
	public Term(String str, int len)
	{
		this.str = str;
		this.len = len;
		freq_s = freq_in = lk = rank = 0;
		cvalue = 0.0;
		contrast = 0.0;
		forms = null;
		formsWithTAG = null;
		status = DEFAULT_STATUS;
		matching = 0;
		contexts = new LinkedList<Context>();
		docFreq = null;
		sentenceRef = null;
	}
	
	public int hashCode()
	{
		return str.hashCode();
	}
	
	public boolean equals(Object obj)
	{
		if (this == obj) return true;
		if (obj == null) return false;
		if (obj instanceof Term) {
			String s = ((Term)obj).str;
			
			return s.equals(str);
		}
		return false;
	}
	
	public Term copy()
	{
		Term cp = new Term();
		
		cp.str = str;
		cp.freq_s = freq_s;
		cp.freq_in = freq_in;
		cp.lk = lk;
		cp.len = len;
		cp.rank = rank;
		cp.cvalue = cvalue;
		cp.contrast = contrast;
		cp.contexts = contexts;
		cp.forms = forms;
		cp.formsWithTAG = formsWithTAG;
		cp.docFreq = docFreq;
		cp.sentenceRef = sentenceRef;
		cp.cterm = cterm;
		cp.status = status;
		cp.matching = matching;
		return cp;
	}
	
	public boolean isMatching()
	{
		return matching != 0;
	}
	
	public void setMatching(boolean m)
	{
		matching = (byte) (m ? 1 : 0);
	}
	
	public void addForm(Form form, int doc, boolean inner, boolean collectAllForms)
	{
		Form frm;
		String str, tag;
		int index;
		
		if (formsWithTAG == null) formsWithTAG = new HashMap<String, Integer>();
		str = form.toString();
		tag = form.getTag();
		if (tag != null) str += "\t" + tag;
		if (formsWithTAG.containsKey(str)) {
			int val = formsWithTAG.get(str);
			
			formsWithTAG.put(str,  val + 1);
		}
		else formsWithTAG.put(str, 1);
		if (forms == null) {
			forms = new LinkedList<Form>();
			forms.add(form);
			form.addOccurence(inner);
		}
		else if (collectAllForms) {
			index = forms.indexOf(form);
			if (index < 0) {
				forms.add(form);
				form.addOccurence(inner);
			}
			else {
				frm = forms.get(index);
				frm.addOccurence(inner);
			}
		}
		freq_s++;
		if (inner) freq_in++;
		if (doc >= 1) {
			DocFreq df;
			
			if (docFreq == null) docFreq = new LinkedList<DocFreq>();
			
			if (docFreq.size() > 0) {
				df = docFreq.getLast();
				if (df.docID != doc) {
					df = new DocFreq(doc);
					df.freq = 1;
					docFreq.add(df);
				}
				else df.freq++;
			}
			else {
				df = new DocFreq(doc);
				df.freq = 1;
				docFreq.add(df);
			}
		}
	}
	
	public void addForm(Form form, boolean collectAllForms)
	{
		Form frm;
		int index;
		
		if (forms == null) {
			forms = new LinkedList<Form>();
			forms.add(form);
		}
		else if (collectAllForms) {
			index = forms.indexOf(form);
			if (index < 0) forms.add(form);
			else {
				frm = forms.get(index);
				frm.addInnerOccurences(form.innerOccurences());
				frm.addStandaloneOccurences(form.standaloneOccurences());
			}
		}
		else {
			if (prefer(form, forms.getFirst())) {
				forms.removeFirst();
				forms.add(form);
			}
		}
		freq_s += form.allOccurences();
		freq_in += form.innerOccurences();
	}
	
	public void addForms(LinkedList<Form> flist, boolean collectAllForms)
	{
		for (Form form : flist) addForm(form, collectAllForms);
	}

	public boolean prefer(Form newForm, Form oldForm)
	{
		boolean n1, n2, s1, s2;
		
		n1 = isNominative(newForm);
		n2 = isNominative(oldForm);
		s1 = isSingular(newForm);
		s2 = isSingular(oldForm);
		if (n2 && s2) return false;
		if (n1 && s1) return true;
		if (n2) return false;
		if (n1) return true;
		if (s2) return false;
		if (s1) return true;
		return false;
	}
	
	public boolean isNominative(Form form)
	{
		LinkedList<MatchedToken> tokens = form.getTokens();
		
		for (MatchedToken mt : tokens) {
			if (mt.computeBaseForm) {
				if (mt.token instanceof UDToken) {
					UDToken t = (UDToken)mt.token;
					if (t.UDTag.contains("case=nom")) return true;
				}
			}
		}
		return false;
	}
	
	public boolean isSingular(Form form)
	{
		LinkedList<MatchedToken> tokens = form.getTokens();
		
		for (MatchedToken mt : tokens) {
			if (mt.computeBaseForm) {
				if (mt.token instanceof UDToken) {
					UDToken t = (UDToken)mt.token;
					if (t.UDTag.contains("number=sing")) return true;
				}
			}
		}
		return false;
	}

	public void registerForms(Term t)
	{
		freq_s += t.freq_s;
		freq_in += t.freq_in;
	}
	
	public void addContext(String lc, String rc)
	{
		if (lc != null || rc != null) {
			boolean found = false;
			
			for (Context c : contexts) {
				if (c.redundant(lc, rc)) {
					found = true;
					break;
				}
			}
			if (!found) contexts.add(new Context(lc, rc));
		}
	}
	
	public void addContext(Context ctx)
	{
		if (ctx.leftContext != null || ctx.rightContext != null) {
			boolean found = false;
			
			for (Context c : contexts) {
				if (c.redundant(ctx.leftContext, ctx.rightContext)) {
					found = true;
					break;
				}
			}
			if (!found) contexts.add(ctx);
		}
	}
	
	public void addContexts(LinkedList<Context> cntxList)
	{
		for (Context c : cntxList) addContext(c);
	}
	
	public void addSentenceRef(SentenceRef ref)
	{
		if (sentenceRef == null) {
			sentenceRef = new LinkedList<SentenceRef>();
			sentenceRef.add(ref);
		}
		else if (ref.getIndex() == 0) {
			SentenceRef lastRef = sentenceRef.getLast();
			
			if (lastRef.getFileID() != ref.getFileID() || 
				lastRef.getStartPos() != ref.getStartPos())
			{
				sentenceRef.add(ref);
			}
		}
		else sentenceRef.add(ref);
	}
	
	public void addSentenceRef(LinkedList<SentenceRef> sr, int fileIndexOffset)
	{
		if (sr != null) {
			for (SentenceRef s : sr) {
				SentenceRefEx se;
				
				if (s instanceof SentenceRefEx) {
					se = (SentenceRefEx)s;
					se.fileID += fileIndexOffset;
				}
				else {
					se = new SentenceRefEx(fileIndexOffset, s.start, s.len, s.index);
				}
				if (sentenceRef == null) sentenceRef = new LinkedList<SentenceRef>();
				else sentenceRef.add(se);
			}
		}
	}
	
	public LinkedList<SentenceRef> getSentenceRef()
	{
		return sentenceRef;
	}
	
	public LinkedList<Form> getForms()
	{
		return forms;
	}

	public HashMap<String,Integer> getFormsWithTAG()
	{
		return formsWithTAG;
	}
	
	public void trimForms()
	{
		if (forms.size() > 1) {
			Form f = forms.getFirst();
			
			forms.clear();
			forms.add(f);
		}
	}
	
	public void trimIndex()
	{
		sentenceRef = null;
	}
	
	public LinkedList<Context> getContexts()
	{
		return contexts;
	}
	
	public void deleteContexts()
	{
		contexts = null;
	}
	
	public void calc(int cntxMethod)
	{
//		int l = (len > 5 ? 5 : len);
		double log2/* = (l == 1 ? 0.1 : Math.log(l)/Math.log(2))*/;
		
		if (len == 1) log2 = 0.1;
		else if (len <= 3) log2 = Math.log(len)/Math.log(2);
		else {
			log2 = Math.max(2.0, 7.0 - len);
			log2 = Math.log(log2)/Math.log(2);
		}
		if (contexts == null) lk = 0;
		else if (cntxMethod == 1) {
			HashSet<String> sl = new HashSet<String>();
			HashSet<String> sr = new HashSet<String>();
			
			for (Context c : contexts) {
				if (c.leftContext != null) sl.add(c.leftContext);
				if (c.rightContext != null) sr.add(c.rightContext);
			}
			lk = Math.max(sl.size(), sr.size());
		}
		else if (cntxMethod == 2) {
			HashSet<String> sl = new HashSet<String>();
			HashSet<String> sr = new HashSet<String>();
			
			for (Context c : contexts) {
				if (c.leftContext != null) {
					if (!(sl.contains(c.leftContext) || sr.contains(c.leftContext))) {
						sl.add(c.leftContext);
					}
				}
				if (c.rightContext != null) {
					if (!(sl.contains(c.rightContext) || sr.contains(c.rightContext))) {
						sr.add(c.rightContext);
					}
				}
			}
			lk = Math.max(sl.size(), sr.size());
		}
		else lk = contexts.size();
		if (freq_in == 0) cvalue = log2 * freq_s;
		else cvalue = log2 * ((double)freq_s - (double)freq_in / (double)lk);
	}
	
	public void setCTerm(Term cterm)
	{
		this.cterm = cterm;
	}
	
	public Term getCTerm()
	{
		return cterm;
	}
	
	public void addDocFreq(LinkedList<DocFreq> flist)
	{
		if (flist != null) {
			if (docFreq == null) docFreq = flist;
			else docFreq.addAll(flist);
		}
	}
	
	public LinkedList<DocFreq> getDocFreq()
	{
		return docFreq;
	}
	
	public LinkedList<SentenceRef> getRef()
	{
		return sentenceRef;
	}
	
	public void setStatus(byte status)
	{
		this.status = status;
	}
	
	public byte getStatus()
	{
		return status;
	}
	
	public Color getColor()
	{
		return TERM_COLORS[status];
	}
	
}
