package termopl;

import java.awt.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.filechooser.FileNameExtensionFilter;
import eawthandler.EAWTHandler;
import desktophandler.DesktopHandler;

public class TermoPL extends Commander
{
	
	public static final int MAX_RECENT_FILES = 5;
	public static final FileNameExtensionFilter TRM_FILTER = 
		new FileNameExtensionFilter("Terms files", "trm");

	public static TermoPL application;
	public static String appLocation;
	public static boolean isMacOS = false;
	public static boolean isWindows = false;
	public static boolean batchMode = false;
	public static boolean showOne = false;
	public static boolean blockExit = false;
	public static MasterFrame mainWindow = null;
	public static JFrame dialogOwner = null;
	public static TermoPLDocument hiddenDocument = null;
	public static Preferences preferences;
	public static LinkedList<TermoPLDocument> documents;
	public static LinkedList<TermoPLWindow> zorder;
	public static HashMap<String, Term> cterms;
	public static LinkedHashMap<String, LanguageInfo> languages;
	public static int nameCounter = 0;
	
	public TermoPL()
	{
		application = this;
		appLocation = getApplicationDir();
		initApplication();
		if (!batchMode) initGUI();
	}
	
	public void initApplication()
	{
		preferences = Preferences.createPreferences(!batchMode);
		documents = new LinkedList<TermoPLDocument>();
	}
	
	public void initGUI()
	{
		String str = System.getProperty("os.name");

		initLanguages();
		zorder = new LinkedList<TermoPLWindow>();
		try {
			UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		} 
		catch (Exception e) { 
		}
		if (str != null && str.contains("Mac OS X")) {
			str = System.getProperty("java.version");
			if (str.startsWith("1.8")) {
				new EAWTHandler() {
					public void handleQuitRequest() 
					{
						exit();
					}
	
					public void handleAboutRequest() 
					{
						about();
					}
	
					public void handlePreferencesRequest() 
					{
						handlePreferences();
					}
				};
			}
			else {
				new DesktopHandler() {
					public void handleQuitRequest() 
					{
						exit();
					}
	
					public void handleAboutRequest() 
					{
						about();
					}
	
					public void handlePreferencesRequest() 
					{
						handlePreferences();
					}
				};
			}
			System.setProperty("apple.laf.useScreenMenuBar", "true");
			isMacOS = true;
			isWindows = false;
			mainWindow = new MacMasterFrame();
		}
		else {
			isMacOS = false;
			if (str.contains("Windows")) isWindows = true;
			mainWindow = new DefaultMasterFrame();
			dialogOwner = mainWindow;
		}
		setMainFrameBounds();
		Commander.setCommander(this);
		mainWindow.setVisible(true);
	}
	
	public void initLanguages()
	{
		File dir = new File(TermoPL.appLocation + File.separator + "languages");
		HashMap<String, String> map = new HashMap<String, String>();
		
		if (dir.exists()) {
			File[] list = dir.listFiles();
			
			for (File f : list) {
				if (f.isDirectory()) {
					File name = new File(f.getAbsoluteFile() + File.separator + "name");
					
					if (name.exists()) {
						BufferedReader reader;
						String line;
						
						try {
							reader = new BufferedReader(new InputStreamReader(new FileInputStream(name), "UTF8"));
							line = reader.readLine();
							line = line.trim();
							reader.close();
							
						}
						catch (IOException exception) {
							line = null;
							exception.printStackTrace();
						}
						if (line != null) {
							String[] info = line.split("\t");
							
							if (info.length == 2) {
								map.put(info[0], info[1]);
							}
						}
					}
				}
			}
		}
		if (!map.containsKey("pl")) map.put("pl", "Polish");
		
		LinkedList<Map.Entry<String, String>> list = new LinkedList<Map.Entry<String, String>>(map.entrySet());
		Collections.sort(list, new Comparator<Map.Entry<String, String> >() {
            public int compare(Map.Entry<String, String> o1, 
                               Map.Entry<String, String> o2)
            {
                return (o1.getValue()).compareTo(o2.getValue());
            }
        });		
		languages = new LinkedHashMap<String, LanguageInfo>();
		for (Map.Entry<String, String> e : list) {
            languages.put(e.getKey(), new LanguageInfo(e.getKey(), e.getValue()));
        }
		if (preferences.language != null) {
			if (!languages.containsKey(preferences.language))
				preferences.language = null;
		}
	}
	
	public void setMainFrameBounds()
	{
		if (isMacOS) {
			mainWindow.setSize(0, 0);
			mainWindow.setLocation(-1, -1);
		}
		else {
			Rectangle rect = preferences.getBounds(0);
			
			if (rect.width == 0) rect = WindowUtils.getScreenRect();
			WindowUtils.setBoundsRelativeToScreen(mainWindow, rect, true);
		}
	}
	
	public void startBatchProcessing(BatchParameters bp)
	{
		LinkedList<File> sourceFiles = null;
		LinkedList<File> trmFiles = null;
		TermoPLDocument doc = null;
		
		for (File f : collectFiles(bp.inputFiles)) {
			if (f.getName().endsWith(".trm")) {
				if (trmFiles == null) trmFiles = new LinkedList<File>();
				trmFiles.add(f);
			}
			else {
				if (sourceFiles == null) sourceFiles = new LinkedList<File>();
				sourceFiles.add(f);
			}
		}
		if (trmFiles != null) {
			for (File f : trmFiles) {
				if (doc == null) {
					open(f);
					if (documents.size() > 0) doc = documents.getLast();
				}
				else doc.merge(f);
			}
		}
		if (sourceFiles != null) {
			if (doc == null) {
				String[] files = new String[sourceFiles.size()];
				int i = 0;
				
				for (File f : sourceFiles) files[i++] = f.getPath();
				bp.inputFiles = files;
				doc = new TermoPLDocument(this, bp);
				doc.startBatchProcessing();
			}
			else doc.addNewResults(sourceFiles.toArray(new File[0]));
		}
		if (doc != null) {
			if (bp.output != null) doc.save(new File(bp.output));
			if (bp.export != null) doc.export(new File(bp.export), TermoPLDocument.EXPORT_RESULTS);
			if (bp.exportForms != null) doc.export(new File(bp.exportForms), TermoPLDocument.EXPORT_FORMS);
			if (bp.exportSentences != null) doc.export(new File(bp.exportSentences), TermoPLDocument.EXPORT_SENTENCES);
			if (bp.exportGroups != null) doc.export(new File(bp.exportGroups), TermoPLDocument.EXPORT_GROUPS);
			doc.waitForSaveThreadsToFinish();
		}
	}
	
	public void executeCommand(int commandID, int modifiers)
	{
		switch (commandID) {
			case Command.NEW_COMMAND:
				createNewDocument();
				break;
			case Command.OPEN_COMMAND:
				open();
				break;
			case Command.WORKSPACE_COMMAND:
				selectWorkspace();
				break;
			case Command.SELECT_CONTRASTIVE_TERMS_COMMAD:
				selectContrastiveCorpus();
				break;
			case Command.BASE_FORMS_COMMAND:
				preferences.calculateBaseForms = !preferences.calculateBaseForms;
				preferences.setModified(true);
				break;
			case Command.COLLECT_COMMAND:
				preferences.collectAllForms = !preferences.collectAllForms;
				preferences.setModified(true);
				break;
			case Command.INDEX_COMMAND:
				preferences.makeIndex = !preferences.makeIndex;
				preferences.setModified(true);
				break;
			case Command.PREFERENCES_COMMAND:
				changePreferences();
				break;
			case Command.INCREASE_COMMAND:
				increaseFont();
				break;
			case Command.DECREASE_COMMAND:
				decreaseFont();
				break;
			case Command.SHOW_ONE_ALL:
				showOneAll();
				break;
			case Command.TILE_COMMAND:
				tile();
				break;
			case Command.CASCADE_COMMAND:
				cascade();
				break;
			case Command.ABOUT_COMMAND:
				about();
				break;
			case Command.HELP_COMMAND:
				help();
				break;
			case Command.EXIT_COMMAND:
				exit();
				break;
			default: super.executeCommand(commandID, modifiers);
		}
	}
	
	public void findMenuStatus(JMenu menu)
	{
	}
	
	public void findCommandStatus(JMenuItem item)
	{
		Command command = (Command)item.getAction();
		
		switch (command.getCommandID()) {
			case Command.MERGE_COMMAND:
			case Command.CLOSE_COMMAND:
			case Command.SAVE_COMMAND:
			case Command.SAVE_AS_COMMAND:
			case Command.EXPORT_COMMAND:
			case Command.EXPORT_FORMS_COMMAND:
			case Command.EXPORT_SENTENCES_COMMAND:
			case Command.EXPORT_GROUPS_COMMAND:
			case Command.EXTRACT_COMMAND:
			case Command.COMPARE_COMMAND:
			case Command.SELECT_FILES_COMMAND:
			case Command.FORMS_COMMAND:
			case Command.SENTENCES_COMMAND:
			case Command.GROUPS_COMMAND:
				item.setEnabled(false);
				break;
			case Command.SELECT_CONTRASTIVE_TERMS_COMMAD:
			case Command.BASE_FORMS_COMMAND:
				item.setSelected(preferences.calculateBaseForms);
				item.setEnabled(!preferences.useCustomTagset);
				break;
			case Command.COLLECT_COMMAND:
				item.setSelected(preferences.collectAllForms);
				break;
			case Command.INDEX_COMMAND:
				item.setSelected(preferences.makeIndex);
				break;
			case Command.INCREASE_COMMAND:
				item.setEnabled(preferences.fontSize < Preferences.MAX_FONT_SIZE);
				break;
			case Command.DECREASE_COMMAND:
				item.setEnabled(preferences.fontSize > Preferences.MIN_FONT_SIZE);
				break;
			case Command.SHOW_ONE_ALL:
				if (showOne) item.setText("Show All Documents");
				else item.setText("Show One Document");
				break;
			case Command.TILE_COMMAND:
			case Command.CASCADE_COMMAND:
				int n = 0;
				
				for (TermoPLDocument doc : documents) {
					for (TermoPLWindow w : doc.getWindows()) {
						if (((Component)w).isVisible()) n++;
					}
				}
				item.setEnabled(n > 1);
				break;
			default: super.findCommandStatus(item);
		}
	}
	
	public void createNewDocument()
	{
		if (showOne) {
			Commander cmd = Commander.getCommander();
			
			if (cmd instanceof TermoPLDocument) hideDocument((TermoPLDocument)cmd);
		}
		
		TermoPLDocument doc = new TermoPLDocument(this);
		
		documents.add(doc);
		MenuFactory.addToDocumentSwitchMenu(doc);
		MenuFactory.resetSelection(doc, zorder.getFirst());
		doc.selectFiles();
	}
	
	public void open()
	{
		JFileChooser chooser = new JFileChooser();
		
		chooser.setDialogTitle("Open");
		chooser.setAcceptAllFileFilterUsed(false);
		chooser.setFileFilter(TRM_FILTER);
		if (preferences.filePath != null) {
			File file = new File(preferences.filePath);
			
			if (file.exists()) chooser.setCurrentDirectory(file);
		}
		if (chooser.showOpenDialog(TermoPL.dialogOwner) == JFileChooser.APPROVE_OPTION) {
			File file = chooser.getSelectedFile();
			
			preferences.filePath = file.getParentFile().getPath();
			open(file);
		}
	}
	
	public void open(String path)
	{
		File file = new File(path);
		
		if (file.exists()) open(file);
		else {
			if (batchMode) {
				System.out.println("File " + file.getName() + " does not exist.");
			}
			else {
				removeRecentFile(path);
				JOptionPane.showMessageDialog(TermoPL.dialogOwner, "File " + file.getName() + " does not exist.", "Error", JOptionPane.ERROR_MESSAGE);
			}
		}
	}
	
	public void open(File file)
	{
		if (!batchMode) {
			if (showOne) {
				Commander cmd = Commander.getCommander();
				
				if (cmd instanceof TermoPLDocument) {
					hiddenDocument = (TermoPLDocument)cmd;
					hideDocument(hiddenDocument);
				}
			}
		}
		new TermoPLDocument(this, file);
	}
	
	public void finishOpening(TermoPLDocument doc)
	{
		String path = doc.getPath();
		
		if (doc.isLoaded()) {
			documents.add(doc);
			if (!batchMode) {
				updateRecentFiles(path);
				MenuFactory.addToDocumentSwitchMenu(doc);
				MenuFactory.resetSelection(doc, zorder.getFirst());
			}
		}
		else {
			if (!batchMode) {
				if (doc.isCorrupted()) removeRecentFile(path);
				else updateRecentFiles(path);
				if (showOne) {
					if (hiddenDocument != null) showDocument(hiddenDocument);
				}
			}
		}
		hiddenDocument = null;
	}
	
	public void selectWorkspace()
	{
		Workspace workspace = new Workspace(preferences.workSpace);
		int res = workspace.doDialog();
		
		if (res == Workspace.OK) {
			preferences.workSpace = preferences.filePath = workspace.getCurrentWorkspace();
			preferences.setModified(true);
		}
		workspace.dispose();
	}
	
	public void closeDocument(TermoPLDocument doc)
	{
		documents.remove(doc);
		if (!batchMode) {
			MenuFactory.removeFromDocumentSwitchMenu(doc);
			if (documents.isEmpty()) Commander.setCommander(this);
			else if (showOne && !documents.isEmpty()) {
				TermoPLWindow wnd = zorder.getFirst();
				TermoPLDocument d = wnd.getDocument();
				
				showDocument(d); 
			}
		}
	}

	public void removeRecentFile(String path)
	{
		preferences.recentFiles.remove(path);
		preferences.setModified(true);
		MenuFactory.removeRecentFile(path);
	}
	
	public void updateRecentFiles(String path)
	{
		preferences.recentFiles.remove(path);
		preferences.recentFiles.addFirst(path);
		preferences.setModified(true);
		if (preferences.recentFiles.size() > MAX_RECENT_FILES) preferences.recentFiles.removeLast();
		MenuFactory.updateRecentFiles(path);
	}
	
	public void handlePreferences()
	{
		Commander.getCommander().executeCommand(Command.PREFERENCES_COMMAND, 0);
	}
	
	public void changePreferences()
	{
		Options options = new Options(preferences);
		
		options.doDialog();
		options.dispose();
		if (preferences.isModified()) preferences.save();
	}
	
	public void about()
	{
		About about = new About();
		
		about.doDialog();
	}
	
	public void help()
	{
		if (Desktop.isDesktopSupported()) {
		    try {
		        File manual = new File(TermoPL.appLocation + File.separator + "TermoPL-user-manual.pdf");
		        if (manual.exists()) Desktop.getDesktop().open(manual);
		        else JOptionPane.showMessageDialog(dialogOwner, "TermoPL user manual is missing.", "File not found", JOptionPane.ERROR_MESSAGE);
		    } 
		    catch (IOException ex) {
		    	JOptionPane.showMessageDialog(dialogOwner, "No application registered for PDFs.", "Error", JOptionPane.ERROR_MESSAGE);
		    }
		}	
	}
	
	public void increaseFont()
	{
		preferences.fontSize++;
		changeFontSize();
	}
	
	public void decreaseFont()
	{
		preferences.fontSize--;
		changeFontSize();
	}
	
	public void changeFontSize()
	{
		Font f;
		
		f = preferences.plainFont.deriveFont((float)preferences.fontSize);
		preferences.plainFont = f;
		f = preferences.boldFont.deriveFont((float)preferences.fontSize);
		preferences.boldFont = f;
		for (TermoPLDocument doc : documents) doc.changeFontSize();
		preferences.setModified(true);
	}
	
	public void showDocument(TermoPLDocument doc)
	{
		TermoPLWindow[] ws = zorder(doc).toArray(new TermoPLWindow[0]);
		
		for (int i = ws.length - 1; i >= 0; i--) {
			if (ws[i].getDocument() == doc) ws[i].select();
		}
	}
	
	public void hideDocument(TermoPLDocument doc)
	{
		TermoPLWindow[] ws = zorder(doc).toArray(new TermoPLWindow[0]);
		
		for (TermoPLWindow wnd : ws) {
			if (wnd.getDocument() == doc) wnd.setVisible(false);
		}
	}
	
	public void bringToFront(TermoPLDocument doc)
	{
		if (showOne) {
			Commander cmd = Commander.getCommander();
			
			if (cmd instanceof TermoPLDocument) {
				TermoPLDocument d = (TermoPLDocument)cmd;
				
				if (d != doc) hideDocument(d);
			}
		}
		showDocument(doc);
	}
	
	public void showOneAll()
	{
		showOne = !showOne;
		if (zorder.size() > 0) {
			TermoPLWindow[] ws = zorder.toArray(new TermoPLWindow[0]);
			
			if (showOne) {
				if (Commander.getCommander() != this) {
					TermoPLDocument doc = (TermoPLDocument)Commander.getCommander();
					
					for (TermoPLWindow wnd : ws) {
						if (wnd.getDocument() != doc) wnd.setVisible(false);
					}
				}
			}
			else {
				for (int i = ws.length - 1; i >= 0; i--) ws[i].setVisible(true);
			}
		}
	}
	
	public void exit()
	{
		if (!documents.isEmpty()) {
			TermoPLDocument[] docs = documents.toArray(new TermoPLDocument[0]);
			
			for (TermoPLDocument d : docs) d.close();
		}
		if (documents.isEmpty()) {
			Rectangle rect = preferences.getBounds(0);
			
			if (!mainWindow.getBounds().equals(rect)) {
				preferences.setBounds(0, mainWindow.getBounds());
				preferences.setModified(true);
			}
			if (preferences.isModified()) preferences.save();
			System.exit(0);
		}
	}
	
	public void tile()
	{
		WndLocation.tile();
	}
	
	public void cascade()
	{
		WndLocation.cascade();
	}
	
	public static void changeZOrder(TermoPLWindow wnd)
	{
		zorder.remove(wnd);
		zorder.addFirst(wnd);
	}
	
	public static LinkedList<TermoPLWindow> zorder(TermoPLDocument doc)
	{
		LinkedList<TermoPLWindow> zo = new LinkedList<TermoPLWindow>();
		
		for (TermoPLWindow wnd : zorder) {
			if (wnd.getDocument() == doc) zo.add(wnd);
		}
		return zo;
	}
	
	public static File[] collectFiles(File[] selected)
	{
		if (selected == null) return null;
		
		LinkedList<File> files = new LinkedList<File>();
		
		for (File f : selected) {
			if (f.isDirectory()) {
				File[] list = collectFiles(f.listFiles());
				
				if (list != null) {
					for (File e : list) files.add(e);
				}
			}
			else files.add(f);
		}
		if (files.isEmpty()) return null;
		return files.toArray(new File[0]);
	}
	
	public static File[] collectFiles(String[] selected)
	{
		if (selected == null) return null;
		
		LinkedList<File> files = new LinkedList<File>();
		
		for (String path : selected) {
			File f = new File(path);
			
			if (f.exists()) {
				if (f.isDirectory()) {
					File[] list = collectFiles(f.listFiles());
					
					if (list != null) {
						for (File e : list) files.add(e);
					}
				}
				else files.add(f);
			}
			else System.out.println("File does not exist -- " + path);		
		}
		if (files.isEmpty()) return null;
		return files.toArray(new File[0]);
	}
	
	public static String selectContrastiveCorpus()
	{
		return selectContrastiveCorpus(dialogOwner);
	}
	
	public static String selectContrastiveCorpus(Component owner)
	{
		JFileChooser chooser = new JFileChooser();
		
		chooser.setDialogTitle("Select Contrastive Corpus");
		chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		chooser.setFileFilter(TRM_FILTER);
		chooser.setMultiSelectionEnabled(false);
		if (preferences.filePath != null) {
			File file = new File(preferences.filePath);
			
			if (file.exists()) chooser.setCurrentDirectory(file);
		}
		if (chooser.showOpenDialog(owner) == JFileChooser.APPROVE_OPTION) {
			File file = chooser.getSelectedFile();
			
			preferences.filePath = file.getParentFile().getPath();
			preferences.setModified(true);
			return file.getPath();
		}
		return null;
	}
	
	public static TermoPL getApplication()
	{
		return application;
	}
	
	public static String getFileName(String path)
	{
		return getFileName(new File(path));
	}
	
	public static String getFileName(File file)
	{
		String name = file.getName();
		int index = name.lastIndexOf('.');
		
		if (index < 0) return name;
		if (index == 0) return name.substring(1);
		return name.substring(0, index);
	}
	
	public static String getDisplayName(String docName)
	{
		boolean found = false;
		
		for (TermoPLDocument doc : documents) {
			if (doc.getDocumentName().equals(docName)) {
				found = true;
				break;
			}
		}
		if (found) {
			return docName + " - [" + ++nameCounter + "]";
		}
		return docName;
	}
	
	public String getApplicationDir()
	{
		File file = new File(getClass().getProtectionDomain().getCodeSource().getLocation().getPath());
		
		return file.getParent();
	}
	
	public static void main(String[] args) 
	{
		if (args.length > 0) batchMode = true;
		else batchMode = false;
		
		TermoPL te = new TermoPL();
		BatchParameters bp = null;
		
		if (batchMode) {
			bp = new BatchParameters(args);
			if (!bp.ok) {
				System.out.println(bp.error);
				System.exit(1);
			}
		}
		if (batchMode && bp.ok) te.startBatchProcessing(bp);
	}
	
}
