package termopl;

import java.io.*;
import java.util.*;
import javax.xml.stream.*;
import javax.xml.stream.events.*;

public class WordNet 
{

	/* Relation types
		1 - S - synonimia
	   10 - S - hiponimia
	   11 - S - hiperonimia
	   51 - L - nosiciel stanu lub cechy
	   52 - L - stan lub cecha
	   61 - L - synonimia midzyparadygmatyczna
	   62 - L - synonimia midzyparadygmatyczna
	   63 - L - synonimia midzyparadygmatyczna
	  108 - S - fuzzynimia synsetów
	  141 - L - synonimia midzyparadygmatyczna
	  142 - L - synonimia midzyparadygmatyczna
	  148 - L - podobieństwo
	  149 - L - charakteryzowanie
	  169 - L - synonimia midzyparadygmatyczna dla relacyjnych
	  242 - L - rola: materiał
	  244 - L - synonimia międzyparadygmatyczna
	*/
	
	public static final String SP_GENERAL_ENTITY_SIZE_LIMIT = "jdk.xml.maxGeneralEntitySizeLimit";
	public static final String SP_TOTAL_ENTITY_SIZE_LIMIT = "jdk.xml.totalEntitySizeLimit";
	
	public static HashMap<String, WordNet> wordNets = new HashMap<String, WordNet>();

	public static final String NOUN      = "rzeczownik";
	public static final String ADJECTIVE = "przymiotnik";
	public static final String VERB      = "czasownik";
	public static final String ADVERB    = "przys\u0142\u00F3wek";
	
	public static final int MAXDEPTH   = 1; 
	public static final int EXACT      = 0;
	public static final int SYNONYMY   = 1;
	public static final int HIPERONYMY = 11;
	
	private File input;
	private HashMap<String, LinkedList<LexicalUnit>> words;
	private HashMap<Integer, LexicalUnit> wordIndex;
	private HashMap<Integer, Synset> synsetIndex;
	private HashMap<Integer, LinkedList<LexicalRelation>> lexicalRelationIndex;
	private HashMap<Integer, LinkedList<SynsetRelation>> synsetRelationIndex;
	private LinkedList<Synset> synsets;
	private LinkedList<LexicalRelation> lexicalRelations;
	private LinkedList<SynsetRelation> synsetRelations;
	private RelationTypes lexRelationTypes;
	private RelationTypes synsetRelationTypes;
	private Synset currentSynset;
	private ArrayList<Integer> unitIDs;
	private boolean unitIDStarted;
	private boolean ignorePWN;
	
	public WordNet(File input, boolean ignorePWN)
	{
		this.input = input;
		this.ignorePWN = ignorePWN;
		currentSynset = null;
		unitIDs = null;
		unitIDStarted = false;
		lexRelationTypes = new RelationTypes();
		lexRelationTypes.setRelationTypes(51, 52, 61, 62, 63, 141, 142, 148, 149, 169, 242, 244);
		synsetRelationTypes = new RelationTypes();
		synsetRelationTypes.setRelationTypes(10, 11, 108);
	}
	
	public WordNet(File input)
	{
		this(input, true);
	}
	
	public static WordNet getWordNet(String wordNetPath)
	{
		return wordNets.get(wordNetPath);
	}
	
	public static WordNet createWordNet(TermoPLDocument doc, String wordNetPath)
	{
		synchronized(wordNetPath) {
			WordNet wdn = getWordNet(wordNetPath);
			
			if (wdn == null) {
				File f = new File(wordNetPath);
				
				if (f.exists()) {
					wdn = new WordNet(f);
					wdn.load(doc);
					wordNets.put(wordNetPath, wdn);
					return wdn;
				}
				return null;
			}
			return wdn;
		}
	}
	
	public LinkedList<WordReplacement> getRelatedWords(String w, String pos)
	{
		LinkedList<LexicalUnit> lexUnits = words.get(w);
		
		if (lexUnits != null) {
			LinkedList<WordReplacement> relatedWords = new LinkedList<WordReplacement>();
			
			for (LexicalUnit lu : lexUnits) {
				if (lu.pos.equals(pos)) {
					for (Synset s : synsets) {
						boolean found = false;
						
						for (int id : s.lexUnits) {
							if (id == lu.id) {
								found = true;
								break;
							}
						}
						if (found) {
							getLexUnitsFromSynset(s, w, relatedWords);
							getLexicalyRelated(lu, w, relatedWords);
							getSynsetRelated(s, w, 0, relatedWords);
						}
					}
				}
			}
			return relatedWords;
		}
		return null;
	}
	
	public void getLexUnitsFromSynset(Synset synset, String w, LinkedList<WordReplacement> relatedWords)
	{
		for (int id : synset.lexUnits) {
			String name = wordIndex.get(id).name;
			
			if (!name.equals(w)) WordReplacement.addReplacement(w, name, SYNONYMY, relatedWords);
		}
	}
	
	public void getLexicalyRelated(LexicalUnit lexUnit, String w, LinkedList<WordReplacement> relatedWords)
	{
		LinkedList<LexicalRelation> lexRelations = lexicalRelationIndex.get(lexUnit.id);
		
		if (lexRelations != null) {
			for (LexicalRelation r : lexRelations) {
				LexicalUnit lu;
				String name;
				
				if (lexUnit.id == r.child) lu = wordIndex.get(r.parent);
				else lu = wordIndex.get(r.child);
				name = lu.name;
				
				if (!name.equals(w)) WordReplacement.addReplacement(w, name, r.relation, relatedWords);
			}
		}
	}
	
	public void getSynsetRelated(Synset synset, String w, int depth, LinkedList<WordReplacement> relatedWords)
	{
		LinkedList<SynsetRelation> synsetRelations = synsetRelationIndex.get(synset.id);
		
		if (synsetRelations != null) {
			for (SynsetRelation r : synsetRelations) {
				Synset s = synsetIndex.get(r.parent);
				
				for (int id : s.lexUnits) {
					LexicalUnit lu = wordIndex.get(id);
					String name = lu.name;
					
					if (!name.equals(w)) WordReplacement.addReplacement(w, name, r.relation, relatedWords);
//					if (r.relation == HIPERONYMY) getSynsetRelated(s, w, depth + 1, relatedWords, new LinkedList<Synset>());
				}
			}
		}
	}
	
/*	public void getSynsetRelated(Synset synset, String w, int depth, 
			LinkedList<WordReplacement> relatedWords, LinkedList<Synset> synsets)
	{
		LinkedList<SynsetRelation> synsetRelations = synsetRelationIndex.get(synset.id);
		
		if (synsetRelations != null) {
			for (SynsetRelation r : synsetRelations) {
				if (r.relation == HIPERONYMY) {
					Synset s = synsetIndex.get(r.parent);
					
					if (!synsets.contains(s)) {
						synsets.add(s);
						for (int id : s.lexUnits) {
							LexicalUnit lu = wordIndex.get(id);
							String name = lu.name;
							
							if (!name.equals(w)) WordReplacement.addReplacement(w, name, HIPERONYMY, relatedWords);
							if (depth < MAXDEPTH) getSynsetRelated(s, w, depth + 1, relatedWords, synsets);
						}
					}
				}
			}
		}
	}*/
	
	public void load(TermoPLDocument doc)
	{
		System.setProperty(SP_GENERAL_ENTITY_SIZE_LIMIT, "0");	
		System.setProperty(SP_TOTAL_ENTITY_SIZE_LIMIT, "0");	
		
		XMLInputFactory inputFactory = XMLInputFactory.newInstance();
		
		words = new HashMap<String, LinkedList<LexicalUnit>>();
		wordIndex = new HashMap<Integer, LexicalUnit>();
		synsetIndex = new HashMap<Integer, Synset>();
		synsets = new LinkedList<Synset>();
		lexicalRelations = new LinkedList<LexicalRelation>();
		synsetRelations = new LinkedList<SynsetRelation>();
		try {
			XMLEventReader reader = inputFactory.createXMLEventReader(new FileInputStream(input));
			
			while (reader.hasNext() && !doc.isCancelled()) {
				XMLEvent event = reader.nextEvent();
				String elementName;
				
				switch (event.getEventType()) {
					case XMLStreamConstants.START_ELEMENT : 
						StartElement startElement = (StartElement)event;
						
						elementName = startElement.getName().getLocalPart();
						if (elementName.equals("lexical-unit")) parseLexicalUnit(startElement);
						else if (elementName.equals("synset")) parseSynset(startElement);
						else if (elementName.equals("unit-id")) parseUnitID(startElement);
						else if (elementName.equals("lexicalrelations")) parseLexicalRelations(startElement);
						else if (elementName.equals("synsetrelations")) parseSynsetRelations(startElement);
						break;
					case XMLStreamConstants.END_ELEMENT :
						EndElement endElement = (EndElement)event;
						
						elementName = endElement.getName().getLocalPart();
						if (elementName.equals("synset")) {
							if (currentSynset != null) {
								int[] lexUnits = new int[unitIDs.size()];
								
								if (lexUnits.length > 0) {
									for (int i = 0; i < lexUnits.length; i++)
										lexUnits[i] = unitIDs.get(i).intValue();
									currentSynset.lexUnits = lexUnits;
									synsets.add(currentSynset);
									synsetIndex.put(currentSynset.id, currentSynset);
								}
							}
							unitIDs = null;
							currentSynset = null;
						}
						else if (elementName.equals("unit-id")) {
							unitIDStarted = false;
						}
						break;
					case XMLStreamConstants.CHARACTERS : 
						Characters characters = (Characters)event;
						if (unitIDStarted) readCharacters(characters);
				}
			}
		}
		catch (Exception exception) {
			exception.printStackTrace();
		}
		System.clearProperty(SP_GENERAL_ENTITY_SIZE_LIMIT);
		System.clearProperty(SP_TOTAL_ENTITY_SIZE_LIMIT);
	}
	
	@SuppressWarnings("rawtypes")
	public void parseLexicalUnit(StartElement element)
	{
		LexicalUnit unit = new LexicalUnit();
		
		for (Iterator it = element.getAttributes(); it.hasNext();) {
			Attribute attr = (Attribute)it.next();
			String name = attr.getName().getLocalPart();
			
			if (name.equals("id")) unit.id = Integer.parseInt(attr.getValue());
			else if (name.equals("name")) unit.name = attr.getValue();
			else if (name.equals("pos")) unit.pos = attr.getValue();
		}
		
		if (!ignorePWN || !unit.pos.endsWith("pwn")) {
			LinkedList<LexicalUnit> lexUnits = words.get(unit.name);
			
			if (lexUnits == null) {
				lexUnits = new LinkedList<LexicalUnit>();
				words.put(unit.name, lexUnits);
			}
			lexUnits.add(unit);
			wordIndex.put(unit.id, unit);
		}
	}
	
	@SuppressWarnings("rawtypes")
	public void parseSynset(StartElement element)
	{
		Synset synset = new Synset();
		
		for (Iterator it = element.getAttributes(); it.hasNext();) {
			Attribute attr = (Attribute)it.next();
			String name = attr.getName().getLocalPart();
			
			if (name.equals("id")) synset.id = Integer.parseInt(attr.getValue());
		}
		currentSynset = synset;
		unitIDs = new ArrayList<Integer>();
	}
	
	public void parseUnitID(StartElement element)
	{
		unitIDStarted = true;
	}
	
	@SuppressWarnings("rawtypes")
	public void parseLexicalRelations(StartElement element)
	{
		LexicalRelation relation = new LexicalRelation();
		boolean addToIndex = false;
		
		for (Iterator it = element.getAttributes(); it.hasNext();) {
			Attribute attr = (Attribute)it.next();
			String name = attr.getName().getLocalPart();
			
			if (name.equals("parent")) relation.parent = Integer.parseInt(attr.getValue());
			else if (name.equals("child")) relation.child = Integer.parseInt(attr.getValue());
			else if (name.equals("relation")) relation.relation = Integer.parseInt(attr.getValue());
		}
		if (ignorePWN) {
			if (wordIndex.get(relation.parent) != null && wordIndex.get(relation.child) != null) {
				if (lexRelationTypes.contains(relation.relation)) {
					lexicalRelations.add(relation);
					addToIndex = true;
				}
			}
		}
		else if (lexRelationTypes.contains(relation.relation)) {
			lexicalRelations.add(relation);
			addToIndex = true;
		}
		if (addToIndex) {
			LinkedList<LexicalRelation> lexRelations;
			
			if (lexicalRelationIndex == null) 
				lexicalRelationIndex = new HashMap<Integer, LinkedList<LexicalRelation>>();
			lexRelations = lexicalRelationIndex.get(relation.child);
			if (lexRelations == null) {
				lexRelations = new LinkedList<LexicalRelation>();
				lexicalRelationIndex.put(relation.child, lexRelations);
			}
			lexRelations.add(relation);
			lexRelations = lexicalRelationIndex.get(relation.parent);
			if (lexRelations == null) {
				lexRelations = new LinkedList<LexicalRelation>();
				lexicalRelationIndex.put(relation.parent, lexRelations);
			}
			lexRelations.add(relation);
		}
	}
	
	@SuppressWarnings("rawtypes")
	public void parseSynsetRelations(StartElement element)
	{
		SynsetRelation relation = new SynsetRelation();
		boolean addToIndex = false;
		
		for (Iterator it = element.getAttributes(); it.hasNext();) {
			Attribute attr = (Attribute)it.next();
			String name = attr.getName().getLocalPart();
			
			if (name.equals("parent")) relation.parent = Integer.parseInt(attr.getValue());
			else if (name.equals("child")) relation.child = Integer.parseInt(attr.getValue());
			else if (name.equals("relation")) relation.relation = Integer.parseInt(attr.getValue());
		}
		if (ignorePWN) {
			if (synsetIndex.get(relation.parent) != null && synsetIndex.get(relation.child) != null) {
				if (synsetRelationTypes.contains(relation.relation)) {
					synsetRelations.add(relation);
					addToIndex = true;
				}
			}
		}
		else if (synsetRelationTypes.contains(relation.relation)) {
			synsetRelations.add(relation);
			addToIndex = true;
		}
		if (addToIndex) {
			LinkedList<SynsetRelation> synsetRelations;
			
			if (synsetRelationIndex == null) 
				synsetRelationIndex = new HashMap<Integer, LinkedList<SynsetRelation>>();
			synsetRelations = synsetRelationIndex.get(relation.child);
			if (synsetRelations == null) {
				synsetRelations = new LinkedList<SynsetRelation>();
				synsetRelationIndex.put(relation.child, synsetRelations);
			}
			synsetRelations.add(relation);
		}
	}
	
	public void readCharacters(Characters characters)
	{
		if (!characters.isIgnorableWhiteSpace() && !characters.isWhiteSpace()) {
			String data = characters.getData();
			
			if (data.length() != 0) {
				Integer id = Integer.parseInt(data);
				
				if (ignorePWN) {
					if (wordIndex.get(id) != null) unitIDs.add(id);
				}
				else unitIDs.add(id);
			}
		}
	}
	
	public static boolean isSynonymic(int relType)
	{
		if (relType < 0) relType = -relType;
		return relType == 1 || relType == 169;
	}
	
	public static String getName(int relType)
	{
		switch (relType) {
			case 1: return "synonymy";
			case 10: return "hyponymy";
			case 11: return "hyperonymy";
			case 51: return "characterized by feature|state" ;
			case 52: return "feature|state";
			case 108: return "fuzzynymy";
			case 148: return "similarity";
			case 149: return "description";
			case 242: return "role: material";
			case 61:
			case 62:
			case 63:
			case 141:
			case 142:
			case 169:
			case 244: return "inter-paradigmatic synonymy";
			default: return "?";
		}
	}
	
	private class RelationTypes 
	{
		
		private HashSet<Integer> relationTypes;
		
		public RelationTypes()
		{
			relationTypes = null;
		}
		
		public boolean contains(int rel)
		{
			if (relationTypes == null) return true;
			return relationTypes.contains(rel);
		}
		
		public void setRelationTypes(int... types)
		{
			relationTypes = null;
			addRelationTypes(types);
		}
		
		public void addRelationTypes(int... types)
		{
			if (relationTypes == null) relationTypes = new HashSet<Integer>();
			
			for (int i = 0; i < types.length; i++) 
				relationTypes.add(types[i]);
		}
		
	}

}
