package termopl;

import java.io.*;
import java.lang.reflect.*;
import java.util.*;

public class GParser extends Parser
{
	
	private File grammarFile;
	private Tagset tagset;
	private LinkedList<Rule> rules;
	private LinkedList<String> headSymbols;
	private Template template;
	private HashMap<String, Tester> tsym;
	
	public GParser(File file, Tagset tagset, Object owner)
	{
		super(owner);
		grammarFile = file;
		this.tagset = tagset;
		rules = new LinkedList<Rule>();
		headSymbols = new LinkedList<String>();
		tsym = new HashMap<String, Tester>();
	}
	
	public void parse()
	{
		try {
			boolean head;
			
			reader = new BufferedReader(new InputStreamReader(
				new FileInputStream(grammarFile), "UTF8"));
			getSymbol();
			do {
				if (symbol == Symbol.CIRCMFX) {
					head = true;
					getSymbol();
				}
				else head = false;
				if (symbol == Symbol.IDENTIFIER) parseRule(head);
				else {
					symbol = Symbol.UNKNOWN;
					error(ParserError.SYMBOL_EXPECTED);
					skip(Symbol.IDENTIFIER);
				}
			}
			while (!tooManyErrors() && symbol != Symbol.EMPTY);
			
			if (noErrors()) {
				if (rules.isEmpty()) error(ParserError.NO_RULES_DEFINED);
				else {
					Collection<Tester> testers = tsym.values();
					boolean found = false;
					
					for (Tester t : testers) {
						if (t.hasAnythingToDo()) {
							found = true;
							break;
						}
					}
					if (!found) error(ParserError.NO_TESTS_DEFINED);
				}
			}
			if (noErrors()) template = compileRules();
		}
		catch (Exception exception) {
			try {
				error(exception.getStackTrace().toString());
			} 
			catch (ParserException e) {
			}
		}
		reportErrors();
	}
	
	public void parseRule(boolean head) throws ParserException
	{
		Rule rule = new Rule(head);
		
		parseSymbol(rule);
		if (symbol == Symbol.SCOLON) getSymbol();
		else error(ParserError.SEMICOLON_EXPECTED);
		if (noErrors()) {
			if (rule.subst != null) rules.add(rule);
			if (head) headSymbols.add(rule.id);
		}
	}
	
	public void parseSymbol(Rule rule) throws ParserException
	{
		String sym = identifier;
		Tester tester = tsym.get(sym);
		
		rule.id = sym;
		if (tester == null) tester = new Tester();
		getSymbol();
		parseCondition(tester);
		if (noErrors()) tsym.put(sym, tester);
		if (symbol == Symbol.COLON) {
			if (rule.head) {
				error(ParserError.HEAD_NON_TERMINAL);
				skip(Symbol.SCOLON);
			}
			else {
				getSymbol();
				parseTemplate(rule);
			}
		}
	}
	
	public void parseCondition(Tester tester) throws ParserException
	{
		if (symbol == Symbol.LBRCT) {
			getSymbol();
			parseTestList(tester);
			if (symbol == Symbol.RBRCT) getSymbol();
			else {
				error(ParserError.RBRCT_EXPECTED);
				skip(Symbol.COLON, Symbol.SCOLON, Symbol.IDENTIFIER);
			}
		}
	}
	
	public void parseTestList(Tester tester) throws ParserException
	{
		boolean done = false;
		
		do {
			parseTest(tester);
			if (symbol == Symbol.SCOLON) getSymbol();
			else done = true;
		} while (!tooManyErrors() && !done);
	}
	
	public void parseTest(Tester tester) throws ParserException
	{
		if (symbol == Symbol.IDENTIFIER) {
			String id = identifier;
			
			getSymbol();
			if (symbol == Symbol.EQ || symbol == Symbol.NEQ || 
				symbol == Symbol.MATCH || symbol == Symbol.NMATCH) 
			{
				Test test = new Test();
				
				if (symbol == Symbol.EQ) test.op = Test.EQ;
				else if (symbol == Symbol.NEQ) test.op = Test.NEQ;
				else if (symbol == Symbol.MATCH) test.op = Test.MATCH;
				else if (symbol == Symbol.NMATCH) test.op = Test.NMATCH;
				
				test = getTest(id, test, Token.class);
				parseArguments(test);
				tester.add(test);
			}
			else {
				String def = tagset.getDefinition(id);
				
				if (def != null) {
					String[] cats = def.split(",");
					
					for (String c : cats) {
						tester.add(getTest(c, null, MatchedElement.class));
					}
				}
				else tester.add(getTest(id, null, MatchedElement.class));
			}
		}
		else {
			error(ParserError.IDENTIFIER_EXPECTED);
			skip(Symbol.SCOLON, Symbol.RBRCT);
		}
	}
	
	public Test getTest(String id, Test test, Class<?> cl)
	{
		Method method;
		Object[] params;
		
		if (test == null) test = new Test();
		method = getMethod(id, cl);
		if (method == null) {
			method = getMethod(cl);
			params = new Object[3];
			params[1] = id;
			params[2] = tagset;
		}
		else {
			params = new Object[2];
			params[1] = tagset;
		}
		test.setParams(params);
		test.method = method;
		return test;
	}

	public Method getMethod(Class<?> cl)
	{
		Class<?> e = Evaluator.class;
		Method method;
		String id = null;
		
		try {
			if (cl.getSimpleName().equals("Token")) {
				id = "categoryValue";
				method = e.getDeclaredMethod(id, cl, String.class, Tagset.class);
			}
			else {
				id = "agreement";
				method = e.getDeclaredMethod(id, cl, String.class, Tagset.class);
			}
		}
		catch (Exception exception) {
			method = null;
		}
		return method;
	}
	
	public Method getMethod(String id, Class<?> cl)
	{
		Class<?> e = Evaluator.class;
		Method method;
		
		try {
			if (id.equals("pos") && tagset.getType() == Tagset.POS_TAG) id = "tag";
			method = e.getDeclaredMethod(id + "_", cl, Tagset.class);
		} 
		catch (Exception e1) {
			method = null;
		}
		return method;
	}
	
	public void parseArguments(Test test) throws ParserException
	{
		boolean done = false;
		LinkedList<String> args = new LinkedList<String>();
		
		getSymbol();
		do {
			if (symbol == Symbol.IDENTIFIER) {
				args.add(identifier);
				getSymbol();
			}
			else if (symbol == Symbol.STRING) {
				args.add(string);
				getSymbol();
			}
			else if (symbol == Symbol.RBRCT) {
				if (args.isEmpty()) error(ParserError.STRING_EXPECTED);
			}
			else {
				error(ParserError.STRING_EXPECTED);
				skip(Symbol.COMMA, Symbol.IDENTIFIER);
			}
			if (symbol == Symbol.COMMA) getSymbol();
			else done = true;
		} while (!tooManyErrors() && !done);
		if (noErrors()) test.args = args.toArray(new String[0]);
	}
	
	public void parseTemplate(Rule rule) throws ParserException
	{
		boolean done = false;
		boolean computeBaseForm;
		QList list1, list2;
		
		list1 = null;
		do {
			list2 = null;
			do {
				if (symbol == Symbol.DOLLAR) {
					computeBaseForm = true;
					getSymbol();
				}
				else computeBaseForm = false;
				if (symbol == Symbol.IDENTIFIER) {
					QSym sym = new QSym(identifier);
					
					sym.computeBaseForm = computeBaseForm;
					if (list2 == null) list2 = new QList(QList.AND);
					list2.add(sym);
					getSymbol();
					if (symbol == Symbol.QMARK) {
						getSymbol();
						sym.quantifier = Template.ZERO_OR_ONE;
					}
					else if (symbol == Symbol.STAR) {
						getSymbol();
						sym.quantifier = Template.ZERO_OR_MORE;
					}
					else if (symbol == Symbol.PLUS) {
						getSymbol();
						sym.quantifier = Template.ONE_OR_MORE;
					}
				}
				else {
					error(ParserError.SYMBOL_EXPECTED);
					skip(Symbol.IDENTIFIER);
				}
			} while (!tooManyErrors() && symbol == Symbol.IDENTIFIER || symbol == Symbol.DOLLAR);
			if (symbol == Symbol.OR) {
				getSymbol();
				if (list1 == null) list1 = new QList(QList.OR);
				if (list2.size() > 1) list1.add(list2);
				else list1.add(list2.getFirst());
			}
			else if (symbol == Symbol.SCOLON) done = true;
		} while (!done && !tooManyErrors());
		if (list1 != null) {
			if (list2.size() > 1) list1.add(list2);
			else list1.add(list2.getFirst());
		}
		rule.subst = (list1 == null ? list2 : list1);
	}
	
	public void getSymbol() throws ParserException
	{
		int ch = SPACE;
		
		while (Character.isWhitespace(ch)) ch = getNextChar();
		scanPos = charCount - 1;
		if (isLetter(ch)) {
			buffer.setLength(0);
			while (isIdentifierChar(ch)) {
				buffer.append((char)ch);
				ch = getNextChar();
			}
			identifier = buffer.toString();
			symbol = Symbol.IDENTIFIER;
			if (noSkipChar(ch)) charCount--;
		}
		else if (ch == EXCLM) {
			ch = getNextChar();
			if (ch == EQ) symbol = Symbol.NEQ;
			else if (ch == TILDE) symbol = Symbol.NMATCH;
			else symbol = Symbol.UNKNOWN;
		}
		else if (ch == QUOTE) getString();
		else if (ch == DOLLAR) symbol = Symbol.DOLLAR;
		else if (ch == CIRCMFX) symbol = Symbol.CIRCMFX;
		else if (ch == EOF) symbol = Symbol.EMPTY;
		else if (ch == EQ) symbol = Symbol.EQ;
		else if (ch == QMARK) symbol = Symbol.QMARK;
		else if (ch == STAR) symbol = Symbol.STAR;
		else if (ch == PLUS) symbol = Symbol.PLUS;
		else if (ch == COLON) symbol = Symbol.COLON;
		else if (ch == SCOLON) symbol = Symbol.SCOLON;
		else if (ch == LBRCT) symbol = Symbol.LBRCT;
		else if (ch == RBRCT) symbol = Symbol.RBRCT;
		else if (ch == COMMA) symbol = Symbol.COMMA;
		else if (ch == VBAR) symbol = Symbol.OR;
		else if (ch == TILDE) symbol = Symbol.MATCH;
		else symbol = Symbol.UNKNOWN;
	}
	
	public boolean noSkipChar(int ch)
	{
		return ch == QUOTE || 
			   ch == DOLLAR || 
			   ch == CIRCMFX ||
			   ch == EQ || 
			   ch == QMARK || 
			   ch == STAR || 
			   ch == PLUS ||
			   ch == COLON ||
			   ch == SCOLON ||
			   ch == LBRCT ||
			   ch == RBRCT ||
			   ch == COMMA ||
			   ch == VBAR ||
			   ch == TILDE;
	}

	public Template getTemplate()
	{
		return template;
	}
	
	public Template compileRules() throws ParserException
	{
		Rule head;
		Template tmp;
		
		head = findHead();
		tmp = compile(head);
		return tmp;
	}
	
	public Template compile(Rule rule) throws ParserException
	{
		if (rule == null) return null;
		LinkedList<String> stack = new LinkedList<String>();
		
		stack.add(rule.id);
		return compile(rule, stack);
	}
	
	public Template compile(Rule rule, LinkedList<String> stack) throws ParserException
	{
		Tester tester = tsym.get(rule.id);
		Template tmp = compile(rule.subst, tester, stack);

		return tmp;
	}
	
	public Template compile(QList subst, Tester tester, LinkedList<String> stack) throws ParserException
	{
		CompoundTemplate tmp;
		
		if (subst.type == QList.AND) tmp = new AndTemplate();
		else tmp = new OrTemplate();
		for (Object obj : subst.elements) {
			if (obj instanceof QSym) {
				QSym qs = (QSym)obj;
				Template t = compile(qs, stack);
				
				if (t == null) break;
				tmp.add(t);
			}
			else {
				QList ql = (QList)obj;
				
				tmp.add(compile(ql, null, stack));
			}
		}
		tmp.setTester(tester);
		return tmp;
	}
	
	public Template compile(QSym qs, LinkedList<String> stack) throws ParserException
	{
		Rule rule = findRule(qs.id);
		Tester tester = tsym.get(qs.id);
		Template tmp;
		
		if (stack.contains(qs.id)) {
			errors.add(new ParserError(ParserError.NO_LOOPS_ALLOWED));
			throw new ParserException();
		}
		if (rule == null) tmp = new SimpleTemplate();
		else {
			stack.add(qs.id);
			tmp = compile(rule, stack);
			stack.removeLast();
		}
		tmp.setQuantifier(qs.quantifier);
		tmp.setTester(tester);
		tmp.setComputeBaseForm(qs.computeBaseForm);
		tmp.setHead(headSymbols.contains(qs.id));
		return tmp;
	}
	
	public Rule findRule(String id)
	{
		for (Rule r : rules) {
			if (r.id.equals(id)) return r;
		}
		return null;
	}
	
	public Rule findHead()
	{
		LinkedList<Rule> head = null;
		
		for (Rule r1 : rules) {
			String sym = r1.id;
			boolean found = false;
			
			for (Rule r2 : rules) {
				QList subst = r2.subst;
				
				if (subst.contains(sym)) {
					if (r1 == r2) {
						errors.add(new ParserError(ParserError.NO_LOOPS_ALLOWED));
						return null;
					}
					found = true;
					break;
				}
			}
			if (noErrors() && !found) {
				if (head == null) head = new LinkedList<Rule>();
				head.add(r1);
			}
		}
		if (noErrors()) {
			if (head == null) {
				errors.add(new ParserError(ParserError.NO_LOOPS_ALLOWED));
				return null;
			}
			else if (head.size() == 1) return head.getFirst();
			else {
				Rule root = new Rule("root");
				QList subst = new QList(QList.OR);
				
				for (Rule r : head) {
					QSym qs = new QSym(r.id);
					
					subst.add(qs);
				}
				root.subst = subst;
				return root;
			}
		}
		return null;
	}
	
}
