package termopl;

import java.awt.*;
import java.io.*;
import java.util.*;

import javax.swing.*;

@SuppressWarnings("serial")
public class SentencesView extends JPanel
{

	private TermoPLDocument doc;
	private JScrollPane sp;
	private Content content;
	private InfoPanel infoPanel;
	private Term term;
	
	public SentencesView(TermoPLDocument document)
	{
		doc = document;
		setLayout(new BorderLayout());
		arrangeComponents();
		
		EventQueue.invokeLater(new Runnable() {
			public void run()
			{
				TermsView tv = doc.getTermsView();
				
				if (tv != null) setData(tv.getSelectedTerm());
			}
		});
	}
	
	public void arrangeComponents()
	{
		content = new Content();
		infoPanel = new InfoPanel();
		sp = new JScrollPane(content);
		sp.getViewport().setBackground(CommonResources.LIGHT_GRAY);
		sp.setBorder(BorderFactory.createCompoundBorder(
			BorderFactory.createEmptyBorder(-1, -1, 0, -1),
			BorderFactory.createLineBorder(Color.lightGray)));
		add(sp, BorderLayout.CENTER);
		add(infoPanel, BorderLayout.SOUTH);
	}
	
	public void setData(Term term)
	{
		this.term = term;
		content.calcSize();
		content.invalidate();
		scrollRectToVisible(CommonResources.nullRect);
		repaint();
		if (term != null) {
			int n = 0, fileID = -1, oldFileID = -1;
			long pos = -1, oldPos = -1;
			
			for (SentenceRef r : term.getSentenceRef()) {
				if (r instanceof SentenceRefEx) fileID = ((SentenceRefEx) r).fileID;
				pos = r.start;
				if (pos != oldPos || fileID != oldFileID) n++;
				oldFileID = fileID;
				oldPos = pos;
			}
			infoPanel.setInfo("Sentences found: " + n + ".");
		}
		else {
			infoPanel.setWarning("Term is not selected.");
		}
	}
	
	public void reset()
	{
		setData(null);
	}
	
	public void changeFontSize()
	{
		content.calcSize();
		invalidate();
		repaint();
	}

	private class Content extends JPanel
	{
		private RandomAccessFile source;
		private File sourceFile;
		private int indent;
		private int[] lineCount;
		
		public Content()
		{
			source = null;
			sourceFile = null;
			indent = 0;
			setBackground(CommonResources.LIGHT_GRAY);
		}
		
		public void setSize(int w, int h)
		{
			super.setSize(w, h);
			calcSize();
		}
		
		public void calcSize()
		{
			int n = 0;
			
			if (term != null) {
				Graphics graphics = getGraphics();
				Graphics2D g = (Graphics2D)graphics;
				LinkedList<SentenceRef> sref = term.getSentenceRef();
				FontMetrics fm;
				int i, h, fileID, oldFileID;
				long pos, oldPos;
				double f;
				
				g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
				i = 0;
				pos = oldPos = -1;
				fileID = oldFileID = -1;
				for (SentenceRef r : sref) {
					if (r instanceof SentenceRefEx) {
						fileID = ((SentenceRefEx) r).fileID;
					}
					pos = r.start;
					if (pos != oldPos || fileID != oldFileID) i++;
				}
				lineCount = new int[i];
				f = Math.log10(i);
				if (f < 1.0) i = 1;
				else i = (int)f + 1;
				fm = graphics.getFontMetrics(TermoPL.preferences.plainFont);
				indent = i * fm.charWidth('0') + 8;
				graphics.setFont(TermoPL.preferences.boldFont);
				fm = graphics.getFontMetrics();
				h = fm.getHeight();
				i = 0;
				pos = oldPos = -1;
				fileID = oldFileID = -1;
				
				LinkedList<Integer> positions = new LinkedList<Integer>();
				LinkedList<Token> sentence = null;
				byte[] termBounds = null;
				
				for (SentenceRef r : sref) {
					if (r instanceof SentenceRefEx) {
						fileID = ((SentenceRefEx) r).fileID;
					}
					pos = r.start;
					if (pos != oldPos || fileID != oldFileID) {
						Pair<LinkedList<Token>, LinkedList<MultiWordToken>> pair;
						LinkedList<MultiWordToken> mwtList = null;
						FileDescr fd = doc.getAnalyzedFile(r.getFileID());
						
						oldPos = pos; oldFileID = fileID;
						if (!positions.isEmpty()) {
							termBounds = getTermBounds(sentence, term, positions);
							positions = new LinkedList<Integer>();
							n += calcSentenceSize(graphics, i, sentence, termBounds);
							i++;
						}
						try {
							if (sourceFile != fd.file) {
								sourceFile = fd.file;
								source = new RandomAccessFile(sourceFile, "r");
							}
							pair = CorpusReader.getSentence(source, fd.type, r.start, r.len);
							sentence = pair.first;
							mwtList = pair.second;
						}
						catch (Exception e) {
							e.printStackTrace();
						}
						if (mwtList != null) sentence = CorpusReader.replaceMWT(sentence, mwtList);
						if (r.index == 0) {
							termBounds = getTermBounds(sentence, term);
							n += calcSentenceSize(graphics, i, sentence, termBounds);
							i++;
						}
						else positions.add(r.index);
					}
				}
				if (!positions.isEmpty()) {
					termBounds = getTermBounds(sentence, term, positions);
					n += calcSentenceSize(graphics, i, sentence, termBounds);
				}
				setPreferredSize(new Dimension(0, n * h));
			}
			else {
				lineCount = null;
				indent = 0;
				setPreferredSize(new Dimension(0, 0));
			}
		}
		
		public int calcSentenceSize(Graphics graphics, int n, LinkedList<Token> sentence, byte[] termBounds)
		{
			FontMetrics fm;
			int s, w, x, dx, index;
			boolean appendSpace;
			
			graphics.setFont(TermoPL.preferences.plainFont);
			fm = graphics.getFontMetrics();
			s = 0;
			x = indent + 4;
			w = getWidth() - indent - 8;
			appendSpace = false;
			index = 0;
			for (Token t : sentence) {
				if (appendSpace) x += fm.stringWidth(" ");
				switch (termBounds[index]) {
					case 0:
					case 1:
						graphics.setFont(TermoPL.preferences.plainFont);
						break;
					case 2:
					default:
						graphics.setFont(TermoPL.preferences.boldFont);
				}
				fm = graphics.getFontMetrics();
				dx = fm.stringWidth(t.form);
				if (x + dx > w) {
					x = indent + 4;
					s++;
				}
				x += dx;
				appendSpace = t.spaceAfter;
				index++;
			}
			lineCount[n] = s + 1;
			return lineCount[n];
		}

		public void paintComponent(Graphics graphics)
		{
			super.paintComponent(graphics);
			
			Graphics2D g = (Graphics2D)graphics;
			
			g.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
			if (term != null) {
				paintElements(g, term);
			}
		}
		
		public void paintElements(Graphics graphics, Term term)
		{
			int index, n, i, y, Y, h, fileID, oldFileID;
			long pos, oldPos;
			FontMetrics fm;
			Rectangle bounds = graphics.getClipBounds();
			LinkedList<SentenceRef> sref = term.getSentenceRef();
			
			graphics.setFont(TermoPL.preferences.boldFont);
			fm = graphics.getFontMetrics();
			h = fm.getHeight();
			y = i = n = index = 0;
			pos = oldPos = -1;
			fileID = oldFileID = -1;
			for (SentenceRef r : sref) {
				if (r instanceof SentenceRefEx) {
					fileID = ((SentenceRefEx) r).fileID;
				}
				pos = r.start;
				if (pos != oldPos || fileID != oldFileID) {
					index = n;
					if (y + h * lineCount[i] > bounds.y) break;
					y += h * lineCount[i];
					oldPos = pos;
					oldFileID = fileID;
					i++;
				}
				n++;
			}
			pos = oldPos = -1;
			fileID = oldFileID = -1;
			Y = bounds.y + bounds.height;
			
			LinkedList<Integer> positions = new LinkedList<Integer>();
			LinkedList<Token> sentence = null;
			ListIterator<SentenceRef> it = sref.listIterator(index);
			byte[] termBounds = null;
			
			while (it.hasNext()) {
				SentenceRef r = it.next();
				
				if (r instanceof SentenceRefEx) {
					fileID = ((SentenceRefEx) r).fileID;
				}
				pos = r.start;
				if (pos != oldPos || fileID != oldFileID) {
					oldPos = pos;
					oldFileID = fileID;
					
					Pair<LinkedList<Token>, LinkedList<MultiWordToken>> pair;
					LinkedList<MultiWordToken> mwtList = null;
					FileDescr fd = doc.getAnalyzedFile(r.getFileID());
					
					oldPos = pos; oldFileID = fileID;
					if (!positions.isEmpty()) {
						termBounds = getTermBounds(sentence, term, positions);
						positions = new LinkedList<Integer>();
						y = drawSentence(graphics, 0, y, i + 1, sentence, termBounds);
						graphics.setColor(CommonResources.LIGHT_GRAY);
						graphics.drawLine(0, y - 1, getWidth(), y - 1);
						i++;
					}
					try {
						if (sourceFile != fd.file) {
							sourceFile = fd.file;
							source = new RandomAccessFile(sourceFile, "r");
						}
						pair = CorpusReader.getSentence(source, fd.type, r.start, r.len);
						sentence = pair.first;
						mwtList = pair.second;
					}
					catch (Exception e) {
						e.printStackTrace();
					}
					if (mwtList != null) sentence = CorpusReader.replaceMWT(sentence, mwtList);
					if (r.index == 0) {
						termBounds = getTermBounds(sentence, term);
						y = drawSentence(graphics, 0, y, i + 1, sentence, termBounds);
						graphics.setColor(CommonResources.LIGHT_GRAY);
						graphics.drawLine(0, y - 1, getWidth(), y - 1);
						i++;
					}
				}
				if (r.index > 0) positions.add(r.index);
				if (y >= Y) break;
			}
			if (y < Y && !positions.isEmpty()) {
				termBounds = getTermBounds(sentence, term, positions);
				y = drawSentence(graphics, 0, y, i + 1, sentence, termBounds);
				graphics.setColor(CommonResources.LIGHT_GRAY);
				graphics.drawLine(0, y - 1, getWidth(), y - 1);
			}
			graphics.setColor(Color.lightGray);
			graphics.drawLine(indent, bounds.y, indent, Y);
		}
		
		public byte[] getTermBounds(LinkedList<Token> sentence, Term term)
		{
			byte[] bounds = new byte[sentence.size()];
			LinkedList<MatchedToken> mtokens = term.getForms().getFirst().getTokens();
			ListIterator<Token> siterator = sentence.listIterator();
			Token[] termTokens = new Token[mtokens.size()];
			int i = 0, n = 0, index = 0, s = -1, e = -1;
			
			for (MatchedToken mt : mtokens) termTokens[i++] = mt.token;
			
			for (i = 0; i < bounds.length; i++) bounds[i] = 0;
			while (siterator.hasNext()) {
				Token t1 = siterator.next();
				Token t2 = termTokens[n];
				
				if (t1.lemma.equals(t2.lemma)) {
					if (s == -1) s = e = index;
					else e++;
					if (n == termTokens.length - 1) {
						n = 0;
						for (i = s; i <= e; i++) bounds[i] = 2;
						s = e = -1;
					}
					else n++;
				}
				else {
					n = 0;
					s = e = -1;
				}
				index++;
			}
			return bounds;
		}
		
		public byte[] getTermBounds(LinkedList<Token> sentence, Term term, LinkedList<Integer> positions)
		{
			byte[] bounds = new byte[sentence.size()];
			ListIterator<Token> siterator = sentence.listIterator();
			int i;
			
			for (i = 0; i < bounds.length; i++) bounds[i] = 0;
			i = 0;
			for (int pos : positions) {
				LinkedList<MatchedToken> mtokens = term.getForms().getFirst().getTokens();
				
				while (siterator.hasNext()) {
					Token st = siterator.next();
					int index;
					
					if (st instanceof UDToken) index = ((UDToken) st).index;
					else if (st instanceof MultiWordToken) index = ((MultiWordToken) st).startToken();
					else index = -1;
					if (index == pos) {
						ListIterator<MatchedToken> miterator = mtokens.listIterator();
						
						bounds[i++] = 2;
						miterator.next();
						if (!miterator.hasNext()) break;
						Token t1 = miterator.next().token;
						while (siterator.hasNext()) {
							Token t2 = siterator.next();
							if (t1.lemma.equals(t2.lemma)) {
								bounds[i] = 2;
								if (!miterator.hasNext()) {
									i++;
									break;
								}
								t1 = miterator.next().token;
							}
							else bounds[i] = 1;
							i++;
						}
						if (!miterator.hasNext()) break;
					}
					else i++;
				}
			}
			return bounds;
		}
		
		public int drawSentence(Graphics graphics, int x, int y, int n, LinkedList<Token> sentence, byte[] termBounds)
		{
			String str = String.valueOf(n);
			FontMetrics fm;
			Color c, c0;
			int index, h, a, w, dx;
			boolean appendSpace;
			
			graphics.setFont(TermoPL.preferences.boldFont);
			fm = graphics.getFontMetrics();
			h = fm.getHeight();
			a = fm.getAscent();
			c = ((n % 2) == 0 ? CommonResources.VLIGHT_BLUE : Color.white);
			graphics.setColor(c);
			graphics.fillRect(0, y, getWidth(), h);
			c0 = Color.black;
			graphics.setColor(c0);
			graphics.setFont(TermoPL.preferences.plainFont);
			fm = graphics.getFontMetrics();
			w = fm.stringWidth(str);
			w = (indent - w) / 2;
			graphics.drawString(str, w, y + a);
			x = indent + 4;
			w = getWidth() - indent - 8;
			appendSpace = false;
			index = 0;
			for (Token t : sentence) {
				if (appendSpace) {
					dx = fm.stringWidth(" ");
					x += dx;
				}
				appendSpace = t.spaceAfter;
				switch (termBounds[index]) {
					case 0:
					case 1:
						graphics.setFont(TermoPL.preferences.plainFont);
						break;
					case 2:
					default:
						graphics.setFont(TermoPL.preferences.boldFont);
				}
				fm = graphics.getFontMetrics();
				dx = fm.stringWidth(t.form);
				if (x + dx > w) {
					x = indent + 4;
					y += h;
					graphics.setColor(c);
					graphics.fillRect(0, y, getWidth(), h);
					graphics.setColor(c0);
				}
				switch (termBounds[index]) {
					case 0:
						graphics.setColor(Color.black);
						break;
					case 1:
						graphics.setColor(Color.blue);
						break;
					case 2:
					default:
						graphics.setColor(Color.red);
				}
				graphics.drawString(t.form, x, y + a);
				x += dx;
				index++;
			}
			return y + h;
		}
	}
	
}
